c     ******************************************************************
c     *     psuedo-random number generators.   rndc, rndd, rnde        *
c     *                                                                *
c     *    In normal applications, the user may use only rndc as       *
c     *                                                                *
c     *       call rndc(u)                                             *
c     *                                                                *
c     *  where u is the output random number in (0., 1.) (excluding    *
c     *  the boundary values).  u is a real*8 variable.                *
c     *----------------------------------------------------------------*
c     *  Internally, there are 3 independent generators.               *
c     *  rndc and rndd use 2 generators: rnd1u or  rnd2u.  the default *
c     *  routine is rnd1u.  rnd1u/rnd2u can be specifed by calling the *
c     *  rndsw subroutine.  rnde uses always the 3rd generators.       *
c     *  the differece between rndc and rndd is the calling sequence.  *
c     *                                                                *
c     * if the user wants to have random numbers in an array,          *
c     *                                                                *
c     *       call rndd(ua, n)                                         *
c     *                                                                *
c     *  may be used, where ua  is an array  to get n random variables *
c     *  in (0., 1.).                                                  *
c     *                                                                *
c     *  for rnde, the calling sequence is                             *
c     *                                                                *
c     *      call rnde(ua, n)                                          *
c     *                                                                *
c     *  which is the same as that for rndd.                           *
c     *                                                                *
c     *    To switch the generators, the user may use                  *
c     *                                                                *
c     *      call rndsw(jold, jnew)                                    *
c     *                                                                *
c     *  where jold is the output value (1 or 2) indicating the        *
c     *  generator (rnd1u or rnd2u) currently used. jnew is the input  *
c     *  integer  (1 or 2) to specify the generator from the next call *
c     *  on.  This can be called any time.                             *
c     *    In some applications, the user may wish to initialize the   *
c     *  random number generators using explicit seeds, or the  user   *
c     *  may need to save the current status of the generators for     *
c     *  restarting the job in a later time, as if the job were        *
c     *  continued without halt.                                       *
c     *                                                                *
c     *   To give explicit seeds for generator 1,  use                 *
c     *                                                                *
c     *      call rnd1i(ir1)   for generator 1                         *
c     *      call rnd2i(ir2)   for generator 2                         *
c     *      call rnd3i(ir3)   for generator 3                         *
c     *                                                                *
c     *  where the argument is the input integer(s).                   *
c     *     ir1: ir1(1) and ir2(2) should contain integer seeds.       *
c     *     ir2: an integer seed.                                      *
c     *     ir3: an intgeger seed.                                     *
c     *  These can be called any time.                                 *
c     *                                                                *
c     *    To save the current status of the generator, use            *
c     *                                                                *
c     *      call rnd1s(ir1sv)     for generator 1                     *
c     *      call rnd2s(ir2sv)     for generator 2                     *
c     *      call rnd3s( r3sv)     for generator 3                     *
c     *                                                                *
c     *  where  ir1sv is an output integer array to get 2 integers,    *
c     *         ir2sv is an output intgger array to get 25 integers,   *
c     *         r3sv  is an output real*8 array to get 102 reals.      *
c     *  These can be called any time.                                 *
c     *                                                                *
c     *                                                                *
c     *     To restore the status when the rndis was called, (i=1,2,3),*
c     *  use,                                                          *
c     *                                                                *
c     *        call rnd1r(ir1sv)  for generator 1                      *
c     *        call rnd2r(ir2sv)  for generator 2                      *
c     *        call rnd3r(r3sv)   for generator 3                      *
c     *  where the arguments are the input which are obtained by       *
c     *  calling rndis (i=1,2,3).                                      *
c     *  These can be called any time.                                 *
c     *                                                                *
c     *                                                                *
c     *  ------------------------------------------------------------  *
c     *    General features of these generators.                       *
c     *                                                                *
c     *      They are all reliable generators with a long period and   *
c     *  free from undesirable correlations between successive random  *
c     *  numbers.  They are given in all standard fortran code so that *
c     *  they can be used in any macihne with integer/real expressed in*
c     *  32 or more bits.                                              *
c     *       Reference.                                               *
c     *                                                                *
c     ********************* tested 91.08.09 ******************k.k*******
      subroutine rndc(u)
         implicit none
         real*8 u
         integer n
         real*8 ua(n)
c
         integer iseed(25), ir1(2), ir2, jold, jnew
         logical first2/.true./
         integer jsw/1/
         save jsw, first2
c
         if(jsw .eq. 1) then
             call rnd1u(u, 1)
         elseif(jsw .eq. 2) then
             if(first2) then
                first2=.false.
                call rnd2ix(31415926)
             endif
             call rnd2u(u, 1)
         else
             write(*,*) ' switch value error=',jsw, ' in rndc '
             stop 9999
         endif
         return
c     ***********
      entry rnd1r(ir1)
c     **********
         call rnd1i(ir1)
         return
c     ***********
      entry rndd(ua, n)
c     **********
         if(jsw .eq. 1) then
             call rnd1u(ua, n)
         elseif(jsw .eq. 2) then
             if(first2) then
                first2=.false.
                call rnd2ix(31415926)
             endif
             call rnd2u(ua, n)
         else
             write(*,*) ' switch value error=',jsw, ' in rndd '
             stop 9999
         endif
         return
c     ***********
      entry rnd2i(ir2)
c     ***********
         call rnd2ix(ir2)
         first2=.false.
         return
c     ***********
      entry rnd2r(iseed)
c     ***********
         call rnd2rx(iseed)
         first2=.false.
         return
c     ***********
      entry rndsw(jold, jnew)
c     ***********
         jold=jsw
         jsw=jnew
         return
      end
      subroutine rnd1u(ua, n)
c            random number generator given by L'ecuyer in
c            comm. acm vol 31, p.742, 1988
c            modified by f. James to return a vector of numbers
c            modified by K.K
c    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c     calling sequences for rnd1u:                                   ++
c         call rnd1u (ua, n)        returns a vector ua of n         ++
c                      64-bit random floating point numbers between  ++
c                      zero and one.                                 ++
c         call rnd1i(irin)      initializes the generator from two   ++
c                      32-bit integer array irin                     ++
c         call rnd1s(irout)     outputs the current values of the    ++
c                      two integer seeds, to be used for restarting  ++
c    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
          implicit none
          integer n
          real*8 ua(n)
c 
         integer irin(2), irout(2), iseed1, iseed2, i, k, iz
          save iseed1,iseed2
          data iseed1,iseed2 /12345,67890/
c
           do   i= 1, n
             k = iseed1/53668
             iseed1 = 40014*(iseed1 - k*53668) - k*12211
             if(iseed1 .lt. 0) iseed1=iseed1+2147483563
c
             k = iseed2/52774
             iseed2 = 40692*(iseed2 - k*52774) - k* 3791
             if(iseed2 .lt. 0) iseed2=iseed2+2147483399
c
             iz = iseed1 - iseed2
             if(iz .lt. 1) iz = iz + 2147483562
c
             ua(i) = iz * 4.656613d-10
           enddo
          return
c     ****************
      entry rnd1i(irin)
c     ****************
          iseed1 = irin(1)
          iseed2 = irin(2)
          return
c
c     ****************
      entry rnd1s(irout)
c     ****************
          irout(1)= iseed1
          irout(2)= iseed2
      end
      subroutine rnd2u(ua,n)
c         add-and-carry random number generator proposed by
c         Marsaglia and Zaman in siam j. scientific and statistical
c         computing, to appear probably 1990.
c         modified with enhanced initialization by F. James, 1990
c         modified by K.K
c    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c     calling sequences for rnd2u:                                   ++
c         call rnd2u (ua, n)        returns a vector ua of n         ++
c                      64-bit random floating point numbers between  ++
c                      zero and one.                                 ++
c         call rnd2ix(int)     initializes the generator from one    ++
c                      32-bit integer int                            ++
c         call rnd2rx(ivec)    restarts the generator from vector    ++
c                      ivec of 25 32-bit integers (see rnd2s)        ++
c         call rnd2s(ivec)     outputs the current values of the 25  ++
c                    32-bit integer seeds, to be used for restarting ++
c    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      implicit none
      integer n
      real*8 ua(n)
c
      integer ns, ns1, i, isd, k
      parameter (ns=24, ns1=ns+1)
      real*8 seeds(ns), uni
      integer iseeds(ns), isdin(ns1), isdout(ns1), icarry
      real*8 twop24, twom24
      integer itwo24, icons, i24, j24, ivec, jseed, inseed
      parameter (twop24=2.**24)
      parameter (twom24=2.**(-24), itwo24=2**24, icons=2147483563)
      real*8 carry
      save  i24, j24, carry, seeds
      data i24, j24, carry/ns, 10, 0./
c
c          the generator proper: "subtract-with-borrow",
c          as proposed by Marsaglia and Zaman,
c          Florida state university, march, 1989
c
       do   ivec= 1, n
          uni = seeds(i24) - seeds(j24) - carry
          if(uni .lt. 0.d0) then
             uni = uni + 1.0d0
             carry = twom24
          else
             carry = 0.
          endif
c                 avoid exact zero
          if(uni .eq. 0.d0) then
              uni = twom24
          endif
          seeds(i24) = uni
          i24 = i24 - 1
          if(i24 .eq. 0) i24 = ns
          j24 = j24 - 1
          if(j24 .eq. 0) j24 = ns
          ua(ivec) = uni     
       enddo
       return
c           entry to restore the previous run
c     ******************
      entry rnd2rx(isdin)
c     ******************
       do   i= 1, ns
         seeds(i) =isdin(i)*twom24
       enddo
      carry = mod(isdin(ns1),10)*twom24
      isd = isdin(ns1)/10
      i24 = mod(isd,100)
      isd = isd/100
      j24 = isd
      return
c                    entry to get current status
c     ******************
      entry rnd2s(isdout)
c     ******************
       do   i= 1, ns
         isdout(i) = int(seeds(i)*twop24)
       enddo
      if(carry .gt. 0.) then
         icarry=1
      else
         icarry=0
      endif
      isdout(ns1) = 1000*j24 + 10*i24 + icarry
      return
c                    entry to initialize from one integer
c     ******************
      entry rnd2ix(inseed)
c     ******************
         jseed = inseed
          do   i= 1, ns
             k = jseed/53668
             jseed = 40014*(jseed-k*53668) -k*12211
             if(jseed .lt. 0) jseed = jseed+icons
             iseeds(i) = mod(jseed,itwo24)
          enddo
          do   i= 1,ns
             seeds(i) =iseeds(i)*twom24
          enddo
         i24 = ns
         j24 = 10
         carry = 0.
         if(seeds(ns) .lt. seeds(14)) carry = twom24
      end
      subroutine rnde(ua,n)
c           random number generator proposed by marsaglia and zaman
c           in report fsu-scri-87-50
c           modified by f. james, 1988 and 1989, to generate a vector
c           of pseudorandom numbers ua of length n.
c           modified by k.k
c    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c     calling sequences for rnde:                                    ++
c         call rnde(ua, n)         returns a vector ua of n          ++
c                      32-bit random floating point numbers between  ++
c                      zero and one.                                 ++
c         call rnd3i(i1)          initializes the generator from one ++
c                      32-bit integer i1
c    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      implicit none
      integer n
      real*8 ua(n), sina(102)
      logical first/.true./
      save  first
      integer ijkl, ijklin
c
      if(first) then
         first=.false.
c           default initialization. user has called rnde without rnd3i.
         ijkl = 54217137
         call rnd3ix(ijkl)
      endif
      call rnd3x(ua, n)
      return
c         initializing routine for rnde, may be called before
c         generating pseudorandom numbers with rnde. the input
c         values should be in the ranges:  0<=ijklin<=900 ooo ooo
c     **************
      entry rnd3i(ijklin)
c     *************
      first=.false.
      call rnd3ix(ijklin)
      return
c     ************
      entry rnd3r(sina)
c     ************
         first=.false.
         call rnd3rx(sina)
      end
      subroutine  rnd3ix(ijkl)
c             the standard values in marsaglia's paper, ijkl=54217137
          implicit none
          integer n
          real*8 ua(n), u(97), uni, s, t, zuni
          real*8 sina(102), sout(102)
          integer  jj, m
          integer ns, ijkl, i97, j97, ij, kl, i, j, k, l, ii, ivec
          real*8 twom24, c, cd, cm
          parameter (ns=24, twom24=2.**(-24))
          save c, cd, cm, i97, j97
c
          ij = ijkl/30082
          kl = ijkl - 30082*ij
          i = mod(ij/177, 177) + 2
          j = mod(ij, 177) + 2
          k = mod(kl/169, 178) + 1
          l = mod(kl, 169)
           do   ii= 1, 97
              s = 0.
              t = .5
               do   jj= 1, ns
                  m = mod(mod(i*j,179)*k, 179)
                  i = j
                  j = k
                  k = m
                  l = mod(53*l+1, 169)
                  if(mod(l*m,64) .ge. 32) s = s+t
                  t = 0.5*t
               enddo
              u(ii) = s
           enddo
          c = 362436.*twom24
          cd = 7654321.*twom24
          cm = 16777213.*twom24
          i97 = 97
          j97 = 33
          return
c     ****************
      entry rnd3x(ua, n)
c     ****************
       do   ivec= 1, n
          uni = u(i97)-u(j97)
          if(uni .lt. 0.) uni=uni+1.
          u(i97) = uni
          i97 = i97-1
          if(i97 .eq. 0) i97=97
          j97 = j97-1
          if(j97 .eq. 0) j97=97
          c = c - cd
          if(c .lt. 0.) c=c+cm
          uni = uni-c
          if(uni .lt. 0.) uni=uni+1.
          ua(ivec) = uni
c                 replace exact zeros by uniform distr. *2**-24
          if(uni .eq. 0.) then
              zuni = twom24*u(2)
c               an exact zero here is very unlikely, but let's be safe.
              if(zuni .eq. 0.) zuni= twom24*twom24
              ua(ivec) = zuni
          endif
       enddo
      return
c     ****************** to get current status
      entry rnd3s(sout)
c     ***********
           do   i=1, 97
              sout(i)=u(i)
           enddo
          sout(98)=c
          sout(99)=cd
          sout(100)=cm
          sout(101)=i97
          sout(102)=j97
          return
c     ****************  to restore the old status
      entry rnd3rx(sina)
           do   i=1, 97
               u(i)=sina(i)
           enddo
          c=sina(98)
          cd=sina(99)
          cm=sina(100)
          i97=sina(101)
          j97=sina(102)
      end
