c*********************************************************************
c*********************************************************************
c*                                                                  **
c*                                                     june 1991    **
c*                                                                  **
c*   the lund monte carlo for jet fragmentation and e+e- physics    **
c*                                                                  **
c*                        jetset version 7.3                        **
c*                                                                  **
c*                        torbjorn sjostrand                        **
c*                                                                  **
c*                    cern/th, ch-1211 geneva 23                    **
c*                bitnet/earn address torsjo@cernvm                 **
c*                       tel. +22 - 767 28 20                       **
c*                                                                  **
c*          lushow is written together with mats bengtsson          **
c*                                                                  **
c*           a complete manual exists on a separate file            **
c*         please report any program errors to the author!          **
c*                                                                  **
c*                   copyright torbjorn sjostrand                   **
c*                                                                  **
c*********************************************************************
c*********************************************************************
c                                                                    *
c  list of subprograms in order of appearance, with main purpose     *
c  (s = subroutine, f = function, b = block data)                    *
c                                                                    *
c  s   lu1ent   to fill one entry (= parton or particle)             *
c  s   lu2ent   to fill two entries                                  *
c  s   lu3ent   to fill three entries                                *
c  s   lu4ent   to fill four entries                                 *
c  s   lujoin   to connect entries with colour flow information      *
c  s   lugive   to fill (or query) commonblock variables             *
c  s   luexec   to administrate fragmentation and decay chain        *
c  s   luprep   to rearrange showered partons along strings          *
c  s   lustrf   to do string fragmentation of jet system             *
c  s   luindf   to do independent fragmentation of one or many jets  *
c  s   ludecy   to do the decay of a particle                        *
c  s   lukfdi   to select parton and hadron flavours in fragm        *
c  s   luptdi   to select transverse momenta in fragm                *
c  s   luzdis   to select longitudinal scaling variable in fragm     *
c  s   lushow   to do timelike parton shower evolution               *
c  s   luboei   to include bose-einstein effects (crudely)           *
c  f   ulmass   to give the mass of a particle or parton             *
c  s   luname   to give the name of a particle or parton             *
c  f   luchge   to give three times the electric charge              *
c  f   lucomp   to compress standard kf flavour code to internal kc  *
c  s   luerrm   to write error messages and abort faulty run         *
c  f   ulalem   to give the alpha_electromagnetic value              *
c  f   ulalps   to give the alpha_strong value                       *
c  f   ulangl   to give the angle from known x and y components      *
c  f   rlu      to provide a random number generator                 *
c  s   rluget   to save the state of the random number generator     *
c  s   rluset   to set the state of the random number generator      *
c  s   lurobo   to rotate and/or boost an event                      *
c  s   luedit   to remove unwanted entries from record               *
c  s   lulist   to list event record or particle data                *
c  s   luupda   to update particle data                              *
c  f   klu      to provide integer-valued event information          *
c  f   plu      to provide real-valued event information             *
c  s   lusphe   to perform sphericity analysis                       *
c  s   luthru   to perform thrust analysis                           *
c  s   luclus   to perform three-dimensional cluster analysis        *
c  s   lucell   to perform cluster analysis in (eta, phi, e_t)       *
c  s   lujmas   to give high and low jet mass of event               *
c  s   lufowo   to give fox-wolfram moments                          *
c  s   lutabu   to analyze events, with tabular output               *
c                                                                    *
c  s   lueevt   to administrate the generation of an e+e- event      *
c  s   luxtot   to give the total cross-section at given cm energy   *
c  s   luradk   to generate initial state photon radiation           *
c  s   luxkfl   to select flavour of primary qqbar pair              *
c  s   luxjet   to select (matrix element) jet multiplicity          *
c  s   lux3jt   to select kinematics of three-jet event              *
c  s   lux4jt   to select kinematics of four-jet event               *
c  s   luxdif   to select angular orientation of event               *
c  s   luonia   to perform generation of onium decay to gluons       *
c                                                                    *
c  s   luhepc   to convert between /lujets/ and /hepevt/ records     *
c  s   lutest   to test the proper functioning of the package        *
c  b   ludata   to contain default values and particle data          *
c                                                                    *
c*********************************************************************

      subroutine lu1ent(ip,kf,pe,the,phi)

c...purpose: to store one parton/particle in commonblock lujets.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...standard checks.
      mstu(28)=0
      if(mstu(12).ge.1) call lulist(0)
      ipa=max(1,iabs(ip))
      if(ipa.gt.mstu(4)) call luerrm(21,
     &'(lu1ent:) writing outside lujets memory')
      kc=lucomp(kf)
      if(kc.eq.0) call luerrm(12,'(lu1ent:) unknown flavour code')

c...find mass. reset k, p and v vectors.
      pm=0.
      if(mstu(10).eq.1) pm=p(ipa,5)
      if(mstu(10).ge.2) pm=ulmass(kf)
      do 100 j=1,5
      k(ipa,j)=0
      p(ipa,j)=0.
  100 v(ipa,j)=0.

c...store parton/particle in k and p vectors.
      k(ipa,1)=1
      if(ip.lt.0) k(ipa,1)=2
      k(ipa,2)=kf
      p(ipa,5)=pm
      p(ipa,4)=max(pe,pm)
      pa=sqrt(p(ipa,4)**2-p(ipa,5)**2)
      p(ipa,1)=pa*sin(the)*cos(phi)
      p(ipa,2)=pa*sin(the)*sin(phi)
      p(ipa,3)=pa*cos(the)

c...set n. optionally fragment/decay.
      n=ipa
      if(ip.eq.0) call luexec

      return
      end

c*********************************************************************

      subroutine lu2ent(ip,kf1,kf2,pecm)

c...purpose: to store two partons/particles in their cm frame,
c...with the first along the +z axis.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...standard checks.
      mstu(28)=0
      if(mstu(12).ge.1) call lulist(0)
      ipa=max(1,iabs(ip))
      if(ipa.gt.mstu(4)-1) call luerrm(21,
     &'(lu2ent:) writing outside lujets memory')
      kc1=lucomp(kf1)
      kc2=lucomp(kf2)
      if(kc1.eq.0.or.kc2.eq.0) call luerrm(12,
     &'(lu2ent:) unknown flavour code')

c...find masses. reset k, p and v vectors.
      pm1=0.
      if(mstu(10).eq.1) pm1=p(ipa,5)
      if(mstu(10).ge.2) pm1=ulmass(kf1)
      pm2=0.
      if(mstu(10).eq.1) pm2=p(ipa+1,5)
      if(mstu(10).ge.2) pm2=ulmass(kf2)
      do 100 i=ipa,ipa+1
      do 100 j=1,5
      k(i,j)=0
      p(i,j)=0.
  100 v(i,j)=0.

c...check flavours.
      kq1=kchg(kc1,2)*isign(1,kf1)
      kq2=kchg(kc2,2)*isign(1,kf2)
      if(mstu(19).eq.1) then
        mstu(19)=0
      else
        if(kq1+kq2.ne.0.and.kq1+kq2.ne.4) call luerrm(2,
     &  '(lu2ent:) unphysical flavour combination')
      endif
      k(ipa,2)=kf1
      k(ipa+1,2)=kf2

c...store partons/particles in k vectors for normal case.
      if(ip.ge.0) then
        k(ipa,1)=1
        if(kq1.ne.0.and.kq2.ne.0) k(ipa,1)=2
        k(ipa+1,1)=1

c...store partons in k vectors for parton shower evolution.
      else
        k(ipa,1)=3
        k(ipa+1,1)=3
        k(ipa,4)=mstu(5)*(ipa+1)
        k(ipa,5)=k(ipa,4)
        k(ipa+1,4)=mstu(5)*ipa
        k(ipa+1,5)=k(ipa+1,4)
      endif

c...check kinematics and store partons/particles in p vectors.
      if(pecm.le.pm1+pm2) call luerrm(13,
     &'(lu2ent:) energy smaller than sum of masses')
      pa=sqrt(max(0.,(pecm**2-pm1**2-pm2**2)**2-(2.*pm1*pm2)**2))/
     &(2.*pecm)
      p(ipa,3)=pa
      p(ipa,4)=sqrt(pm1**2+pa**2)
      p(ipa,5)=pm1
      p(ipa+1,3)=-pa
      p(ipa+1,4)=sqrt(pm2**2+pa**2)
      p(ipa+1,5)=pm2

c...set n. optionally fragment/decay.
      n=ipa+1
      if(ip.eq.0) call luexec

      return
      end

c*********************************************************************

      subroutine lu3ent(ip,kf1,kf2,kf3,pecm,x1,x3)

c...purpose: to store three partons or particles in their cm frame,
c...with the first along the +z axis and the third in the (x,z)
c...plane with x > 0.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...standard checks.
      mstu(28)=0
      if(mstu(12).ge.1) call lulist(0)
      ipa=max(1,iabs(ip))
      if(ipa.gt.mstu(4)-2) call luerrm(21,
     &'(lu3ent:) writing outside lujets memory')
      kc1=lucomp(kf1)
      kc2=lucomp(kf2)
      kc3=lucomp(kf3)
      if(kc1.eq.0.or.kc2.eq.0.or.kc3.eq.0) call luerrm(12,
     &'(lu3ent:) unknown flavour code')

c...find masses. reset k, p and v vectors.
      pm1=0.
      if(mstu(10).eq.1) pm1=p(ipa,5)
      if(mstu(10).ge.2) pm1=ulmass(kf1)
      pm2=0.
      if(mstu(10).eq.1) pm2=p(ipa+1,5)
      if(mstu(10).ge.2) pm2=ulmass(kf2)
      pm3=0.
      if(mstu(10).eq.1) pm3=p(ipa+2,5)
      if(mstu(10).ge.2) pm3=ulmass(kf3)
      do 100 i=ipa,ipa+2
      do 100 j=1,5
      k(i,j)=0
      p(i,j)=0.
  100 v(i,j)=0.

c...check flavours.
      kq1=kchg(kc1,2)*isign(1,kf1)
      kq2=kchg(kc2,2)*isign(1,kf2)
      kq3=kchg(kc3,2)*isign(1,kf3)
      if(mstu(19).eq.1) then
        mstu(19)=0
      elseif(kq1.eq.0.and.kq2.eq.0.and.kq3.eq.0) then
      elseif(kq1.ne.0.and.kq2.eq.2.and.(kq1+kq3.eq.0.or.
     &kq1+kq3.eq.4)) then
      else
        call luerrm(2,'(lu3ent:) unphysical flavour combination')
      endif
      k(ipa,2)=kf1
      k(ipa+1,2)=kf2
      k(ipa+2,2)=kf3

c...store partons/particles in k vectors for normal case.
      if(ip.ge.0) then
        k(ipa,1)=1
        if(kq1.ne.0.and.(kq2.ne.0.or.kq3.ne.0)) k(ipa,1)=2
        k(ipa+1,1)=1
        if(kq2.ne.0.and.kq3.ne.0) k(ipa+1,1)=2
        k(ipa+2,1)=1

c...store partons in k vectors for parton shower evolution.
      else
        k(ipa,1)=3
        k(ipa+1,1)=3
        k(ipa+2,1)=3
        kcs=4
        if(kq1.eq.-1) kcs=5
        k(ipa,kcs)=mstu(5)*(ipa+1)
        k(ipa,9-kcs)=mstu(5)*(ipa+2)
        k(ipa+1,kcs)=mstu(5)*(ipa+2)
        k(ipa+1,9-kcs)=mstu(5)*ipa
        k(ipa+2,kcs)=mstu(5)*ipa
        k(ipa+2,9-kcs)=mstu(5)*(ipa+1)
      endif

c...check kinematics.
      mkerr=0
      if(0.5*x1*pecm.le.pm1.or.0.5*(2.-x1-x3)*pecm.le.pm2.or.
     &0.5*x3*pecm.le.pm3) mkerr=1
      pa1=sqrt(max(1e-10,(0.5*x1*pecm)**2-pm1**2))
      pa2=sqrt(max(1e-10,(0.5*(2.-x1-x3)*pecm)**2-pm2**2))
      pa3=sqrt(max(1e-10,(0.5*x3*pecm)**2-pm3**2))
      cthe2=(pa3**2-pa1**2-pa2**2)/(2.*pa1*pa2)
      cthe3=(pa2**2-pa1**2-pa3**2)/(2.*pa1*pa3)
      if(abs(cthe2).ge.1.001.or.abs(cthe3).ge.1.001) mkerr=1
      cthe3=max(-1.,min(1.,cthe3))
      if(mkerr.ne.0) call luerrm(13,
     &'(lu3ent:) unphysical kinematical variable setup')

c...store partons/particles in p vectors.
      p(ipa,3)=pa1
      p(ipa,4)=sqrt(pa1**2+pm1**2)
      p(ipa,5)=pm1
      p(ipa+2,1)=pa3*sqrt(1.-cthe3**2)
      p(ipa+2,3)=pa3*cthe3
      p(ipa+2,4)=sqrt(pa3**2+pm3**2)
      p(ipa+2,5)=pm3
      p(ipa+1,1)=-p(ipa+2,1)
      p(ipa+1,3)=-p(ipa,3)-p(ipa+2,3)
      p(ipa+1,4)=sqrt(p(ipa+1,1)**2+p(ipa+1,3)**2+pm2**2)
      p(ipa+1,5)=pm2

c...set n. optionally fragment/decay.
      n=ipa+2
      if(ip.eq.0) call luexec

      return
      end

c*********************************************************************

      subroutine lu4ent(ip,kf1,kf2,kf3,kf4,pecm,x1,x2,x4,x12,x14)

c...purpose: to store four partons or particles in their cm frame, with
c...the first along the +z axis, the last in the xz plane with x > 0
c...and the second having y < 0 and y > 0 with equal probability.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...standard checks.
      mstu(28)=0
      if(mstu(12).ge.1) call lulist(0)
      ipa=max(1,iabs(ip))
      if(ipa.gt.mstu(4)-3) call luerrm(21,
     &'(lu4ent:) writing outside lujets momory')
      kc1=lucomp(kf1)
      kc2=lucomp(kf2)
      kc3=lucomp(kf3)
      kc4=lucomp(kf4)
      if(kc1.eq.0.or.kc2.eq.0.or.kc3.eq.0.or.kc4.eq.0) call luerrm(12,
     &'(lu4ent:) unknown flavour code')

c...find masses. reset k, p and v vectors.
      pm1=0.
      if(mstu(10).eq.1) pm1=p(ipa,5)
      if(mstu(10).ge.2) pm1=ulmass(kf1)
      pm2=0.
      if(mstu(10).eq.1) pm2=p(ipa+1,5)
      if(mstu(10).ge.2) pm2=ulmass(kf2)
      pm3=0.
      if(mstu(10).eq.1) pm3=p(ipa+2,5)
      if(mstu(10).ge.2) pm3=ulmass(kf3)
      pm4=0.
      if(mstu(10).eq.1) pm4=p(ipa+3,5)
      if(mstu(10).ge.2) pm4=ulmass(kf4)
      do 100 i=ipa,ipa+3
      do 100 j=1,5
      k(i,j)=0
      p(i,j)=0.
  100 v(i,j)=0.

c...check flavours.
      kq1=kchg(kc1,2)*isign(1,kf1)
      kq2=kchg(kc2,2)*isign(1,kf2)
      kq3=kchg(kc3,2)*isign(1,kf3)
      kq4=kchg(kc4,2)*isign(1,kf4)
      if(mstu(19).eq.1) then
        mstu(19)=0
      elseif(kq1.eq.0.and.kq2.eq.0.and.kq3.eq.0.and.kq4.eq.0) then
      elseif(kq1.ne.0.and.kq2.eq.2.and.kq3.eq.2.and.(kq1+kq4.eq.0.or.
     &kq1+kq4.eq.4)) then
      elseif(kq1.ne.0.and.kq1+kq2.eq.0.and.kq3.ne.0.and.kq3+kq4.eq.0.)
     &then
      else
        call luerrm(2,'(lu4ent:) unphysical flavour combination')
      endif
      k(ipa,2)=kf1
      k(ipa+1,2)=kf2
      k(ipa+2,2)=kf3
      k(ipa+3,2)=kf4

c...store partons/particles in k vectors for normal case.
      if(ip.ge.0) then
        k(ipa,1)=1
        if(kq1.ne.0.and.(kq2.ne.0.or.kq3.ne.0.or.kq4.ne.0)) k(ipa,1)=2
        k(ipa+1,1)=1
        if(kq2.ne.0.and.kq1+kq2.ne.0.and.(kq3.ne.0.or.kq4.ne.0))
     &  k(ipa+1,1)=2
        k(ipa+2,1)=1
        if(kq3.ne.0.and.kq4.ne.0) k(ipa+2,1)=2
        k(ipa+3,1)=1

c...store partons for parton shower evolution from q-g-g-qbar or
c...g-g-g-g event.
      elseif(kq1+kq2.ne.0) then
        k(ipa,1)=3
        k(ipa+1,1)=3
        k(ipa+2,1)=3
        k(ipa+3,1)=3
        kcs=4
        if(kq1.eq.-1) kcs=5
        k(ipa,kcs)=mstu(5)*(ipa+1)
        k(ipa,9-kcs)=mstu(5)*(ipa+3)
        k(ipa+1,kcs)=mstu(5)*(ipa+2)
        k(ipa+1,9-kcs)=mstu(5)*ipa
        k(ipa+2,kcs)=mstu(5)*(ipa+3)
        k(ipa+2,9-kcs)=mstu(5)*(ipa+1)
        k(ipa+3,kcs)=mstu(5)*ipa
        k(ipa+3,9-kcs)=mstu(5)*(ipa+2)

c...store partons for parton shower evolution from q-qbar-q-qbar event.
      else
        k(ipa,1)=3
        k(ipa+1,1)=3
        k(ipa+2,1)=3
        k(ipa+3,1)=3
        k(ipa,4)=mstu(5)*(ipa+1)
        k(ipa,5)=k(ipa,4)
        k(ipa+1,4)=mstu(5)*ipa
        k(ipa+1,5)=k(ipa+1,4)
        k(ipa+2,4)=mstu(5)*(ipa+3)
        k(ipa+2,5)=k(ipa+2,4)
        k(ipa+3,4)=mstu(5)*(ipa+2)
        k(ipa+3,5)=k(ipa+3,4)
      endif

c...check kinematics.
      mkerr=0
      if(0.5*x1*pecm.le.pm1.or.0.5*x2*pecm.le.pm2.or.0.5*(2.-x1-x2-x4)*
     &pecm.le.pm3.or.0.5*x4*pecm.le.pm4) mkerr=1
      pa1=sqrt(max(1e-10,(0.5*x1*pecm)**2-pm1**2))
      pa2=sqrt(max(1e-10,(0.5*x2*pecm)**2-pm2**2))
      pa4=sqrt(max(1e-10,(0.5*x4*pecm)**2-pm4**2))
      x24=x1+x2+x4-1.-x12-x14+(pm3**2-pm1**2-pm2**2-pm4**2)/pecm**2
      cthe4=(x1*x4-2.*x14)*pecm**2/(4.*pa1*pa4)
      if(abs(cthe4).ge.1.002) mkerr=1
      cthe4=max(-1.,min(1.,cthe4))
      sthe4=sqrt(1.-cthe4**2)
      cthe2=(x1*x2-2.*x12)*pecm**2/(4.*pa1*pa2)
      if(abs(cthe2).ge.1.002) mkerr=1
      cthe2=max(-1.,min(1.,cthe2))
      sthe2=sqrt(1.-cthe2**2)
      cphi2=((x2*x4-2.*x24)*pecm**2-4.*pa2*cthe2*pa4*cthe4)/
     &max(1e-8*pecm**2,4.*pa2*sthe2*pa4*sthe4)
      if(abs(cphi2).ge.1.05) mkerr=1
      cphi2=max(-1.,min(1.,cphi2))
      if(mkerr.eq.1) call luerrm(13,
     &'(lu4ent:) unphysical kinematical variable setup')

c...store partons/particles in p vectors.
      p(ipa,3)=pa1
      p(ipa,4)=sqrt(pa1**2+pm1**2)
      p(ipa,5)=pm1
      p(ipa+3,1)=pa4*sthe4
      p(ipa+3,3)=pa4*cthe4
      p(ipa+3,4)=sqrt(pa4**2+pm4**2)
      p(ipa+3,5)=pm4
      p(ipa+1,1)=pa2*sthe2*cphi2
      p(ipa+1,2)=pa2*sthe2*sqrt(1.-cphi2**2)*(-1.)**int(rlu(0)+0.5)
      p(ipa+1,3)=pa2*cthe2
      p(ipa+1,4)=sqrt(pa2**2+pm2**2)
      p(ipa+1,5)=pm2
      p(ipa+2,1)=-p(ipa+1,1)-p(ipa+3,1)
      p(ipa+2,2)=-p(ipa+1,2)
      p(ipa+2,3)=-p(ipa,3)-p(ipa+1,3)-p(ipa+3,3)
      p(ipa+2,4)=sqrt(p(ipa+2,1)**2+p(ipa+2,2)**2+p(ipa+2,3)**2+pm3**2)
      p(ipa+2,5)=pm3

c...set n. optionally fragment/decay.
      n=ipa+3
      if(ip.eq.0) call luexec

      return
      end

c*********************************************************************

      subroutine lujoin(njoin,ijoin)

c...purpose: to connect a sequence of partons with colour flow indices,
c...as required for subsequent shower evolution (or other operations).
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension ijoin(*)

c...check that partons are of right types to be connected.
      if(njoin.lt.2) goto 120
      kqsum=0
      do 100 ijn=1,njoin
      i=ijoin(ijn)
      if(i.le.0.or.i.gt.n) goto 120
      if(k(i,1).lt.1.or.k(i,1).gt.3) goto 120
      kc=lucomp(k(i,2))
      if(kc.eq.0) goto 120
      kq=kchg(kc,2)*isign(1,k(i,2))
      if(kq.eq.0) goto 120
      if(ijn.ne.1.and.ijn.ne.njoin.and.kq.ne.2) goto 120
      if(kq.ne.2) kqsum=kqsum+kq
  100 if(ijn.eq.1) kqs=kq
      if(kqsum.ne.0) goto 120

c...connect the partons sequentially (closing for gluon loop).
      kcs=(9-kqs)/2
      if(kqs.eq.2) kcs=int(4.5+rlu(0))
      do 110 ijn=1,njoin
      i=ijoin(ijn)
      k(i,1)=3
      if(ijn.ne.1) ip=ijoin(ijn-1)
      if(ijn.eq.1) ip=ijoin(njoin)
      if(ijn.ne.njoin) in=ijoin(ijn+1)
      if(ijn.eq.njoin) in=ijoin(1)
      k(i,kcs)=mstu(5)*in
      k(i,9-kcs)=mstu(5)*ip
      if(ijn.eq.1.and.kqs.ne.2) k(i,9-kcs)=0
  110 if(ijn.eq.njoin.and.kqs.ne.2) k(i,kcs)=0

c...error exit: no action taken.
      return
  120 call luerrm(12,
     &'(lujoin:) given entries can not be joined by one string')

      return
      end

c*********************************************************************

      subroutine lugive(chin)

c...purpose: to set values of commonblock variables (also in pythia!).
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      common/ludat4/chaf(500)
      character chaf*8
      common/ludatr/mrlu(6),rrlu(100)
      common/pysubs/msel,msub(200),kfin(2,-40:40),ckin(200)
      common/pypars/mstp(200),parp(200),msti(200),pari(200)
      common/pyint1/mint(400),vint(400)
      common/pyint2/iset(200),kfpr(200,2),coef(200,20),icol(40,4,2)
      common/pyint3/xsfx(2,-40:40),isig(1000,3),sigh(1000)
      common/pyint4/widp(21:40,0:40),wide(21:40,0:40),wids(21:40,3)
      common/pyint5/ngen(0:200,3),xsec(0:200,3)
      common/pyint6/proc(0:200)
      character proc*28
      save /lujets/,/ludat1/,/ludat2/,/ludat3/,/ludat4/,/ludatr/
      save /pysubs/,/pypars/,/pyint1/,/pyint2/,/pyint3/,/pyint4/,
     &/pyint5/,/pyint6/
      character chin*(*),chfix*104,chbit*104,chold*8,chnew*8,chold2*28,
     &chnew2*28,chnam*4,chvar(42)*4,chalp(2)*26,chind*8,chini*10,
     &chinr*16
      dimension msvar(42,8)

c...for each variable to be translated give: name,
c...integer/real/character, no. of indices, lower&upper index bounds.
      data chvar/'n','k','p','v','mstu','paru','mstj','parj','kchg',
     &'pmas','parf','vckm','mdcy','mdme','brat','kfdp','chaf','mrlu',
     &'rrlu','msel','msub','kfin','ckin','mstp','parp','msti','pari',
     &'mint','vint','iset','kfpr','coef','icol','xsfx','isig','sigh',
     &'widp','wide','wids','ngen','xsec','proc'/
      data ((msvar(i,j),j=1,8),i=1,42)/ 1,7*0,  1,2,1,4000,1,5,2*0,
     & 2,2,1,4000,1,5,2*0,  2,2,1,4000,1,5,2*0,  1,1,1,200,4*0,
     & 2,1,1,200,4*0,  1,1,1,200,4*0,  2,1,1,200,4*0,
     & 1,2,1,500,1,3,2*0,  2,2,1,500,1,4,2*0,  2,1,1,2000,4*0,
     & 2,2,1,4,1,4,2*0,  1,2,1,500,1,3,2*0,  1,2,1,2000,1,2,2*0,
     & 2,1,1,2000,4*0,  1,2,1,2000,1,5,2*0,  3,1,1,500,4*0,
     & 1,1,1,6,4*0,  2,1,1,100,4*0,
     & 1,7*0,  1,1,1,200,4*0,  1,2,1,2,-40,40,2*0,  2,1,1,200,4*0,
     & 1,1,1,200,4*0,  2,1,1,200,4*0,  1,1,1,200,4*0,  2,1,1,200,4*0,
     & 1,1,1,400,4*0,  2,1,1,400,4*0,  1,1,1,200,4*0,
     & 1,2,1,200,1,2,2*0,  2,2,1,200,1,20,2*0,  1,3,1,40,1,4,1,2,
     & 2,2,1,2,-40,40,2*0,  1,2,1,1000,1,3,2*0,  2,1,1,1000,4*0,
     & 2,2,21,40,0,40,2*0,  2,2,21,40,0,40,2*0,  2,2,21,40,1,3,2*0,
     & 1,2,0,200,1,3,2*0,  2,2,0,200,1,3,2*0,  4,1,0,200,4*0/
      data chalp/'abcdefghijklmnopqrstuvwxyz',
     &'abcdefghijklmnopqrstuvwxyz'/

c...length of character variable. subdivide it into instructions.
      if(mstu(12).ge.1) call lulist(0)
      chbit=chin//' '
      lbit=101
  100 lbit=lbit-1
      if(chbit(lbit:lbit).eq.' ') goto 100
      ltot=0
      do 110 lcom=1,lbit
      if(chbit(lcom:lcom).eq.' ') goto 110
      ltot=ltot+1
      chfix(ltot:ltot)=chbit(lcom:lcom)
  110 continue
      llow=0
  120 lhig=llow+1
  130 lhig=lhig+1
      if(lhig.le.ltot.and.chfix(lhig:lhig).ne.';') goto 130
      lbit=lhig-llow-1
      chbit(1:lbit)=chfix(llow+1:lhig-1)

c...identify commonblock variable.
      lnam=1
  140 lnam=lnam+1
      if(chbit(lnam:lnam).ne.'('.and.chbit(lnam:lnam).ne.'='.and.
     &lnam.le.4) goto 140
      chnam=chbit(1:lnam-1)//' '
      do 150 lcom=1,lnam-1
      do 150 lalp=1,26
  150 if(chnam(lcom:lcom).eq.chalp(1)(lalp:lalp)) chnam(lcom:lcom)=
     &chalp(2)(lalp:lalp)
      ivar=0
      do 160 iv=1,42
  160 if(chnam.eq.chvar(iv)) ivar=iv
      if(ivar.eq.0) then
        call luerrm(18,'(lugive:) do not recognize variable '//chnam)
        llow=lhig
        if(llow.lt.ltot) goto 120
        return
      endif

c...identify any indices.
      i1=0
      i2=0
      i3=0
      nindx=0
      if(chbit(lnam:lnam).eq.'(') then
        lind=lnam
  170   lind=lind+1
        if(chbit(lind:lind).ne.')'.and.chbit(lind:lind).ne.',') goto 170
        chind=' '
        if((chbit(lnam+1:lnam+1).eq.'c'.or.chbit(lnam+1:lnam+1).eq.'c').
     &  and.(ivar.eq.9.or.ivar.eq.10.or.ivar.eq.13.or.ivar.eq.17)) then
          chind(lnam-lind+11:8)=chbit(lnam+2:lind-1)
          read(chind,'(i8)') kf
          i1=lucomp(kf)
        elseif(chbit(lnam+1:lnam+1).eq.'c'.or.chbit(lnam+1:lnam+1).eq.
     &  'c') then
          call luerrm(18,'(lugive:) not allowed to use c index for '//
     &    chnam)
          llow=lhig
          if(llow.lt.ltot) goto 120
          return
        else
          chind(lnam-lind+10:8)=chbit(lnam+1:lind-1)
          read(chind,'(i8)') i1
        endif
        lnam=lind
        if(chbit(lnam:lnam).eq.')') lnam=lnam+1
        nindx=1
      endif
      if(chbit(lnam:lnam).eq.',') then
        lind=lnam
  180   lind=lind+1
        if(chbit(lind:lind).ne.')'.and.chbit(lind:lind).ne.',') goto 180
        chind=' '
        chind(lnam-lind+10:8)=chbit(lnam+1:lind-1)
        read(chind,'(i8)') i2
        lnam=lind
        if(chbit(lnam:lnam).eq.')') lnam=lnam+1
        nindx=2
      endif
      if(chbit(lnam:lnam).eq.',') then
        lind=lnam
  190   lind=lind+1
        if(chbit(lind:lind).ne.')'.and.chbit(lind:lind).ne.',') goto 190
        chind=' '
        chind(lnam-lind+10:8)=chbit(lnam+1:lind-1)
        read(chind,'(i8)') i3
        lnam=lind+1
        nindx=3
      endif

c...check that indices allowed.
      ierr=0
      if(nindx.ne.msvar(ivar,2)) ierr=1
      if(nindx.ge.1.and.(i1.lt.msvar(ivar,3).or.i1.gt.msvar(ivar,4)))
     &ierr=2
      if(nindx.ge.2.and.(i2.lt.msvar(ivar,5).or.i2.gt.msvar(ivar,6)))
     &ierr=3
      if(nindx.eq.3.and.(i3.lt.msvar(ivar,7).or.i3.gt.msvar(ivar,8)))
     &ierr=4
      if(chbit(lnam:lnam).ne.'=') ierr=5
      if(ierr.ge.1) then
        call luerrm(18,'(lugive:) unallowed indices for '//
     &  chbit(1:lnam-1))
        llow=lhig
        if(llow.lt.ltot) goto 120
        return
      endif

c...save old value of variable.
      if(ivar.eq.1) then
        iold=n
      elseif(ivar.eq.2) then
        iold=k(i1,i2)
      elseif(ivar.eq.3) then
        rold=p(i1,i2)
      elseif(ivar.eq.4) then
        rold=v(i1,i2)
      elseif(ivar.eq.5) then
        iold=mstu(i1)
      elseif(ivar.eq.6) then
        rold=paru(i1)
      elseif(ivar.eq.7) then
        iold=mstj(i1)
      elseif(ivar.eq.8) then
        rold=parj(i1)
      elseif(ivar.eq.9) then
        iold=kchg(i1,i2)
      elseif(ivar.eq.10) then
        rold=pmas(i1,i2)
      elseif(ivar.eq.11) then
        rold=parf(i1)
      elseif(ivar.eq.12) then
        rold=vckm(i1,i2)
      elseif(ivar.eq.13) then
        iold=mdcy(i1,i2)
      elseif(ivar.eq.14) then
        iold=mdme(i1,i2)
      elseif(ivar.eq.15) then
        rold=brat(i1)
      elseif(ivar.eq.16) then
        iold=kfdp(i1,i2)
      elseif(ivar.eq.17) then
        chold=chaf(i1)
      elseif(ivar.eq.18) then
        iold=mrlu(i1)
      elseif(ivar.eq.19) then
        rold=rrlu(i1)
      elseif(ivar.eq.20) then
        iold=msel
      elseif(ivar.eq.21) then
        iold=msub(i1)
      elseif(ivar.eq.22) then
        iold=kfin(i1,i2)
      elseif(ivar.eq.23) then
        rold=ckin(i1)
      elseif(ivar.eq.24) then
        iold=mstp(i1)
      elseif(ivar.eq.25) then
        rold=parp(i1)
      elseif(ivar.eq.26) then
        iold=msti(i1)
      elseif(ivar.eq.27) then
        rold=pari(i1)
      elseif(ivar.eq.28) then
        iold=mint(i1)
      elseif(ivar.eq.29) then
        rold=vint(i1)
      elseif(ivar.eq.30) then
        iold=iset(i1)
      elseif(ivar.eq.31) then
        iold=kfpr(i1,i2)
      elseif(ivar.eq.32) then
        rold=coef(i1,i2)
      elseif(ivar.eq.33) then
        iold=icol(i1,i2,i3)
      elseif(ivar.eq.34) then
        rold=xsfx(i1,i2)
      elseif(ivar.eq.35) then
        iold=isig(i1,i2)
      elseif(ivar.eq.36) then
        rold=sigh(i1)
      elseif(ivar.eq.37) then
        rold=widp(i1,i2)
      elseif(ivar.eq.38) then
        rold=wide(i1,i2)
      elseif(ivar.eq.39) then
        rold=wids(i1,i2)
      elseif(ivar.eq.40) then
        iold=ngen(i1,i2)
      elseif(ivar.eq.41) then
        rold=xsec(i1,i2)
      elseif(ivar.eq.42) then
        chold2=proc(i1)
      endif

c...print current value of variable. loop back.
      if(lnam.ge.lbit) then
        chbit(lnam:14)=' '
        chbit(15:60)=' has the value                                '
        if(msvar(ivar,1).eq.1) then
          write(chbit(51:60),'(i10)') iold
        elseif(msvar(ivar,1).eq.2) then
          write(chbit(47:60),'(f14.5)') rold
        elseif(msvar(ivar,1).eq.3) then
          chbit(53:60)=chold
        else
          chbit(33:60)=chold
        endif
        if(mstu(13).ge.1) write(mstu(11),5000) chbit(1:60)
        llow=lhig
        if(llow.lt.ltot) goto 120
        return
      endif

c...read in new variable value.
      if(msvar(ivar,1).eq.1) then
        chini=' '
        chini(lnam-lbit+11:10)=chbit(lnam+1:lbit)
        read(chini,'(i10)') inew
      elseif(msvar(ivar,1).eq.2) then
        chinr=' '
        chinr(lnam-lbit+17:16)=chbit(lnam+1:lbit)
        read(chinr,'(f16.2)') rnew
      elseif(msvar(ivar,1).eq.3) then
        chnew=chbit(lnam+1:lbit)//' '
      else
        chnew2=chbit(lnam+1:lbit)//' '
      endif

c...store new variable value.
      if(ivar.eq.1) then
        n=inew
      elseif(ivar.eq.2) then
        k(i1,i2)=inew
      elseif(ivar.eq.3) then
        p(i1,i2)=rnew
      elseif(ivar.eq.4) then
        v(i1,i2)=rnew
      elseif(ivar.eq.5) then
        mstu(i1)=inew
      elseif(ivar.eq.6) then
        paru(i1)=rnew
      elseif(ivar.eq.7) then
        mstj(i1)=inew
      elseif(ivar.eq.8) then
        parj(i1)=rnew
      elseif(ivar.eq.9) then
        kchg(i1,i2)=inew
      elseif(ivar.eq.10) then
        pmas(i1,i2)=rnew
      elseif(ivar.eq.11) then
        parf(i1)=rnew
      elseif(ivar.eq.12) then
        vckm(i1,i2)=rnew
      elseif(ivar.eq.13) then
        mdcy(i1,i2)=inew
      elseif(ivar.eq.14) then
        mdme(i1,i2)=inew
      elseif(ivar.eq.15) then
        brat(i1)=rnew
      elseif(ivar.eq.16) then
        kfdp(i1,i2)=inew
      elseif(ivar.eq.17) then
        chaf(i1)=chnew
      elseif(ivar.eq.18) then
        mrlu(i1)=inew
      elseif(ivar.eq.19) then
        rrlu(i1)=rnew
      elseif(ivar.eq.20) then
        msel=inew
      elseif(ivar.eq.21) then
        msub(i1)=inew
      elseif(ivar.eq.22) then
        kfin(i1,i2)=inew
      elseif(ivar.eq.23) then
        ckin(i1)=rnew
      elseif(ivar.eq.24) then
        mstp(i1)=inew
      elseif(ivar.eq.25) then
        parp(i1)=rnew
      elseif(ivar.eq.26) then
        msti(i1)=inew
      elseif(ivar.eq.27) then
        pari(i1)=rnew
      elseif(ivar.eq.28) then
        mint(i1)=inew
      elseif(ivar.eq.29) then
        vint(i1)=rnew
      elseif(ivar.eq.30) then
        iset(i1)=inew
      elseif(ivar.eq.31) then
        kfpr(i1,i2)=inew
      elseif(ivar.eq.32) then
        coef(i1,i2)=rnew
      elseif(ivar.eq.33) then
        icol(i1,i2,i3)=inew
      elseif(ivar.eq.34) then
        xsfx(i1,i2)=rnew
      elseif(ivar.eq.35) then
        isig(i1,i2)=inew
      elseif(ivar.eq.36) then
        sigh(i1)=rnew
      elseif(ivar.eq.37) then
        widp(i1,i2)=rnew
      elseif(ivar.eq.38) then
        wide(i1,i2)=rnew
      elseif(ivar.eq.39) then
        wids(i1,i2)=rnew
      elseif(ivar.eq.40) then
        ngen(i1,i2)=inew
      elseif(ivar.eq.41) then
        xsec(i1,i2)=rnew
      elseif(ivar.eq.42) then
        proc(i1)=chnew2
      endif

c...write old and new value. loop back.
      chbit(lnam:14)=' '
      chbit(15:60)=' changed from                to               '
      if(msvar(ivar,1).eq.1) then
        write(chbit(33:42),'(i10)') iold
        write(chbit(51:60),'(i10)') inew
        if(mstu(13).ge.1) write(mstu(11),5000) chbit(1:60)
      elseif(msvar(ivar,1).eq.2) then
        write(chbit(29:42),'(f14.5)') rold
        write(chbit(47:60),'(f14.5)') rnew
        if(mstu(13).ge.1) write(mstu(11),5000) chbit(1:60)
      elseif(msvar(ivar,1).eq.3) then
        chbit(35:42)=chold
        chbit(53:60)=chnew
        if(mstu(13).ge.1) write(mstu(11),5000) chbit(1:60)
      else
        chbit(15:88)=' changed from '//chold2//' to '//chnew2
        if(mstu(13).ge.1) write(mstu(11),5100) chbit(1:88)
      endif
      llow=lhig
      if(llow.lt.ltot) goto 120

c...format statement for output on unit mstu(11) (by default 6).
 5000 format(5x,a60)
 5100 format(5x,a88)

      return
      end

c*********************************************************************

      subroutine luexec

c...purpose: to administrate the fragmentation and decay chain.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      save /lujets/,/ludat1/,/ludat2/,/ludat3/
      dimension ps(2,6)

c...initialize and reset.
      mstu(24)=0
      if(mstu(12).ge.1) call lulist(0)
      mstu(31)=mstu(31)+1
      mstu(1)=0
      mstu(2)=0
      mstu(3)=0
      if(mstu(17).le.0) mstu(90)=0
      mcons=1

c...sum up momentum, energy and charge for starting entries.
      nsav=n
      do 100 i=1,2
      do 100 j=1,6
  100 ps(i,j)=0.
      do 120 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 120
      do 110 j=1,4
  110 ps(1,j)=ps(1,j)+p(i,j)
      ps(1,6)=ps(1,6)+luchge(k(i,2))
  120 continue
      paru(21)=ps(1,4)

c...prepare system for subsequent fragmentation/decay.
      call luprep(0)

c...loop through jet fragmentation and particle decays.
      mbe=0
  130 mbe=mbe+1
      ip=0
  140 ip=ip+1
      kc=0
      if(k(ip,1).gt.0.and.k(ip,1).le.10) kc=lucomp(k(ip,2))
      if(kc.eq.0) then

c...particle decay if unstable and allowed. save long-lived particle
c...decays until second pass after bose-einstein effects.
      elseif(kchg(kc,2).eq.0) then
        if(mstj(21).ge.1.and.mdcy(kc,1).ge.1.and.(mstj(51).le.0.or.mbe.
     &  eq.2.or.pmas(kc,2).ge.parj(91).or.iabs(k(ip,2)).eq.311))
     &  call ludecy(ip)

c...decay products may develop a shower.
        if(mstj(92).gt.0) then
          ip1=mstj(92)
          qmax=sqrt(max(0.,(p(ip1,4)+p(ip1+1,4))**2-(p(ip1,1)+p(ip1+1,
     &    1))**2-(p(ip1,2)+p(ip1+1,2))**2-(p(ip1,3)+p(ip1+1,3))**2))
          call lushow(ip1,ip1+1,qmax)
          call luprep(ip1)
          mstj(92)=0
        elseif(mstj(92).lt.0) then
          ip1=-mstj(92)
          call lushow(ip1,-3,p(ip,5))
          call luprep(ip1)
          mstj(92)=0
        endif

c...jet fragmentation: string or independent fragmentation.
      elseif(k(ip,1).eq.1.or.k(ip,1).eq.2) then
        mfrag=mstj(1)
        if(mfrag.ge.1.and.k(ip,1).eq.1) mfrag=2
        if(mstj(21).ge.2.and.k(ip,1).eq.2.and.n.gt.ip) then
          if(k(ip+1,1).eq.1.and.k(ip+1,3).eq.k(ip,3).and.
     &    k(ip,3).gt.0.and.k(ip,3).lt.ip) then
            if(kchg(lucomp(k(k(ip,3),2)),2).eq.0) mfrag=min(1,mfrag)
          endif
        endif
        if(mfrag.eq.1) call lustrf(ip)
        if(mfrag.eq.2) call luindf(ip)
        if(mfrag.eq.2.and.k(ip,1).eq.1) mcons=0
        if(mfrag.eq.2.and.(mstj(3).le.0.or.mod(mstj(3),5).eq.0)) mcons=0
      endif

c...loop back if enough space left in lujets and no error abort.
      if(mstu(24).ne.0.and.mstu(21).ge.2) then
      elseif(ip.lt.n.and.n.lt.mstu(4)-20-mstu(32)) then
        goto 140
      elseif(ip.lt.n) then
        call luerrm(11,'(luexec:) no more memory left in lujets')
      endif

c...include simple bose-einstein effect parametrization if desired.
      if(mbe.eq.1.and.mstj(51).ge.1) then
        call luboei(nsav)
        goto 130
      endif

c...check that momentum, energy and charge were conserved.
      do 160 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 160
      do 150 j=1,4
  150 ps(2,j)=ps(2,j)+p(i,j)
      ps(2,6)=ps(2,6)+luchge(k(i,2))
  160 continue
      pdev=(abs(ps(2,1)-ps(1,1))+abs(ps(2,2)-ps(1,2))+abs(ps(2,3)-
     &ps(1,3))+abs(ps(2,4)-ps(1,4)))/(1.+abs(ps(2,4))+abs(ps(1,4)))
      if(mcons.eq.1.and.pdev.gt.paru(11)) call luerrm(15,
     &'(luexec:) four-momentum was not conserved')
      if(mcons.eq.1.and.abs(ps(2,6)-ps(1,6)).gt.0.1) call luerrm(15,
     &'(luexec:) charge was not conserved')

      return
      end

c*********************************************************************

      subroutine luprep(ip)

c...purpose: to rearrange partons along strings, to allow small systems
c...to collapse into one or two particles and to check flavours.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      save /lujets/,/ludat1/,/ludat2/,/ludat3/
      dimension dps(5),dpc(5),ue(3)

c...rearrange parton shower product listing along strings: begin loop.
      i1=n
      do 130 mqgst=1,2
      do 120 i=max(1,ip),n
      if(k(i,1).ne.3) goto 120
      kc=lucomp(k(i,2))
      if(kc.eq.0) goto 120
      kq=kchg(kc,2)
      if(kq.eq.0.or.(mqgst.eq.1.and.kq.eq.2)) goto 120

c...pick up loose string end.
      kcs=4
      if(kq*isign(1,k(i,2)).lt.0) kcs=5
      ia=i
      nstp=0
  100 nstp=nstp+1
      if(nstp.gt.4*n) then
        call luerrm(14,'(luprep:) caught in infinite loop')
        return
      endif

c...copy undecayed parton.
      if(k(ia,1).eq.3) then
        if(i1.ge.mstu(4)-mstu(32)-5) then
          call luerrm(11,'(luprep:) no more memory left in lujets')
          return
        endif
        i1=i1+1
        k(i1,1)=2
        if(nstp.ge.2.and.iabs(k(ia,2)).ne.21) k(i1,1)=1
        k(i1,2)=k(ia,2)
        k(i1,3)=ia
        k(i1,4)=0
        k(i1,5)=0
        do 110 j=1,5
        p(i1,j)=p(ia,j)
  110   v(i1,j)=v(ia,j)
        k(ia,1)=k(ia,1)+10
        if(k(i1,1).eq.1) goto 120
      endif

c...go to next parton in colour space.
      ib=ia
      if(mod(k(ib,kcs)/mstu(5)**2,2).eq.0.and.mod(k(ib,kcs),mstu(5)).
     &ne.0) then
        ia=mod(k(ib,kcs),mstu(5))
        k(ib,kcs)=k(ib,kcs)+mstu(5)**2
        mrev=0
      else
        if(k(ib,kcs).ge.2*mstu(5)**2.or.mod(k(ib,kcs)/mstu(5),mstu(5)).
     &  eq.0) kcs=9-kcs
        ia=mod(k(ib,kcs)/mstu(5),mstu(5))
        k(ib,kcs)=k(ib,kcs)+2*mstu(5)**2
        mrev=1
      endif
      if(ia.le.0.or.ia.gt.n) then
        call luerrm(12,'(luprep:) colour rearrangement failed')
        return
      endif
      if(mod(k(ia,4)/mstu(5),mstu(5)).eq.ib.or.mod(k(ia,5)/mstu(5),
     &mstu(5)).eq.ib) then
        if(mrev.eq.1) kcs=9-kcs
        if(mod(k(ia,kcs)/mstu(5),mstu(5)).ne.ib) kcs=9-kcs
        k(ia,kcs)=k(ia,kcs)+2*mstu(5)**2
      else
        if(mrev.eq.0) kcs=9-kcs
        if(mod(k(ia,kcs),mstu(5)).ne.ib) kcs=9-kcs
        k(ia,kcs)=k(ia,kcs)+mstu(5)**2
      endif
      if(ia.ne.i) goto 100
      k(i1,1)=1
  120 continue
  130 continue
      n=i1
      if(mstj(14).lt.0) return

c...find lowest-mass colour singlet jet system, ok if above threshold.
      if(mstj(14).eq.0) goto 320
      ns=n
  140 nsin=n-ns
      pdm=1.+parj(32)
      ic=0
      do 190 i=max(1,ip),ns
      if(k(i,1).ne.1.and.k(i,1).ne.2) then
      elseif(k(i,1).eq.2.and.ic.eq.0) then
        nsin=nsin+1
        ic=i
        do 150 j=1,4
  150   dps(j)=p(i,j)
        mstj(93)=1
        dps(5)=ulmass(k(i,2))
      elseif(k(i,1).eq.2) then
        do 160 j=1,4
  160   dps(j)=dps(j)+p(i,j)
      elseif(ic.ne.0.and.kchg(lucomp(k(i,2)),2).ne.0) then
        do 170 j=1,4
  170   dps(j)=dps(j)+p(i,j)
        mstj(93)=1
        dps(5)=dps(5)+ulmass(k(i,2))
        pd=sqrt(max(0d0,dps(4)**2-dps(1)**2-dps(2)**2-dps(3)**2))-dps(5)
        if(pd.lt.pdm) then
          pdm=pd
          do 180 j=1,5
  180     dpc(j)=dps(j)
          ic1=ic
          ic2=i
        endif
        ic=0
      else
        nsin=nsin+1
      endif
  190 continue
      if(pdm.ge.parj(32)) goto 320

c...fill small-mass system as cluster.
      nsav=n
      pecm=sqrt(max(0d0,dpc(4)**2-dpc(1)**2-dpc(2)**2-dpc(3)**2))
      k(n+1,1)=11
      k(n+1,2)=91
      k(n+1,3)=ic1
      k(n+1,4)=n+2
      k(n+1,5)=n+3
      p(n+1,1)=dpc(1)
      p(n+1,2)=dpc(2)
      p(n+1,3)=dpc(3)
      p(n+1,4)=dpc(4)
      p(n+1,5)=pecm

c...form two particles from flavours of lowest-mass system, if feasible.
      k(n+2,1)=1
      k(n+3,1)=1
      if(mstu(16).ne.2) then
        k(n+2,3)=n+1
        k(n+3,3)=n+1
      else
        k(n+2,3)=ic1
        k(n+3,3)=ic2
      endif
      k(n+2,4)=0
      k(n+3,4)=0
      k(n+2,5)=0
      k(n+3,5)=0
      if(iabs(k(ic1,2)).ne.21) then
        kc1=lucomp(k(ic1,2))
        kc2=lucomp(k(ic2,2))
        if(kc1.eq.0.or.kc2.eq.0) goto 320
        kq1=kchg(kc1,2)*isign(1,k(ic1,2))
        kq2=kchg(kc2,2)*isign(1,k(ic2,2))
        if(kq1+kq2.ne.0) goto 320
  200   call lukfdi(k(ic1,2),0,kfln,k(n+2,2))
        call lukfdi(k(ic2,2),-kfln,kfldmp,k(n+3,2))
        if(k(n+2,2).eq.0.or.k(n+3,2).eq.0) goto 200
      else
        if(iabs(k(ic2,2)).ne.21) goto 320
  210   call lukfdi(1+int((2.+parj(2))*rlu(0)),0,kfln,kfdmp)
        call lukfdi(kfln,0,kflm,k(n+2,2))
        call lukfdi(-kfln,-kflm,kfldmp,k(n+3,2))
        if(k(n+2,2).eq.0.or.k(n+3,2).eq.0) goto 210
      endif
      p(n+2,5)=ulmass(k(n+2,2))
      p(n+3,5)=ulmass(k(n+3,2))
      if(p(n+2,5)+p(n+3,5)+parj(64).ge.pecm.and.nsin.eq.1) goto 320
      if(p(n+2,5)+p(n+3,5)+parj(64).ge.pecm) goto 260

c...perform two-particle decay of jet system, if possible.
      if(pecm.ge.0.02*dpc(4)) then
        pa=sqrt((pecm**2-(p(n+2,5)+p(n+3,5))**2)*(pecm**2-
     &  (p(n+2,5)-p(n+3,5))**2))/(2.*pecm)
        ue(3)=2.*rlu(0)-1.
        phi=paru(2)*rlu(0)
        ue(1)=sqrt(1.-ue(3)**2)*cos(phi)
        ue(2)=sqrt(1.-ue(3)**2)*sin(phi)
        do 220 j=1,3
        p(n+2,j)=pa*ue(j)
  220   p(n+3,j)=-pa*ue(j)
        p(n+2,4)=sqrt(pa**2+p(n+2,5)**2)
        p(n+3,4)=sqrt(pa**2+p(n+3,5)**2)
        mstu(33)=1
        call ludbrb(n+2,n+3,0.,0.,dpc(1)/dpc(4),dpc(2)/dpc(4),
     &  dpc(3)/dpc(4))
      else
        np=0
        do 230 i=ic1,ic2
  230   if(k(i,1).eq.1.or.k(i,1).eq.2) np=np+1
        ha=p(ic1,4)*p(ic2,4)-p(ic1,1)*p(ic2,1)-p(ic1,2)*p(ic2,2)-
     &  p(ic1,3)*p(ic2,3)
        if(np.ge.3.or.ha.le.1.25*p(ic1,5)*p(ic2,5)) goto 260
        hd1=0.5*(p(n+2,5)**2-p(ic1,5)**2)
        hd2=0.5*(p(n+3,5)**2-p(ic2,5)**2)
        hr=sqrt(max(0.,((ha-hd1-hd2)**2-(p(n+2,5)*p(n+3,5))**2)/
     &  (ha**2-(p(ic1,5)*p(ic2,5))**2)))-1.
        hc=p(ic1,5)**2+2.*ha+p(ic2,5)**2
        hk1=((p(ic2,5)**2+ha)*hr+hd1-hd2)/hc
        hk2=((p(ic1,5)**2+ha)*hr+hd2-hd1)/hc
        do 240 j=1,4
        p(n+2,j)=(1.+hk1)*p(ic1,j)-hk2*p(ic2,j)
  240   p(n+3,j)=(1.+hk2)*p(ic2,j)-hk1*p(ic1,j)
      endif
      do 250 j=1,4
      v(n+1,j)=v(ic1,j)
      v(n+2,j)=v(ic1,j)
  250 v(n+3,j)=v(ic2,j)
      v(n+1,5)=0.
      v(n+2,5)=0.
      v(n+3,5)=0.
      n=n+3
      goto 300

c...else form one particle from the flavours available, if possible.
  260 k(n+1,5)=n+2
      if(iabs(k(ic1,2)).gt.100.and.iabs(k(ic2,2)).gt.100) then
        goto 320
      elseif(iabs(k(ic1,2)).ne.21) then
        call lukfdi(k(ic1,2),k(ic2,2),kfldmp,k(n+2,2))
      else
        kfln=1+int((2.+parj(2))*rlu(0))
        call lukfdi(kfln,-kfln,kfldmp,k(n+2,2))
      endif
      if(k(n+2,2).eq.0) goto 260
      p(n+2,5)=ulmass(k(n+2,2))

c...find parton/particle which combines to largest extra mass.
      ir=0
      ha=0.
      hsm=0.
      do 280 mcomb=1,3
      if(ir.ne.0) goto 280
      do 270 i=max(1,ip),n
      if(k(i,1).le.0.or.k(i,1).gt.10.or.(i.ge.ic1.and.i.le.ic2.
     &and.k(i,1).ge.1.and.k(i,1).le.2)) goto 270
      if(mcomb.eq.1) kci=lucomp(k(i,2))
      if(mcomb.eq.1.and.kci.eq.0) goto 270
      if(mcomb.eq.1.and.kchg(kci,2).eq.0.and.i.le.ns) goto 270
      if(mcomb.eq.2.and.iabs(k(i,2)).gt.10.and.iabs(k(i,2)).le.100)
     &goto 270
      hcr=dpc(4)*p(i,4)-dpc(1)*p(i,1)-dpc(2)*p(i,2)-dpc(3)*p(i,3)
      hsr=2.*hcr+pecm**2-p(n+2,5)**2-2.*p(n+2,5)*p(i,5)
      if(hsr.gt.hsm) then
        ir=i
        ha=hcr
        hsm=hsr
      endif
  270 continue
  280 continue

c...shuffle energy and momentum to put new particle on mass shell.
      if(ir.ne.0) then
        hb=pecm**2+ha
        hc=p(n+2,5)**2+ha
        hd=p(ir,5)**2+ha
        hk2=0.5*(hb*sqrt(max(0.,((hb+hc)**2-4.*(hb+hd)*p(n+2,5)**2)/
     &  (ha**2-(pecm*p(ir,5))**2)))-(hb+hc))/(hb+hd)
        hk1=(0.5*(p(n+2,5)**2-pecm**2)+hd*hk2)/hb
        do 290 j=1,4
        p(n+2,j)=(1.+hk1)*dpc(j)-hk2*p(ir,j)
        p(ir,j)=(1.+hk2)*p(ir,j)-hk1*dpc(j)
        v(n+1,j)=v(ic1,j)
  290   v(n+2,j)=v(ic1,j)
        v(n+1,5)=0.
        v(n+2,5)=0.
        n=n+2
      else
        call luerrm(3,'(luprep:) no match for collapsing cluster')
        return
      endif

c...mark collapsed system and store daughter pointers. iterate.
  300 do 310 i=ic1,ic2
      if((k(i,1).eq.1.or.k(i,1).eq.2).and.kchg(lucomp(k(i,2)),2).ne.0)
     &then
        k(i,1)=k(i,1)+10
        if(mstu(16).ne.2) then
          k(i,4)=nsav+1
          k(i,5)=nsav+1
        else
          k(i,4)=nsav+2
          k(i,5)=n
        endif
      endif
  310 continue
      if(n.lt.mstu(4)-mstu(32)-5) goto 140

c...check flavours and invariant masses in parton systems.
  320 np=0
      kfn=0
      kqs=0
      do 330 j=1,5
  330 dps(j)=0.
      do 360 i=max(1,ip),n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 360
      kc=lucomp(k(i,2))
      if(kc.eq.0) goto 360
      kq=kchg(kc,2)*isign(1,k(i,2))
      if(kq.eq.0) goto 360
      np=np+1
      if(kq.ne.2) then
        kfn=kfn+1
        kqs=kqs+kq
        mstj(93)=1
        dps(5)=dps(5)+ulmass(k(i,2))
      endif
      do 340 j=1,4
  340 dps(j)=dps(j)+p(i,j)
      if(k(i,1).eq.1) then
        if(np.ne.1.and.(kfn.eq.1.or.kfn.ge.3.or.kqs.ne.0)) call
     &  luerrm(2,'(luprep:) unphysical flavour combination')
        if(np.ne.1.and.dps(4)**2-dps(1)**2-dps(2)**2-dps(3)**2.lt.
     &  (0.9*parj(32)+dps(5))**2) call luerrm(3,
     &  '(luprep:) too small mass in jet system')
        np=0
        kfn=0
        kqs=0
        do 350 j=1,5
  350   dps(j)=0.
      endif
  360 continue

      return
      end

c*********************************************************************

      subroutine lustrf(ip)
c...purpose: to handle the fragmentation of an arbitrary colour singlet
c...jet system according to the lund string fragmentation model.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension dps(5),kfl(3),pmq(3),px(3),py(3),gam(3),ie(2),pr(2),
     &in(9),dhm(4),dhg(4),dp(5,5),irank(2),mju(4),iju(3),pju(5,5),
     &tju(5),kfjh(2),njs(2),kfjs(2),pjs(4,5),mstu9t(8),paru9t(8)

c...function: four-product of two vectors.
      four(i,j)=p(i,4)*p(j,4)-p(i,1)*p(j,1)-p(i,2)*p(j,2)-p(i,3)*p(j,3)
      dfour(i,j)=dp(i,4)*dp(j,4)-dp(i,1)*dp(j,1)-dp(i,2)*dp(j,2)-
     &dp(i,3)*dp(j,3)

c...reset counters. identify parton system.
      mstj(91)=0
      nsav=n
      mstu90=mstu(90)
      np=0
      kqsum=0
      do 100 j=1,5
  100 dps(j)=0d0
      mju(1)=0
      mju(2)=0
      i=ip-1
  110 i=i+1
      if(i.gt.min(n,mstu(4)-mstu(32))) then
        call luerrm(12,'(lustrf:) failed to reconstruct jet system')
        if(mstu(21).ge.1) return
      endif
      if(k(i,1).ne.1.and.k(i,1).ne.2.and.k(i,1).ne.41) goto 110
      kc=lucomp(k(i,2))
      if(kc.eq.0) goto 110
      kq=kchg(kc,2)*isign(1,k(i,2))
      if(kq.eq.0) goto 110
      if(n+5*np+11.gt.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lustrf:) no more memory left in lujets')
        if(mstu(21).ge.1) return
      endif

c...take copy of partons to be considered. check flavour sum.
      np=np+1
      do 120 j=1,5
      k(n+np,j)=k(i,j)
      p(n+np,j)=p(i,j)
  120 if(j.ne.4) dps(j)=dps(j)+p(i,j)
      dps(4)=dps(4)+sqrt(dble(p(i,1))**2+dble(p(i,2))**2+
     &dble(p(i,3))**2+dble(p(i,5))**2)
      k(n+np,3)=i
      if(kq.ne.2) kqsum=kqsum+kq
      if(k(i,1).eq.41) then
        kqsum=kqsum+2*kq
        if(kqsum.eq.kq) mju(1)=n+np
        if(kqsum.ne.kq) mju(2)=n+np
      endif
      if(k(i,1).eq.2.or.k(i,1).eq.41) goto 110
      if(kqsum.ne.0) then
        call luerrm(12,'(lustrf:) unphysical flavour combination')
        if(mstu(21).ge.1) return
      endif

c...boost copied system to cm frame (for better numerical precision).
      if(abs(dps(3)).lt.0.99d0*dps(4)) then
        mbst=0
        mstu(33)=1
        call ludbrb(n+1,n+np,0.,0.,-dps(1)/dps(4),-dps(2)/dps(4),
     &  -dps(3)/dps(4))
      else
        mbst=1
        hhbz=sqrt(max(1d-6,dps(4)+dps(3))/max(1d-6,dps(4)-dps(3)))
        do 130 i=n+1,n+np
        hhpmt=p(i,1)**2+p(i,2)**2+p(i,5)**2
        if(p(i,3).gt.0.) then
          hhpez=(p(i,4)+p(i,3))/hhbz
          p(i,3)=0.5*(hhpez-hhpmt/hhpez)
          p(i,4)=0.5*(hhpez+hhpmt/hhpez)
        else
          hhpez=(p(i,4)-p(i,3))*hhbz
          p(i,3)=-0.5*(hhpez-hhpmt/hhpez)
          p(i,4)=0.5*(hhpez+hhpmt/hhpez)
        endif
  130   continue
      endif

c...search for very nearby partons that may be recombined.
      ntryr=0
      paru12=paru(12)
      paru13=paru(13)
      mju(3)=mju(1)
      mju(4)=mju(2)
      nr=np
  140 if(nr.ge.3) then
        pdrmin=2.*paru12
        do 150 i=n+1,n+nr
        if(i.eq.n+nr.and.iabs(k(n+1,2)).ne.21) goto 150
        i1=i+1
        if(i.eq.n+nr) i1=n+1
        if(k(i,1).eq.41.or.k(i1,1).eq.41) goto 150
        if(mju(1).ne.0.and.i1.lt.mju(1).and.iabs(k(i1,2)).ne.21)
     &  goto 150
        if(mju(2).ne.0.and.i.gt.mju(2).and.iabs(k(i,2)).ne.21) goto 150
        pap=sqrt((p(i,1)**2+p(i,2)**2+p(i,3)**2)*(p(i1,1)**2+
     &  p(i1,2)**2+p(i1,3)**2))
        pvp=p(i,1)*p(i1,1)+p(i,2)*p(i1,2)+p(i,3)*p(i1,3)
        pdr=4.*(pap-pvp)**2/max(1e-6,paru13**2*pap+2.*(pap-pvp))
        if(pdr.lt.pdrmin) then
          ir=i
          pdrmin=pdr
        endif
  150   continue

c...recombine very nearby partons to avoid machine precision problems.
        if(pdrmin.lt.paru12.and.ir.eq.n+nr) then
          do 160 j=1,4
  160     p(n+1,j)=p(n+1,j)+p(n+nr,j)
          p(n+1,5)=sqrt(max(0.,p(n+1,4)**2-p(n+1,1)**2-p(n+1,2)**2-
     &    p(n+1,3)**2))
          nr=nr-1
          goto 140
        elseif(pdrmin.lt.paru12) then
          do 170 j=1,4
  170     p(ir,j)=p(ir,j)+p(ir+1,j)
          p(ir,5)=sqrt(max(0.,p(ir,4)**2-p(ir,1)**2-p(ir,2)**2-
     &    p(ir,3)**2))
          do 180 i=ir+1,n+nr-1
          k(i,2)=k(i+1,2)
          do 180 j=1,5
  180     p(i,j)=p(i+1,j)
          if(ir.eq.n+nr-1) k(ir,2)=k(n+nr,2)
          nr=nr-1
          if(mju(1).gt.ir) mju(1)=mju(1)-1
          if(mju(2).gt.ir) mju(2)=mju(2)-1
          goto 140
        endif
      endif
      ntryr=ntryr+1

c...reset particle counter. skip ahead if no junctions are present;
c...this is usually the case!
      nrs=max(5*nr+11,np)
      ntry=0
  190 ntry=ntry+1
      if(ntry.gt.100.and.ntryr.le.4) then
        paru12=4.*paru12
        paru13=2.*paru13
        goto 140
      elseif(ntry.gt.100) then
        call luerrm(14,'(lustrf:) caught in infinite loop')
        if(mstu(21).ge.1) return
      endif
      i=n+nrs
      mstu(90)=mstu90
      if(mju(1).eq.0.and.mju(2).eq.0) goto 510
      do 500 jt=1,2
      njs(jt)=0
      if(mju(jt).eq.0) goto 500
      js=3-2*jt

c...find and sum up momentum on three sides of junction. check flavours.
      do 200 iu=1,3
      iju(iu)=0
      do 200 j=1,5
  200 pju(iu,j)=0.
      iu=0
      do 210 i1=n+1+(jt-1)*(nr-1),n+nr+(jt-1)*(1-nr),js
      if(k(i1,2).ne.21.and.iu.le.2) then
        iu=iu+1
        iju(iu)=i1
      endif
      do 210 j=1,4
  210 pju(iu,j)=pju(iu,j)+p(i1,j)
      do 220 iu=1,3
  220 pju(iu,5)=sqrt(pju(iu,1)**2+pju(iu,2)**2+pju(iu,3)**2)
      if(k(iju(3),2)/100.ne.10*k(iju(1),2)+k(iju(2),2).and.
     &k(iju(3),2)/100.ne.10*k(iju(2),2)+k(iju(1),2)) then
        call luerrm(12,'(lustrf:) unphysical flavour combination')
        if(mstu(21).ge.1) return
      endif

c...calculate (approximate) boost to rest frame of junction.
      t12=(pju(1,1)*pju(2,1)+pju(1,2)*pju(2,2)+pju(1,3)*pju(2,3))/
     &(pju(1,5)*pju(2,5))
      t13=(pju(1,1)*pju(3,1)+pju(1,2)*pju(3,2)+pju(1,3)*pju(3,3))/
     &(pju(1,5)*pju(3,5))
      t23=(pju(2,1)*pju(3,1)+pju(2,2)*pju(3,2)+pju(2,3)*pju(3,3))/
     &(pju(2,5)*pju(3,5))
      t11=sqrt((2./3.)*(1.-t12)*(1.-t13)/(1.-t23))
      t22=sqrt((2./3.)*(1.-t12)*(1.-t23)/(1.-t13))
      tsq=sqrt((2.*t11*t22+t12-1.)*(1.+t12))
      t1f=(tsq-t22*(1.+t12))/(1.-t12**2)
      t2f=(tsq-t11*(1.+t12))/(1.-t12**2)
      do 230 j=1,3
  230 tju(j)=-(t1f*pju(1,j)/pju(1,5)+t2f*pju(2,j)/pju(2,5))
      tju(4)=sqrt(1.+tju(1)**2+tju(2)**2+tju(3)**2)
      do 240 iu=1,3
  240 pju(iu,5)=tju(4)*pju(iu,4)-tju(1)*pju(iu,1)-tju(2)*pju(iu,2)-
     &tju(3)*pju(iu,3)

c...put junction at rest if motion could give inconsistencies.
      if(pju(1,5)+pju(2,5).gt.pju(1,4)+pju(2,4)) then
        do 250 j=1,3
  250   tju(j)=0.
        tju(4)=1.
        pju(1,5)=pju(1,4)
        pju(2,5)=pju(2,4)
        pju(3,5)=pju(3,4)
      endif

c...start preparing for fragmentation of two strings from junction.
      ista=i
      do 480 iu=1,2
      ns=iju(iu+1)-iju(iu)

c...junction strings: find longitudinal string directions.
      do 270 is=1,ns
      is1=iju(iu)+is-1
      is2=iju(iu)+is
      do 260 j=1,5
      dp(1,j)=0.5*p(is1,j)
      if(is.eq.1) dp(1,j)=p(is1,j)
      dp(2,j)=0.5*p(is2,j)
  260 if(is.eq.ns) dp(2,j)=-pju(iu,j)
      if(is.eq.ns) dp(2,4)=sqrt(pju(iu,1)**2+pju(iu,2)**2+pju(iu,3)**2)
      if(is.eq.ns) dp(2,5)=0.
      dp(3,5)=dfour(1,1)
      dp(4,5)=dfour(2,2)
      dhkc=dfour(1,2)
      if(dp(3,5)+2.*dhkc+dp(4,5).le.0.) then
        dp(1,4)=sqrt(dp(1,1)**2+dp(1,2)**2+dp(1,3)**2)
        dp(2,4)=sqrt(dp(2,1)**2+dp(2,2)**2+dp(2,3)**2)
        dp(3,5)=0d0
        dp(4,5)=0d0
        dhkc=dfour(1,2)
      endif
      dhks=sqrt(dhkc**2-dp(3,5)*dp(4,5))
      dhk1=0.5*((dp(4,5)+dhkc)/dhks-1.)
      dhk2=0.5*((dp(3,5)+dhkc)/dhks-1.)
      in1=n+nr+4*is-3
      p(in1,5)=sqrt(dp(3,5)+2.*dhkc+dp(4,5))
      do 270 j=1,4
      p(in1,j)=(1.+dhk1)*dp(1,j)-dhk2*dp(2,j)
  270 p(in1+1,j)=(1.+dhk2)*dp(2,j)-dhk1*dp(1,j)

c...junction strings: initialize flavour, momentum and starting pos.
      isav=i
      mstu91=mstu(90)
  280 ntry=ntry+1
      if(ntry.gt.100.and.ntryr.le.4) then
        paru12=4.*paru12
        paru13=2.*paru13
        goto 140
      elseif(ntry.gt.100) then
        call luerrm(14,'(lustrf:) caught in infinite loop')
        if(mstu(21).ge.1) return
      endif
      i=isav
      mstu(90)=mstu91
      irankj=0
      ie(1)=k(n+1+(jt/2)*(np-1),3)
      in(4)=n+nr+1
      in(5)=in(4)+1
      in(6)=n+nr+4*ns+1
      do 290 jq=1,2
      do 290 in1=n+nr+2+jq,n+nr+4*ns-2+jq,4
      p(in1,1)=2-jq
      p(in1,2)=jq-1
  290 p(in1,3)=1.
      kfl(1)=k(iju(iu),2)
      px(1)=0.
      py(1)=0.
      gam(1)=0.
      do 300 j=1,5
  300 pju(iu+3,j)=0.

c...junction strings: find initial transverse directions.
      do 310 j=1,4
      dp(1,j)=p(in(4),j)
      dp(2,j)=p(in(4)+1,j)
      dp(3,j)=0.
  310 dp(4,j)=0.
      dp(1,4)=sqrt(dp(1,1)**2+dp(1,2)**2+dp(1,3)**2)
      dp(2,4)=sqrt(dp(2,1)**2+dp(2,2)**2+dp(2,3)**2)
      dp(5,1)=dp(1,1)/dp(1,4)-dp(2,1)/dp(2,4)
      dp(5,2)=dp(1,2)/dp(1,4)-dp(2,2)/dp(2,4)
      dp(5,3)=dp(1,3)/dp(1,4)-dp(2,3)/dp(2,4)
      if(dp(5,1)**2.le.dp(5,2)**2+dp(5,3)**2) dp(3,1)=1.
      if(dp(5,1)**2.gt.dp(5,2)**2+dp(5,3)**2) dp(3,3)=1.
      if(dp(5,2)**2.le.dp(5,1)**2+dp(5,3)**2) dp(4,2)=1.
      if(dp(5,2)**2.gt.dp(5,1)**2+dp(5,3)**2) dp(4,3)=1.
      dhc12=dfour(1,2)
      dhcx1=dfour(3,1)/dhc12
      dhcx2=dfour(3,2)/dhc12
      dhcxx=1d0/sqrt(1d0+2d0*dhcx1*dhcx2*dhc12)
      dhcy1=dfour(4,1)/dhc12
      dhcy2=dfour(4,2)/dhc12
      dhcyx=dhcxx*(dhcx1*dhcy2+dhcx2*dhcy1)*dhc12
      dhcyy=1d0/sqrt(1d0+2d0*dhcy1*dhcy2*dhc12-dhcyx**2)
      do 320 j=1,4
      dp(3,j)=dhcxx*(dp(3,j)-dhcx2*dp(1,j)-dhcx1*dp(2,j))
      p(in(6),j)=dp(3,j)
  320 p(in(6)+1,j)=dhcyy*(dp(4,j)-dhcy2*dp(1,j)-dhcy1*dp(2,j)-
     &dhcyx*dp(3,j))

c...junction strings: produce new particle, origin.
  330 i=i+1
      if(2*i-nsav.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lustrf:) no more memory left in lujets')
        if(mstu(21).ge.1) return
      endif
      irankj=irankj+1
      k(i,1)=1
      k(i,3)=ie(1)
      k(i,4)=0
      k(i,5)=0

c...junction strings: generate flavour, hadron, pt, z and gamma.
  340 call lukfdi(kfl(1),0,kfl(3),k(i,2))
      if(k(i,2).eq.0) goto 280
      if(mstj(12).ge.3.and.irankj.eq.1.and.iabs(kfl(1)).le.10.and.
     &iabs(kfl(3)).gt.10) then
        if(rlu(0).gt.parj(19)) goto 340
      endif
      p(i,5)=ulmass(k(i,2))
      call luptdi(kfl(1),px(3),py(3))
      pr(1)=p(i,5)**2+(px(1)+px(3))**2+(py(1)+py(3))**2
      call luzdis(kfl(1),kfl(3),pr(1),z)
      if(iabs(kfl(1)).ge.4.and.iabs(kfl(1)).le.8.and.
     &mstu(90).lt.8) then
        mstu(90)=mstu(90)+1
        mstu(90+mstu(90))=i
        paru(90+mstu(90))=z
      endif
      gam(3)=(1.-z)*(gam(1)+pr(1)/z)
      do 350 j=1,3
  350 in(j)=in(3+j)

c...junction strings: stepping within or from 'low' string region easy.
      if(in(1)+1.eq.in(2).and.z*p(in(1)+2,3)*p(in(2)+2,3)*
     &p(in(1),5)**2.ge.pr(1)) then
        p(in(1)+2,4)=z*p(in(1)+2,3)
        p(in(2)+2,4)=pr(1)/(p(in(1)+2,4)*p(in(1),5)**2)
        do 360 j=1,4
  360   p(i,j)=(px(1)+px(3))*p(in(3),j)+(py(1)+py(3))*p(in(3)+1,j)
        goto 430
      elseif(in(1)+1.eq.in(2)) then
        p(in(2)+2,4)=p(in(2)+2,3)
        p(in(2)+2,1)=1.
        in(2)=in(2)+4
        if(in(2).gt.n+nr+4*ns) goto 280
        if(four(in(1),in(2)).le.1e-2) then
          p(in(1)+2,4)=p(in(1)+2,3)
          p(in(1)+2,1)=0.
          in(1)=in(1)+4
        endif
      endif

c...junction strings: find new transverse directions.
  370 if(in(1).gt.n+nr+4*ns.or.in(2).gt.n+nr+4*ns.or.
     &in(1).gt.in(2)) goto 280
      if(in(1).ne.in(4).or.in(2).ne.in(5)) then
        do 380 j=1,4
        dp(1,j)=p(in(1),j)
        dp(2,j)=p(in(2),j)
        dp(3,j)=0.
  380   dp(4,j)=0.
        dp(1,4)=sqrt(dp(1,1)**2+dp(1,2)**2+dp(1,3)**2)
        dp(2,4)=sqrt(dp(2,1)**2+dp(2,2)**2+dp(2,3)**2)
        dhc12=dfour(1,2)
        if(dhc12.le.1e-2) then
          p(in(1)+2,4)=p(in(1)+2,3)
          p(in(1)+2,1)=0.
          in(1)=in(1)+4
          goto 370
        endif
        in(3)=n+nr+4*ns+5
        dp(5,1)=dp(1,1)/dp(1,4)-dp(2,1)/dp(2,4)
        dp(5,2)=dp(1,2)/dp(1,4)-dp(2,2)/dp(2,4)
        dp(5,3)=dp(1,3)/dp(1,4)-dp(2,3)/dp(2,4)
        if(dp(5,1)**2.le.dp(5,2)**2+dp(5,3)**2) dp(3,1)=1.
        if(dp(5,1)**2.gt.dp(5,2)**2+dp(5,3)**2) dp(3,3)=1.
        if(dp(5,2)**2.le.dp(5,1)**2+dp(5,3)**2) dp(4,2)=1.
        if(dp(5,2)**2.gt.dp(5,1)**2+dp(5,3)**2) dp(4,3)=1.
        dhcx1=dfour(3,1)/dhc12
        dhcx2=dfour(3,2)/dhc12
        dhcxx=1d0/sqrt(1d0+2d0*dhcx1*dhcx2*dhc12)
        dhcy1=dfour(4,1)/dhc12
        dhcy2=dfour(4,2)/dhc12
        dhcyx=dhcxx*(dhcx1*dhcy2+dhcx2*dhcy1)*dhc12
        dhcyy=1d0/sqrt(1d0+2d0*dhcy1*dhcy2*dhc12-dhcyx**2)
        do 390 j=1,4
        dp(3,j)=dhcxx*(dp(3,j)-dhcx2*dp(1,j)-dhcx1*dp(2,j))
        p(in(3),j)=dp(3,j)
  390   p(in(3)+1,j)=dhcyy*(dp(4,j)-dhcy2*dp(1,j)-dhcy1*dp(2,j)-
     &  dhcyx*dp(3,j))
c...express pt with respect to new axes, if sensible.
        pxp=-(px(3)*four(in(6),in(3))+py(3)*four(in(6)+1,in(3)))
        pyp=-(px(3)*four(in(6),in(3)+1)+py(3)*four(in(6)+1,in(3)+1))
        if(abs(pxp**2+pyp**2-px(3)**2-py(3)**2).lt.0.01) then
          px(3)=pxp
          py(3)=pyp
        endif
      endif

c...junction strings: sum up known four-momentum, coefficients for m2.
      do 410 j=1,4
      dhg(j)=0.
      p(i,j)=px(1)*p(in(6),j)+py(1)*p(in(6)+1,j)+px(3)*p(in(3),j)+
     &py(3)*p(in(3)+1,j)
      do 400 in1=in(4),in(1)-4,4
  400 p(i,j)=p(i,j)+p(in1+2,3)*p(in1,j)
      do 410 in2=in(5),in(2)-4,4
  410 p(i,j)=p(i,j)+p(in2+2,3)*p(in2,j)
      dhm(1)=four(i,i)
      dhm(2)=2.*four(i,in(1))
      dhm(3)=2.*four(i,in(2))
      dhm(4)=2.*four(in(1),in(2))

c...junction strings: find coefficients for gamma expression.
      do 420 in2=in(1)+1,in(2),4
      do 420 in1=in(1),in2-1,4
      dhc=2.*four(in1,in2)
      dhg(1)=dhg(1)+p(in1+2,1)*p(in2+2,1)*dhc
      if(in1.eq.in(1)) dhg(2)=dhg(2)-p(in2+2,1)*dhc
      if(in2.eq.in(2)) dhg(3)=dhg(3)+p(in1+2,1)*dhc
  420 if(in1.eq.in(1).and.in2.eq.in(2)) dhg(4)=dhg(4)-dhc

c...junction strings: solve (m2, gamma) equation system for energies.
      dhs1=dhm(3)*dhg(4)-dhm(4)*dhg(3)
      if(abs(dhs1).lt.1e-4) goto 280
      dhs2=dhm(4)*(gam(3)-dhg(1))-dhm(2)*dhg(3)-dhg(4)*
     &(p(i,5)**2-dhm(1))+dhg(2)*dhm(3)
      dhs3=dhm(2)*(gam(3)-dhg(1))-dhg(2)*(p(i,5)**2-dhm(1))
      p(in(2)+2,4)=0.5*(sqrt(max(0d0,dhs2**2-4.*dhs1*dhs3))/abs(dhs1)-
     &dhs2/dhs1)
      if(dhm(2)+dhm(4)*p(in(2)+2,4).le.0.) goto 280
      p(in(1)+2,4)=(p(i,5)**2-dhm(1)-dhm(3)*p(in(2)+2,4))/
     &(dhm(2)+dhm(4)*p(in(2)+2,4))

c...junction strings: step to new region if necessary.
      if(p(in(2)+2,4).gt.p(in(2)+2,3)) then
        p(in(2)+2,4)=p(in(2)+2,3)
        p(in(2)+2,1)=1.
        in(2)=in(2)+4
        if(in(2).gt.n+nr+4*ns) goto 280
        if(four(in(1),in(2)).le.1e-2) then
          p(in(1)+2,4)=p(in(1)+2,3)
          p(in(1)+2,1)=0.
          in(1)=in(1)+4
        endif
        goto 370
      elseif(p(in(1)+2,4).gt.p(in(1)+2,3)) then
        p(in(1)+2,4)=p(in(1)+2,3)
        p(in(1)+2,1)=0.
        in(1)=in(1)+js
        goto 720
      endif

c...junction strings: particle four-momentum, remainder, loop back.
  430 do 440 j=1,4
      p(i,j)=p(i,j)+p(in(1)+2,4)*p(in(1),j)+p(in(2)+2,4)*p(in(2),j)
  440 pju(iu+3,j)=pju(iu+3,j)+p(i,j)
      if(p(i,4).lt.p(i,5)) goto 280
      pju(iu+3,5)=tju(4)*pju(iu+3,4)-tju(1)*pju(iu+3,1)-
     &tju(2)*pju(iu+3,2)-tju(3)*pju(iu+3,3)
      if(pju(iu+3,5).lt.pju(iu,5)) then
        kfl(1)=-kfl(3)
        px(1)=-px(3)
        py(1)=-py(3)
        gam(1)=gam(3)
        if(in(3).ne.in(6)) then
          do 450 j=1,4
          p(in(6),j)=p(in(3),j)
  450     p(in(6)+1,j)=p(in(3)+1,j)
        endif
        do 460 jq=1,2
        in(3+jq)=in(jq)
        p(in(jq)+2,3)=p(in(jq)+2,3)-p(in(jq)+2,4)
  460   p(in(jq)+2,1)=p(in(jq)+2,1)-(3-2*jq)*p(in(jq)+2,4)
        goto 330
      endif

c...junction strings: save quantities left after each string.
      if(iabs(kfl(1)).gt.10) goto 280
      i=i-1
      kfjh(iu)=kfl(1)
      do 470 j=1,4
  470 pju(iu+3,j)=pju(iu+3,j)-p(i+1,j)
  480 continue

c...junction strings: put together to new effective string endpoint.
      njs(jt)=i-ista
      kfjs(jt)=k(k(mju(jt+2),3),2)
      kfls=2*int(rlu(0)+3.*parj(4)/(1.+3.*parj(4)))+1
      if(kfjh(1).eq.kfjh(2)) kfls=3
      if(ista.ne.i) kfjs(jt)=isign(1000*max(iabs(kfjh(1)),
     &iabs(kfjh(2)))+100*min(iabs(kfjh(1)),iabs(kfjh(2)))+
     &kfls,kfjh(1))
      do 490 j=1,4
      pjs(jt,j)=pju(1,j)+pju(2,j)+p(mju(jt),j)
  490 pjs(jt+2,j)=pju(4,j)+pju(5,j)
      pjs(jt,5)=sqrt(max(0.,pjs(jt,4)**2-pjs(jt,1)**2-pjs(jt,2)**2-
     &pjs(jt,3)**2))
  500 continue

c...open versus closed strings. choose breakup region for latter.
  510 if(mju(1).ne.0.and.mju(2).ne.0) then
        ns=mju(2)-mju(1)
        nb=mju(1)-n
      elseif(mju(1).ne.0) then
        ns=n+nr-mju(1)
        nb=mju(1)-n
      elseif(mju(2).ne.0) then
        ns=mju(2)-n
        nb=1
      elseif(iabs(k(n+1,2)).ne.21) then
        ns=nr-1
        nb=1
      else
        ns=nr+1
        w2sum=0.
        do 520 is=1,nr
        p(n+nr+is,1)=0.5*four(n+is,n+is+1-nr*(is/nr))
  520   w2sum=w2sum+p(n+nr+is,1)
        w2ran=rlu(0)*w2sum
        nb=0
  530   nb=nb+1
        w2sum=w2sum-p(n+nr+nb,1)
        if(w2sum.gt.w2ran.and.nb.lt.nr) goto 530
      endif

c...find longitudinal string directions (i.e. lightlike four-vectors).
      do 550 is=1,ns
      is1=n+is+nb-1-nr*((is+nb-2)/nr)
      is2=n+is+nb-nr*((is+nb-1)/nr)
      do 540 j=1,5
      dp(1,j)=p(is1,j)
      if(iabs(k(is1,2)).eq.21) dp(1,j)=0.5*dp(1,j)
      if(is1.eq.mju(1)) dp(1,j)=pjs(1,j)-pjs(3,j)
      dp(2,j)=p(is2,j)
      if(iabs(k(is2,2)).eq.21) dp(2,j)=0.5*dp(2,j)
  540 if(is2.eq.mju(2)) dp(2,j)=pjs(2,j)-pjs(4,j)
      dp(3,5)=dfour(1,1)
      dp(4,5)=dfour(2,2)
      dhkc=dfour(1,2)
      if(dp(3,5)+2.*dhkc+dp(4,5).le.0.) then
        dp(3,5)=dp(1,5)**2
        dp(4,5)=dp(2,5)**2
        dp(1,4)=sqrt(dp(1,1)**2+dp(1,2)**2+dp(1,3)**2+dp(1,5)**2)
        dp(2,4)=sqrt(dp(2,1)**2+dp(2,2)**2+dp(2,3)**2+dp(2,5)**2)
        dhkc=dfour(1,2)
      endif
      dhks=sqrt(dhkc**2-dp(3,5)*dp(4,5))
      dhk1=0.5*((dp(4,5)+dhkc)/dhks-1.)
      dhk2=0.5*((dp(3,5)+dhkc)/dhks-1.)
      in1=n+nr+4*is-3
      p(in1,5)=sqrt(dp(3,5)+2.*dhkc+dp(4,5))
      do 550 j=1,4
      p(in1,j)=(1.+dhk1)*dp(1,j)-dhk2*dp(2,j)
  550 p(in1+1,j)=(1.+dhk2)*dp(2,j)-dhk1*dp(1,j)

c...begin initialization: sum up energy, set starting position.
      isav=i
      mstu91=mstu(90)
  560 ntry=ntry+1
      if(ntry.gt.100.and.ntryr.le.4) then
        paru12=4.*paru12
        paru13=2.*paru13
        goto 140
      elseif(ntry.gt.100) then
        call luerrm(14,'(lustrf:) caught in infinite loop')
        if(mstu(21).ge.1) return
      endif
      i=isav
      mstu(90)=mstu91
      do 570 j=1,4
      p(n+nrs,j)=0.
      do 570 is=1,nr
  570 p(n+nrs,j)=p(n+nrs,j)+p(n+is,j)
      do 580 jt=1,2
      irank(jt)=0
      if(mju(jt).ne.0) irank(jt)=njs(jt)
      if(ns.gt.nr) irank(jt)=1
      ie(jt)=k(n+1+(jt/2)*(np-1),3)
      in(3*jt+1)=n+nr+1+4*(jt/2)*(ns-1)
      in(3*jt+2)=in(3*jt+1)+1
      in(3*jt+3)=n+nr+4*ns+2*jt-1
      do 580 in1=n+nr+2+jt,n+nr+4*ns-2+jt,4
      p(in1,1)=2-jt
      p(in1,2)=jt-1
  580 p(in1,3)=1.

c...initialize flavour and pt variables for open string.
      if(ns.lt.nr) then
        px(1)=0.
        py(1)=0.
        if(ns.eq.1.and.mju(1)+mju(2).eq.0) call luptdi(0,px(1),py(1))
        px(2)=-px(1)
        py(2)=-py(1)
        do 590 jt=1,2
        kfl(jt)=k(ie(jt),2)
        if(mju(jt).ne.0) kfl(jt)=kfjs(jt)
        mstj(93)=1
        pmq(jt)=ulmass(kfl(jt))
  590   gam(jt)=0.

c...closed string: random initial breakup flavour, pt and vertex.
      else
        kfl(3)=int(1.+(2.+parj(2))*rlu(0))*(-1)**int(rlu(0)+0.5)
        call lukfdi(kfl(3),0,kfl(1),kdump)
        kfl(2)=-kfl(1)
        if(iabs(kfl(1)).gt.10.and.rlu(0).gt.0.5) then
          kfl(2)=-(kfl(1)+isign(10000,kfl(1)))
        elseif(iabs(kfl(1)).gt.10) then
          kfl(1)=-(kfl(2)+isign(10000,kfl(2)))
        endif
        call luptdi(kfl(1),px(1),py(1))
        px(2)=-px(1)
        py(2)=-py(1)
        pr3=min(25.,0.1*p(n+nr+1,5)**2)
  600   call luzdis(kfl(1),kfl(2),pr3,z)
        zr=pr3/(z*p(n+nr+1,5)**2)
        if(zr.ge.1.) goto 600
        do 610 jt=1,2
        mstj(93)=1
        pmq(jt)=ulmass(kfl(jt))
        gam(jt)=pr3*(1.-z)/z
        in1=n+nr+3+4*(jt/2)*(ns-1)
        p(in1,jt)=1.-z
        p(in1,3-jt)=jt-1
        p(in1,3)=(2-jt)*(1.-z)+(jt-1)*z
        p(in1+1,jt)=zr
        p(in1+1,3-jt)=2-jt
  610   p(in1+1,3)=(2-jt)*(1.-zr)+(jt-1)*zr
      endif

c...find initial transverse directions (i.e. spacelike four-vectors).
      do 650 jt=1,2
      if(jt.eq.1.or.ns.eq.nr-1) then
        in1=in(3*jt+1)
        in3=in(3*jt+3)
        do 620 j=1,4
        dp(1,j)=p(in1,j)
        dp(2,j)=p(in1+1,j)
        dp(3,j)=0.
  620   dp(4,j)=0.
        dp(1,4)=sqrt(dp(1,1)**2+dp(1,2)**2+dp(1,3)**2)
        dp(2,4)=sqrt(dp(2,1)**2+dp(2,2)**2+dp(2,3)**2)
        dp(5,1)=dp(1,1)/dp(1,4)-dp(2,1)/dp(2,4)
        dp(5,2)=dp(1,2)/dp(1,4)-dp(2,2)/dp(2,4)
        dp(5,3)=dp(1,3)/dp(1,4)-dp(2,3)/dp(2,4)
        if(dp(5,1)**2.le.dp(5,2)**2+dp(5,3)**2) dp(3,1)=1.
        if(dp(5,1)**2.gt.dp(5,2)**2+dp(5,3)**2) dp(3,3)=1.
        if(dp(5,2)**2.le.dp(5,1)**2+dp(5,3)**2) dp(4,2)=1.
        if(dp(5,2)**2.gt.dp(5,1)**2+dp(5,3)**2) dp(4,3)=1.
        dhc12=dfour(1,2)
        dhcx1=dfour(3,1)/dhc12
        dhcx2=dfour(3,2)/dhc12
        dhcxx=1d0/sqrt(1d0+2d0*dhcx1*dhcx2*dhc12)
        dhcy1=dfour(4,1)/dhc12
        dhcy2=dfour(4,2)/dhc12
        dhcyx=dhcxx*(dhcx1*dhcy2+dhcx2*dhcy1)*dhc12
        dhcyy=1d0/sqrt(1d0+2d0*dhcy1*dhcy2*dhc12-dhcyx**2)
        do 630 j=1,4
        dp(3,j)=dhcxx*(dp(3,j)-dhcx2*dp(1,j)-dhcx1*dp(2,j))
        p(in3,j)=dp(3,j)
  630   p(in3+1,j)=dhcyy*(dp(4,j)-dhcy2*dp(1,j)-dhcy1*dp(2,j)-
     &  dhcyx*dp(3,j))
      else
        do 640 j=1,4
        p(in3+2,j)=p(in3,j)
  640   p(in3+3,j)=p(in3+1,j)
      endif
  650 continue

c...remove energy used up in junction string fragmentation.
      if(mju(1)+mju(2).gt.0) then
        do 670 jt=1,2
        if(njs(jt).eq.0) goto 670
        do 660 j=1,4
  660   p(n+nrs,j)=p(n+nrs,j)-pjs(jt+2,j)
  670   continue
      endif

c...produce new particle: side, origin.
  680 i=i+1
      if(2*i-nsav.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lustrf:) no more memory left in lujets')
        if(mstu(21).ge.1) return
      endif
      jt=1.5+rlu(0)
      if(iabs(kfl(3-jt)).gt.10) jt=3-jt
      if(iabs(kfl(3-jt)).ge.4.and.iabs(kfl(3-jt)).le.8) jt=3-jt
      jr=3-jt
      js=3-2*jt
      irank(jt)=irank(jt)+1
      k(i,1)=1
      k(i,3)=ie(jt)
      k(i,4)=0
      k(i,5)=0

c...generate flavour, hadron and pt.
  690 call lukfdi(kfl(jt),0,kfl(3),k(i,2))
      if(k(i,2).eq.0) goto 560
      if(mstj(12).ge.3.and.irank(jt).eq.1.and.iabs(kfl(jt)).le.10.and.
     &iabs(kfl(3)).gt.10) then
        if(rlu(0).gt.parj(19)) goto 690
      endif
      p(i,5)=ulmass(k(i,2))
      call luptdi(kfl(jt),px(3),py(3))
      pr(jt)=p(i,5)**2+(px(jt)+px(3))**2+(py(jt)+py(3))**2

c...final hadrons for small invariant mass.
      mstj(93)=1
      pmq(3)=ulmass(kfl(3))
      parjst=parj(33)
      if(mstj(11).eq.2) parjst=parj(34)
      wmin=parjst+pmq(1)+pmq(2)+parj(36)*pmq(3)
      if(iabs(kfl(jt)).gt.10.and.iabs(kfl(3)).gt.10) wmin=
     &wmin-0.5*parj(36)*pmq(3)
      wrem2=four(n+nrs,n+nrs)
      if(wrem2.lt.0.10) goto 560
      if(wrem2.lt.max(wmin*(1.+(2.*rlu(0)-1.)*parj(37)),
     &parj(32)+pmq(1)+pmq(2))**2) goto 820

c...choose z, which gives gamma. shift z for heavy flavours.
      call luzdis(kfl(jt),kfl(3),pr(jt),z)
      if(iabs(kfl(jt)).ge.4.and.iabs(kfl(jt)).le.8.and.
     &mstu(90).lt.8) then
        mstu(90)=mstu(90)+1
        mstu(90+mstu(90))=i
        paru(90+mstu(90))=z
      endif
      kfl1a=iabs(kfl(1))
      kfl2a=iabs(kfl(2))
      if(max(mod(kfl1a,10),mod(kfl1a/1000,10),mod(kfl2a,10),
     &mod(kfl2a/1000,10)).ge.4) then
        pr(jr)=(pmq(jr)+pmq(3))**2+(px(jr)-px(3))**2+(py(jr)-py(3))**2
        pw12=sqrt(max(0.,(wrem2-pr(1)-pr(2))**2-4.*pr(1)*pr(2)))
        z=(wrem2+pr(jt)-pr(jr)+pw12*(2.*z-1.))/(2.*wrem2)
        pr(jr)=(pmq(jr)+parjst)**2+(px(jr)-px(3))**2+(py(jr)-py(3))**2
        if((1.-z)*(wrem2-pr(jt)/z).lt.pr(jr)) goto 820
      endif
      gam(3)=(1.-z)*(gam(jt)+pr(jt)/z)
      do 700 j=1,3
  700 in(j)=in(3*jt+j)

c...stepping within or from 'low' string region easy.
      if(in(1)+1.eq.in(2).and.z*p(in(1)+2,3)*p(in(2)+2,3)*
     &p(in(1),5)**2.ge.pr(jt)) then
        p(in(jt)+2,4)=z*p(in(jt)+2,3)
        p(in(jr)+2,4)=pr(jt)/(p(in(jt)+2,4)*p(in(1),5)**2)
        do 710 j=1,4
  710   p(i,j)=(px(jt)+px(3))*p(in(3),j)+(py(jt)+py(3))*p(in(3)+1,j)
        goto 780
      elseif(in(1)+1.eq.in(2)) then
        p(in(jr)+2,4)=p(in(jr)+2,3)
        p(in(jr)+2,jt)=1.
        in(jr)=in(jr)+4*js
        if(js*in(jr).gt.js*in(4*jr)) goto 560
        if(four(in(1),in(2)).le.1e-2) then
          p(in(jt)+2,4)=p(in(jt)+2,3)
          p(in(jt)+2,jt)=0.
          in(jt)=in(jt)+4*js
        endif
      endif

c...find new transverse directions (i.e. spacelike string vectors).
  720 if(js*in(1).gt.js*in(3*jr+1).or.js*in(2).gt.js*in(3*jr+2).or.
     &in(1).gt.in(2)) goto 560
      if(in(1).ne.in(3*jt+1).or.in(2).ne.in(3*jt+2)) then
        do 730 j=1,4
        dp(1,j)=p(in(1),j)
        dp(2,j)=p(in(2),j)
        dp(3,j)=0.
  730   dp(4,j)=0.
        dp(1,4)=sqrt(dp(1,1)**2+dp(1,2)**2+dp(1,3)**2)
        dp(2,4)=sqrt(dp(2,1)**2+dp(2,2)**2+dp(2,3)**2)
        dhc12=dfour(1,2)
        if(dhc12.le.1e-2) then
          p(in(jt)+2,4)=p(in(jt)+2,3)
          p(in(jt)+2,jt)=0.
          in(jt)=in(jt)+4*js
          goto 720
        endif
        in(3)=n+nr+4*ns+5
        dp(5,1)=dp(1,1)/dp(1,4)-dp(2,1)/dp(2,4)
        dp(5,2)=dp(1,2)/dp(1,4)-dp(2,2)/dp(2,4)
        dp(5,3)=dp(1,3)/dp(1,4)-dp(2,3)/dp(2,4)
        if(dp(5,1)**2.le.dp(5,2)**2+dp(5,3)**2) dp(3,1)=1.
        if(dp(5,1)**2.gt.dp(5,2)**2+dp(5,3)**2) dp(3,3)=1.
        if(dp(5,2)**2.le.dp(5,1)**2+dp(5,3)**2) dp(4,2)=1.
        if(dp(5,2)**2.gt.dp(5,1)**2+dp(5,3)**2) dp(4,3)=1.
        dhcx1=dfour(3,1)/dhc12
        dhcx2=dfour(3,2)/dhc12
        dhcxx=1d0/sqrt(1d0+2d0*dhcx1*dhcx2*dhc12)
        dhcy1=dfour(4,1)/dhc12
        dhcy2=dfour(4,2)/dhc12
        dhcyx=dhcxx*(dhcx1*dhcy2+dhcx2*dhcy1)*dhc12
        dhcyy=1d0/sqrt(1d0+2d0*dhcy1*dhcy2*dhc12-dhcyx**2)
        do 740 j=1,4
        dp(3,j)=dhcxx*(dp(3,j)-dhcx2*dp(1,j)-dhcx1*dp(2,j))
        p(in(3),j)=dp(3,j)
  740   p(in(3)+1,j)=dhcyy*(dp(4,j)-dhcy2*dp(1,j)-dhcy1*dp(2,j)-
     &  dhcyx*dp(3,j))
c...express pt with respect to new axes, if sensible.
        pxp=-(px(3)*four(in(3*jt+3),in(3))+py(3)*
     &  four(in(3*jt+3)+1,in(3)))
        pyp=-(px(3)*four(in(3*jt+3),in(3)+1)+py(3)*
     &  four(in(3*jt+3)+1,in(3)+1))
        if(abs(pxp**2+pyp**2-px(3)**2-py(3)**2).lt.0.01) then
          px(3)=pxp
          py(3)=pyp
        endif
      endif

c...sum up known four-momentum. gives coefficients for m2 expression.
      do 760 j=1,4
      dhg(j)=0.
      p(i,j)=px(jt)*p(in(3*jt+3),j)+py(jt)*p(in(3*jt+3)+1,j)+
     &px(3)*p(in(3),j)+py(3)*p(in(3)+1,j)
      do 750 in1=in(3*jt+1),in(1)-4*js,4*js
  750 p(i,j)=p(i,j)+p(in1+2,3)*p(in1,j)
      do 760 in2=in(3*jt+2),in(2)-4*js,4*js
  760 p(i,j)=p(i,j)+p(in2+2,3)*p(in2,j)
      dhm(1)=four(i,i)
      dhm(2)=2.*four(i,in(1))
      dhm(3)=2.*four(i,in(2))
      dhm(4)=2.*four(in(1),in(2))

c...find coefficients for gamma expression.
      do 770 in2=in(1)+1,in(2),4
      do 770 in1=in(1),in2-1,4
      dhc=2.*four(in1,in2)
      dhg(1)=dhg(1)+p(in1+2,jt)*p(in2+2,jt)*dhc
      if(in1.eq.in(1)) dhg(2)=dhg(2)-js*p(in2+2,jt)*dhc
      if(in2.eq.in(2)) dhg(3)=dhg(3)+js*p(in1+2,jt)*dhc
  770 if(in1.eq.in(1).and.in2.eq.in(2)) dhg(4)=dhg(4)-dhc

c...solve (m2, gamma) equation system for energies taken.
      dhs1=dhm(jr+1)*dhg(4)-dhm(4)*dhg(jr+1)
      if(abs(dhs1).lt.1e-4) goto 560
      dhs2=dhm(4)*(gam(3)-dhg(1))-dhm(jt+1)*dhg(jr+1)-dhg(4)*
     &(p(i,5)**2-dhm(1))+dhg(jt+1)*dhm(jr+1)
      dhs3=dhm(jt+1)*(gam(3)-dhg(1))-dhg(jt+1)*(p(i,5)**2-dhm(1))
      p(in(jr)+2,4)=0.5*(sqrt(max(0d0,dhs2**2-4.*dhs1*dhs3))/abs(dhs1)-
     &dhs2/dhs1)
      if(dhm(jt+1)+dhm(4)*p(in(jr)+2,4).le.0.) goto 560
      p(in(jt)+2,4)=(p(i,5)**2-dhm(1)-dhm(jr+1)*p(in(jr)+2,4))/
     &(dhm(jt+1)+dhm(4)*p(in(jr)+2,4))

c...step to new region if necessary.
      if(p(in(jr)+2,4).gt.p(in(jr)+2,3)) then
        p(in(jr)+2,4)=p(in(jr)+2,3)
        p(in(jr)+2,jt)=1.
        in(jr)=in(jr)+4*js
        if(js*in(jr).gt.js*in(4*jr)) goto 560
        if(four(in(1),in(2)).le.1e-2) then
          p(in(jt)+2,4)=p(in(jt)+2,3)
          p(in(jt)+2,jt)=0.
          in(jt)=in(jt)+4*js
        endif
        goto 720
      elseif(p(in(jt)+2,4).gt.p(in(jt)+2,3)) then
        p(in(jt)+2,4)=p(in(jt)+2,3)
        p(in(jt)+2,jt)=0.
        in(jt)=in(jt)+4*js
        goto 720
      endif

c...four-momentum of particle. remaining quantities. loop back.
  780 do 790 j=1,4
      p(i,j)=p(i,j)+p(in(1)+2,4)*p(in(1),j)+p(in(2)+2,4)*p(in(2),j)
  790 p(n+nrs,j)=p(n+nrs,j)-p(i,j)
      if(p(i,4).lt.p(i,5)) goto 560
      kfl(jt)=-kfl(3)
      pmq(jt)=pmq(3)
      px(jt)=-px(3)
      py(jt)=-py(3)
      gam(jt)=gam(3)
      if(in(3).ne.in(3*jt+3)) then
        do 800 j=1,4
        p(in(3*jt+3),j)=p(in(3),j)
  800   p(in(3*jt+3)+1,j)=p(in(3)+1,j)
      endif
      do 810 jq=1,2
      in(3*jt+jq)=in(jq)
      p(in(jq)+2,3)=p(in(jq)+2,3)-p(in(jq)+2,4)
  810 p(in(jq)+2,jt)=p(in(jq)+2,jt)-js*(3-2*jq)*p(in(jq)+2,4)
      goto 680

c...final hadron: side, flavour, hadron, mass.
  820 i=i+1
      k(i,1)=1
      k(i,3)=ie(jr)
      k(i,4)=0
      k(i,5)=0
      call lukfdi(kfl(jr),-kfl(3),kfldmp,k(i,2))
      if(k(i,2).eq.0) goto 560
      p(i,5)=ulmass(k(i,2))
      pr(jr)=p(i,5)**2+(px(jr)-px(3))**2+(py(jr)-py(3))**2

c...final two hadrons: find common setup of four-vectors.
      jq=1
      if(p(in(4)+2,3)*p(in(5)+2,3)*four(in(4),in(5)).lt.p(in(7),3)*
     &p(in(8),3)*four(in(7),in(8))) jq=2
      dhc12=four(in(3*jq+1),in(3*jq+2))
      dhr1=four(n+nrs,in(3*jq+2))/dhc12
      dhr2=four(n+nrs,in(3*jq+1))/dhc12
      if(in(4).ne.in(7).or.in(5).ne.in(8)) then
        px(3-jq)=-four(n+nrs,in(3*jq+3))-px(jq)
        py(3-jq)=-four(n+nrs,in(3*jq+3)+1)-py(jq)
        pr(3-jq)=p(i+(jt+jq-3)**2-1,5)**2+(px(3-jq)+(2*jq-3)*js*
     &  px(3))**2+(py(3-jq)+(2*jq-3)*js*py(3))**2
      endif

c...solve kinematics for final two hadrons, if possible.
      wrem2=wrem2+(px(1)+px(2))**2+(py(1)+py(2))**2
      fd=(sqrt(pr(1))+sqrt(pr(2)))/sqrt(wrem2)
      if(mju(1)+mju(2).ne.0.and.i.eq.isav+2.and.fd.ge.1.) goto 190
      if(fd.ge.1.) goto 560
      fa=wrem2+pr(jt)-pr(jr)
      if(mstj(11).ne.2) prev=0.5*exp(max(-80.,log(fd)*parj(38)*
     &(pr(1)+pr(2))**2))
      if(mstj(11).eq.2) prev=0.5*fd**parj(39)
      fb=sign(sqrt(max(0.,fa**2-4.*wrem2*pr(jt))),js*(rlu(0)-prev))
      kfl1a=iabs(kfl(1))
      kfl2a=iabs(kfl(2))
      if(max(mod(kfl1a,10),mod(kfl1a/1000,10),mod(kfl2a,10),
     &mod(kfl2a/1000,10)).ge.6) fb=sign(sqrt(max(0.,fa**2-
     &4.*wrem2*pr(jt))),float(js))
      do 830 j=1,4
      p(i-1,j)=(px(jt)+px(3))*p(in(3*jq+3),j)+(py(jt)+py(3))*
     &p(in(3*jq+3)+1,j)+0.5*(dhr1*(fa+fb)*p(in(3*jq+1),j)+
     &dhr2*(fa-fb)*p(in(3*jq+2),j))/wrem2
  830 p(i,j)=p(n+nrs,j)-p(i-1,j)
      if(p(i-1,4).lt.p(i-1,5).or.p(i,4).lt.p(i,5)) goto 560

c...mark jets as fragmented and give daughter pointers.
      n=i-nrs+1
      do 840 i=nsav+1,nsav+np
      im=k(i,3)
      k(im,1)=k(im,1)+10
      if(mstu(16).ne.2) then
        k(im,4)=nsav+1
        k(im,5)=nsav+1
      else
        k(im,4)=nsav+2
        k(im,5)=n
      endif
  840 continue

c...document string system. move up particles.
      nsav=nsav+1
      k(nsav,1)=11
      k(nsav,2)=92
      k(nsav,3)=ip
      k(nsav,4)=nsav+1
      k(nsav,5)=n
      do 850 j=1,4
      p(nsav,j)=dps(j)
  850 v(nsav,j)=v(ip,j)
      p(nsav,5)=sqrt(max(0d0,dps(4)**2-dps(1)**2-dps(2)**2-dps(3)**2))
      v(nsav,5)=0.
      do 860 i=nsav+1,n
      do 860 j=1,5
      k(i,j)=k(i+nrs-1,j)
      p(i,j)=p(i+nrs-1,j)
  860 v(i,j)=0.
      mstu91=mstu(90)
      do 870 iz=mstu90+1,mstu91
      mstu9t(iz)=mstu(90+iz)-nrs+1-nsav+n
  870 paru9t(iz)=paru(90+iz)
      mstu(90)=mstu90

c...order particles in rank along the chain. update mother pointer.
      do 880 i=nsav+1,n
      do 880 j=1,5
      k(i-nsav+n,j)=k(i,j)
  880 p(i-nsav+n,j)=p(i,j)
      i1=nsav
      do 910 i=n+1,2*n-nsav
      if(k(i,3).ne.ie(1)) goto 910
      i1=i1+1
      do 890 j=1,5
      k(i1,j)=k(i,j)
  890 p(i1,j)=p(i,j)
      if(mstu(16).ne.2) k(i1,3)=nsav
      do 900 iz=mstu90+1,mstu91
      if(mstu9t(iz).eq.i) then
        mstu(90)=mstu(90)+1
        mstu(90+mstu(90))=i1
        paru(90+mstu(90))=paru9t(iz)
      endif
  900 continue
  910 continue
      do 940 i=2*n-nsav,n+1,-1
      if(k(i,3).eq.ie(1)) goto 940
      i1=i1+1
      do 920 j=1,5
      k(i1,j)=k(i,j)
  920 p(i1,j)=p(i,j)
      if(mstu(16).ne.2) k(i1,3)=nsav
      do 930 iz=mstu90+1,mstu91
      if(mstu9t(iz).eq.i) then
        mstu(90)=mstu(90)+1
        mstu(90+mstu(90))=i1
        paru(90+mstu(90))=paru9t(iz)
      endif
  930 continue
  940 continue

c...boost back particle system. set production vertices.
      if(mbst.eq.0) then
        mstu(33)=1
        call ludbrb(nsav+1,n,0.,0.,dps(1)/dps(4),dps(2)/dps(4),
     &  dps(3)/dps(4))
      else
        do 950 i=nsav+1,n
        hhpmt=p(i,1)**2+p(i,2)**2+p(i,5)**2
        if(p(i,3).gt.0.) then
          hhpez=(p(i,4)+p(i,3))*hhbz
          p(i,3)=0.5*(hhpez-hhpmt/hhpez)
          p(i,4)=0.5*(hhpez+hhpmt/hhpez)
        else
          hhpez=(p(i,4)-p(i,3))/hhbz
          p(i,3)=-0.5*(hhpez-hhpmt/hhpez)
          p(i,4)=0.5*(hhpez+hhpmt/hhpez)
        endif
  950   continue
      endif
      do 960 i=nsav+1,n
      do 960 j=1,4
  960 v(i,j)=v(ip,j)

      return
      end

c*********************************************************************

      subroutine luindf(ip)

c...purpose: to handle the fragmentation of a jet system (or a single
c...jet) according to independent fragmentation models.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension dps(5),psi(4),nfi(3),nfl(3),ifet(3),kflf(3),
     &kflo(2),pxo(2),pyo(2),wo(2)

c...reset counters. identify parton system and take copy. check flavour.
      nsav=n
      mstu90=mstu(90)
      njet=0
      kqsum=0
      do 100 j=1,5
  100 dps(j)=0.
      i=ip-1
  110 i=i+1
      if(i.gt.min(n,mstu(4)-mstu(32))) then
        call luerrm(12,'(luindf:) failed to reconstruct jet system')
        if(mstu(21).ge.1) return
      endif
      if(k(i,1).ne.1.and.k(i,1).ne.2) goto 110
      kc=lucomp(k(i,2))
      if(kc.eq.0) goto 110
      kq=kchg(kc,2)*isign(1,k(i,2))
      if(kq.eq.0) goto 110
      njet=njet+1
      if(kq.ne.2) kqsum=kqsum+kq
      do 120 j=1,5
      k(nsav+njet,j)=k(i,j)
      p(nsav+njet,j)=p(i,j)
  120 dps(j)=dps(j)+p(i,j)
      k(nsav+njet,3)=i
      if(k(i,1).eq.2.or.(mstj(3).le.5.and.n.gt.i.and.
     &k(i+1,1).eq.2)) goto 110
      if(njet.ne.1.and.kqsum.ne.0) then
        call luerrm(12,'(luindf:) unphysical flavour combination')
        if(mstu(21).ge.1) return
      endif

c...boost copied system to cm frame. find cm energy and sum flavours.
      if(njet.ne.1) then
        mstu(33)=1
        call ludbrb(nsav+1,nsav+njet,0.,0.,-dps(1)/dps(4),
     &  -dps(2)/dps(4),-dps(3)/dps(4))
      endif
      pecm=0.
      do 130 j=1,3
  130 nfi(j)=0
      do 140 i=nsav+1,nsav+njet
      pecm=pecm+p(i,4)
      kfa=iabs(k(i,2))
      if(kfa.le.3) then
        nfi(kfa)=nfi(kfa)+isign(1,k(i,2))
      elseif(kfa.gt.1000) then
        kfla=mod(kfa/1000,10)
        kflb=mod(kfa/100,10)
        if(kfla.le.3) nfi(kfla)=nfi(kfla)+isign(1,k(i,2))
        if(kflb.le.3) nfi(kflb)=nfi(kflb)+isign(1,k(i,2))
      endif
  140 continue

c...loop over attempts made. reset counters.
      ntry=0
  150 ntry=ntry+1
      if(ntry.gt.200) then
        call luerrm(14,'(luindf:) caught in infinite loop')
        if(mstu(21).ge.1) return
      endif
      n=nsav+njet
      mstu(90)=mstu90
      do 160 j=1,3
      nfl(j)=nfi(j)
      ifet(j)=0
  160 kflf(j)=0

c...loop over jets to be fragmented.
      do 230 ip1=nsav+1,nsav+njet
      mstj(91)=0
      nsav1=n
      mstu91=mstu(90)

c...initial flavour and momentum values. jet along +z axis.
      kflh=iabs(k(ip1,2))
      if(kflh.gt.10) kflh=mod(kflh/1000,10)
      kflo(2)=0
      wf=p(ip1,4)+sqrt(p(ip1,1)**2+p(ip1,2)**2+p(ip1,3)**2)

c...initial values for quark or diquark jet.
  170 if(iabs(k(ip1,2)).ne.21) then
        nstr=1
        kflo(1)=k(ip1,2)
        call luptdi(0,pxo(1),pyo(1))
        wo(1)=wf

c...initial values for gluon treated like random quark jet.
      elseif(mstj(2).le.2) then
        nstr=1
        if(mstj(2).eq.2) mstj(91)=1
        kflo(1)=int(1.+(2.+parj(2))*rlu(0))*(-1)**int(rlu(0)+0.5)
        call luptdi(0,pxo(1),pyo(1))
        wo(1)=wf

c...initial values for gluon treated like quark-antiquark jet pair,
c...sharing energy according to altarelli-parisi splitting function.
      else
        nstr=2
        if(mstj(2).eq.4) mstj(91)=1
        kflo(1)=int(1.+(2.+parj(2))*rlu(0))*(-1)**int(rlu(0)+0.5)
        kflo(2)=-kflo(1)
        call luptdi(0,pxo(1),pyo(1))
        pxo(2)=-pxo(1)
        pyo(2)=-pyo(1)
        wo(1)=wf*rlu(0)**(1./3.)
        wo(2)=wf-wo(1)
      endif

c...initial values for rank, flavour, pt and w+.
      do 220 istr=1,nstr
  180 i=n
      mstu(90)=mstu91
      irank=0
      kfl1=kflo(istr)
      px1=pxo(istr)
      py1=pyo(istr)
      w=wo(istr)

c...new hadron. generate flavour and hadron species.
  190 i=i+1
      if(i.ge.mstu(4)-mstu(32)-njet-5) then
        call luerrm(11,'(luindf:) no more memory left in lujets')
        if(mstu(21).ge.1) return
      endif
      irank=irank+1
      k(i,1)=1
      k(i,3)=ip1
      k(i,4)=0
      k(i,5)=0
  200 call lukfdi(kfl1,0,kfl2,k(i,2))
      if(k(i,2).eq.0) goto 180
      if(mstj(12).ge.3.and.irank.eq.1.and.iabs(kfl1).le.10.and.
     &iabs(kfl2).gt.10) then
        if(rlu(0).gt.parj(19)) goto 200
      endif

c...find hadron mass. generate four-momentum.
      p(i,5)=ulmass(k(i,2))
      call luptdi(kfl1,px2,py2)
      p(i,1)=px1+px2
      p(i,2)=py1+py2
      pr=p(i,5)**2+p(i,1)**2+p(i,2)**2
      call luzdis(kfl1,kfl2,pr,z)
      mzsav=0
      if(iabs(kfl1).ge.4.and.iabs(kfl1).le.8.and.mstu(90).lt.8) then
        mzsav=1
        mstu(90)=mstu(90)+1
        mstu(90+mstu(90))=i
        paru(90+mstu(90))=z
      endif
      p(i,3)=0.5*(z*w-pr/(z*w))
      p(i,4)=0.5*(z*w+pr/(z*w))
      if(mstj(3).ge.1.and.irank.eq.1.and.kflh.ge.4.and.
     &p(i,3).le.0.001) then
        if(w.ge.p(i,5)+0.5*parj(32)) goto 180
        p(i,3)=0.0001
        p(i,4)=sqrt(pr)
        z=p(i,4)/w
      endif

c...remaining flavour and momentum.
      kfl1=-kfl2
      px1=-px2
      py1=-py2
      w=(1.-z)*w
      do 210 j=1,5
  210 v(i,j)=0.

c...check if pl acceptable. go back for new hadron if enough energy.
      if(mstj(3).ge.0.and.p(i,3).lt.0.) then
        i=i-1
        if(mzsav.eq.1) mstu(90)=mstu(90)-1
      endif
      if(w.gt.parj(31)) goto 190
  220 n=i
      if(mod(mstj(3),5).eq.4.and.n.eq.nsav1) wf=wf+0.1*parj(32)
      if(mod(mstj(3),5).eq.4.and.n.eq.nsav1) goto 170

c...rotate jet to new direction.
      the=ulangl(p(ip1,3),sqrt(p(ip1,1)**2+p(ip1,2)**2))
      phi=ulangl(p(ip1,1),p(ip1,2))
      mstu(33)=1
      call ludbrb(nsav1+1,n,the,phi,0d0,0d0,0d0)
      k(k(ip1,3),4)=nsav1+1
      k(k(ip1,3),5)=n

c...end of jet generation loop. skip conservation in some cases.
  230 continue
      if(njet.eq.1.or.mstj(3).le.0) goto 470
      if(mod(mstj(3),5).ne.0.and.n-nsav-njet.lt.2) goto 150

c...subtract off produced hadron flavours, finished if zero.
      do 240 i=nsav+njet+1,n
      kfa=iabs(k(i,2))
      kfla=mod(kfa/1000,10)
      kflb=mod(kfa/100,10)
      kflc=mod(kfa/10,10)
      if(kfla.eq.0) then
        if(kflb.le.3) nfl(kflb)=nfl(kflb)-isign(1,k(i,2))*(-1)**kflb
        if(kflc.le.3) nfl(kflc)=nfl(kflc)+isign(1,k(i,2))*(-1)**kflb
      else
        if(kfla.le.3) nfl(kfla)=nfl(kfla)-isign(1,k(i,2))
        if(kflb.le.3) nfl(kflb)=nfl(kflb)-isign(1,k(i,2))
        if(kflc.le.3) nfl(kflc)=nfl(kflc)-isign(1,k(i,2))
      endif
  240 continue
      nreq=(iabs(nfl(1))+iabs(nfl(2))+iabs(nfl(3))-iabs(nfl(1)+
     &nfl(2)+nfl(3)))/2+iabs(nfl(1)+nfl(2)+nfl(3))/3
      if(nreq.eq.0) goto 320

c...take away flavour of low-momentum particles until enough freedom.
      nrem=0
  250 irem=0
      p2min=pecm**2
      do 260 i=nsav+njet+1,n
      p2=p(i,1)**2+p(i,2)**2+p(i,3)**2
      if(k(i,1).eq.1.and.p2.lt.p2min) irem=i
  260 if(k(i,1).eq.1.and.p2.lt.p2min) p2min=p2
      if(irem.eq.0) goto 150
      k(irem,1)=7
      kfa=iabs(k(irem,2))
      kfla=mod(kfa/1000,10)
      kflb=mod(kfa/100,10)
      kflc=mod(kfa/10,10)
      if(kfla.ge.4.or.kflb.ge.4) k(irem,1)=8
      if(k(irem,1).eq.8) goto 250
      if(kfla.eq.0) then
        isgn=isign(1,k(irem,2))*(-1)**kflb
        if(kflb.le.3) nfl(kflb)=nfl(kflb)+isgn
        if(kflc.le.3) nfl(kflc)=nfl(kflc)-isgn
      else
        if(kfla.le.3) nfl(kfla)=nfl(kfla)+isign(1,k(irem,2))
        if(kflb.le.3) nfl(kflb)=nfl(kflb)+isign(1,k(irem,2))
        if(kflc.le.3) nfl(kflc)=nfl(kflc)+isign(1,k(irem,2))
      endif
      nrem=nrem+1
      nreq=(iabs(nfl(1))+iabs(nfl(2))+iabs(nfl(3))-iabs(nfl(1)+
     &nfl(2)+nfl(3)))/2+iabs(nfl(1)+nfl(2)+nfl(3))/3
      if(nreq.gt.nrem) goto 250
      do 270 i=nsav+njet+1,n
  270 if(k(i,1).eq.8) k(i,1)=1

c...find combination of existing and new flavours for hadron.
  280 nfet=2
      if(nfl(1)+nfl(2)+nfl(3).ne.0) nfet=3
      if(nreq.lt.nrem) nfet=1
      if(iabs(nfl(1))+iabs(nfl(2))+iabs(nfl(3)).eq.0) nfet=0
      do 290 j=1,nfet
      ifet(j)=1+(iabs(nfl(1))+iabs(nfl(2))+iabs(nfl(3)))*rlu(0)
      kflf(j)=isign(1,nfl(1))
      if(ifet(j).gt.iabs(nfl(1))) kflf(j)=isign(2,nfl(2))
  290 if(ifet(j).gt.iabs(nfl(1))+iabs(nfl(2))) kflf(j)=isign(3,nfl(3))
      if(nfet.eq.2.and.(ifet(1).eq.ifet(2).or.kflf(1)*kflf(2).gt.0))
     &goto 280
      if(nfet.eq.3.and.(ifet(1).eq.ifet(2).or.ifet(1).eq.ifet(3).or.
     &ifet(2).eq.ifet(3).or.kflf(1)*kflf(2).lt.0.or.kflf(1)*kflf(3).
     &lt.0.or.kflf(1)*(nfl(1)+nfl(2)+nfl(3)).lt.0)) goto 280
      if(nfet.eq.0) kflf(1)=1+int((2.+parj(2))*rlu(0))
      if(nfet.eq.0) kflf(2)=-kflf(1)
      if(nfet.eq.1) kflf(2)=isign(1+int((2.+parj(2))*rlu(0)),-kflf(1))
      if(nfet.le.2) kflf(3)=0
      if(kflf(3).ne.0) then
        kflfc=isign(1000*max(iabs(kflf(1)),iabs(kflf(3)))+
     &  100*min(iabs(kflf(1)),iabs(kflf(3)))+1,kflf(1))
        if(kflf(1).eq.kflf(3).or.(1.+3.*parj(4))*rlu(0).gt.1.)
     &  kflfc=kflfc+isign(2,kflfc)
      else
        kflfc=kflf(1)
      endif
      call lukfdi(kflfc,kflf(2),kfldmp,kf)
      if(kf.eq.0) goto 280
      do 300 j=1,max(2,nfet)
  300 nfl(iabs(kflf(j)))=nfl(iabs(kflf(j)))-isign(1,kflf(j))

c...store hadron at random among free positions.
      npos=min(1+int(rlu(0)*nrem),nrem)
      do 310 i=nsav+njet+1,n
      if(k(i,1).eq.7) npos=npos-1
      if(k(i,1).eq.1.or.npos.ne.0) goto 310
      k(i,1)=1
      k(i,2)=kf
      p(i,5)=ulmass(k(i,2))
      p(i,4)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2+p(i,5)**2)
  310 continue
      nrem=nrem-1
      nreq=(iabs(nfl(1))+iabs(nfl(2))+iabs(nfl(3))-iabs(nfl(1)+
     &nfl(2)+nfl(3)))/2+iabs(nfl(1)+nfl(2)+nfl(3))/3
      if(nrem.gt.0) goto 280

c...compensate for missing momentum in global scheme (3 options).
  320 if(mod(mstj(3),5).ne.0.and.mod(mstj(3),5).ne.4) then
        do 330 j=1,3
        psi(j)=0.
        do 330 i=nsav+njet+1,n
  330   psi(j)=psi(j)+p(i,j)
        psi(4)=psi(1)**2+psi(2)**2+psi(3)**2
        pws=0.
        do 340 i=nsav+njet+1,n
        if(mod(mstj(3),5).eq.1) pws=pws+p(i,4)
        if(mod(mstj(3),5).eq.2) pws=pws+sqrt(p(i,5)**2+(psi(1)*p(i,1)+
     &  psi(2)*p(i,2)+psi(3)*p(i,3))**2/psi(4))
  340   if(mod(mstj(3),5).eq.3) pws=pws+1.
        do 360 i=nsav+njet+1,n
        if(mod(mstj(3),5).eq.1) pw=p(i,4)
        if(mod(mstj(3),5).eq.2) pw=sqrt(p(i,5)**2+(psi(1)*p(i,1)+
     &  psi(2)*p(i,2)+psi(3)*p(i,3))**2/psi(4))
        if(mod(mstj(3),5).eq.3) pw=1.
        do 350 j=1,3
  350   p(i,j)=p(i,j)-psi(j)*pw/pws
  360   p(i,4)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2+p(i,5)**2)

c...compensate for missing momentum withing each jet separately.
      elseif(mod(mstj(3),5).eq.4) then
        do 370 i=n+1,n+njet
        k(i,1)=0
        do 370 j=1,5
  370   p(i,j)=0.
        do 390 i=nsav+njet+1,n
        ir1=k(i,3)
        ir2=n+ir1-nsav
        k(ir2,1)=k(ir2,1)+1
        pls=(p(i,1)*p(ir1,1)+p(i,2)*p(ir1,2)+p(i,3)*p(ir1,3))/
     &  (p(ir1,1)**2+p(ir1,2)**2+p(ir1,3)**2)
        do 380 j=1,3
  380   p(ir2,j)=p(ir2,j)+p(i,j)-pls*p(ir1,j)
        p(ir2,4)=p(ir2,4)+p(i,4)
  390   p(ir2,5)=p(ir2,5)+pls
        pss=0.
        do 400 i=n+1,n+njet
  400   if(k(i,1).ne.0) pss=pss+p(i,4)/(pecm*(0.8*p(i,5)+0.2))
        do 420 i=nsav+njet+1,n
        ir1=k(i,3)
        ir2=n+ir1-nsav
        pls=(p(i,1)*p(ir1,1)+p(i,2)*p(ir1,2)+p(i,3)*p(ir1,3))/
     &  (p(ir1,1)**2+p(ir1,2)**2+p(ir1,3)**2)
        do 410 j=1,3
  410   p(i,j)=p(i,j)-p(ir2,j)/k(ir2,1)+(1./(p(ir2,5)*pss)-1.)*pls*
     &  p(ir1,j)
  420   p(i,4)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2+p(i,5)**2)
      endif

c...scale momenta for energy conservation.
      if(mod(mstj(3),5).ne.0) then
        pms=0.
        pes=0.
        pqs=0.
        do 430 i=nsav+njet+1,n
        pms=pms+p(i,5)
        pes=pes+p(i,4)
  430   pqs=pqs+p(i,5)**2/p(i,4)
        if(pms.ge.pecm) goto 150
        neco=0
  440   neco=neco+1
        pfac=(pecm-pqs)/(pes-pqs)
        pes=0.
        pqs=0.
        do 460 i=nsav+njet+1,n
        do 450 j=1,3
  450   p(i,j)=pfac*p(i,j)
        p(i,4)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2+p(i,5)**2)
        pes=pes+p(i,4)
  460   pqs=pqs+p(i,5)**2/p(i,4)
        if(neco.lt.10.and.abs(pecm-pes).gt.2e-6*pecm) goto 440
      endif

c...origin of produced particles and parton daughter pointers.
  470 do 480 i=nsav+njet+1,n
      if(mstu(16).ne.2) k(i,3)=nsav+1
  480 if(mstu(16).eq.2) k(i,3)=k(k(i,3),3)
      do 490 i=nsav+1,nsav+njet
      i1=k(i,3)
      k(i1,1)=k(i1,1)+10
      if(mstu(16).ne.2) then
        k(i1,4)=nsav+1
        k(i1,5)=nsav+1
      else
        k(i1,4)=k(i1,4)-njet+1
        k(i1,5)=k(i1,5)-njet+1
        if(k(i1,5).lt.k(i1,4)) then
          k(i1,4)=0
          k(i1,5)=0
        endif
      endif
  490 continue

c...document independent fragmentation system. remove copy of jets.
      nsav=nsav+1
      k(nsav,1)=11
      k(nsav,2)=93
      k(nsav,3)=ip
      k(nsav,4)=nsav+1
      k(nsav,5)=n-njet+1
      do 500 j=1,4
      p(nsav,j)=dps(j)
  500 v(nsav,j)=v(ip,j)
      p(nsav,5)=sqrt(max(0d0,dps(4)**2-dps(1)**2-dps(2)**2-dps(3)**2))
      v(nsav,5)=0.
      do 510 i=nsav+njet,n
      do 510 j=1,5
      k(i-njet+1,j)=k(i,j)
      p(i-njet+1,j)=p(i,j)
  510 v(i-njet+1,j)=v(i,j)
      n=n-njet+1
      do 520 iz=mstu90+1,mstu(90)
  520 mstu(90+iz)=mstu(90+iz)-njet+1

c...boost back particle system. set production vertices.
      if(njet.ne.1) call ludbrb(nsav+1,n,0.,0.,dps(1)/dps(4),
     &dps(2)/dps(4),dps(3)/dps(4))
      do 530 i=nsav+1,n
      do 530 j=1,4
  530 v(i,j)=v(ip,j)

      return
      end

c*********************************************************************

      subroutine ludecy(ip)

c...purpose: to handle the decay of unstable particles.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      save /lujets/,/ludat1/,/ludat2/,/ludat3/
      dimension vdcy(4),kflo(4),kfl1(4),pv(10,5),rord(10),ue(3),be(3),
     &wtcor(10)
      data wtcor/2.,5.,15.,60.,250.,1500.,1.2e4,1.2e5,150.,16./

c...functions: momentum in two-particle decays, four-product and
c...matrix element times phase space in weak decays.
      pawt(a,b,c)=sqrt((a**2-(b+c)**2)*(a**2-(b-c)**2))/(2.*a)
      four(i,j)=p(i,4)*p(j,4)-p(i,1)*p(j,1)-p(i,2)*p(j,2)-p(i,3)*p(j,3)
      hmeps(ha)=((1.-hrq-ha)**2+3.*ha*(1.+hrq-ha))*
     &sqrt((1.-hrq-ha)**2-4.*hrq*ha)

c...initial values.
      ntry=0
      nsav=n
      kfa=iabs(k(ip,2))
      kfs=isign(1,k(ip,2))
      kc=lucomp(kfa)
      mstj(92)=0

c...choose lifetime and determine decay vertex.
      if(k(ip,1).eq.5) then
        v(ip,5)=0.
      elseif(k(ip,1).ne.4) then
        v(ip,5)=-pmas(kc,4)*log(rlu(0))
      endif
      do 100 j=1,4
  100 vdcy(j)=v(ip,j)+v(ip,5)*p(ip,j)/p(ip,5)

c...determine whether decay allowed or not.
      mout=0
      if(mstj(22).eq.2) then
        if(pmas(kc,4).gt.parj(71)) mout=1
      elseif(mstj(22).eq.3) then
        if(vdcy(1)**2+vdcy(2)**2+vdcy(3)**2.gt.parj(72)**2) mout=1
      elseif(mstj(22).eq.4) then
        if(vdcy(1)**2+vdcy(2)**2.gt.parj(73)**2) mout=1
        if(abs(vdcy(3)).gt.parj(74)) mout=1
      endif
      if(mout.eq.1.and.k(ip,1).ne.5) then
        k(ip,1)=4
        return
      endif

c...b-b~ mixing: flip sign of meson appropriately.
      mmix=0
      if((kfa.eq.511.or.kfa.eq.531).and.mstj(26).ge.1) then
        xbbmix=parj(76)
        if(kfa.eq.531) xbbmix=parj(77)
        if(sin(0.5*xbbmix*v(ip,5)/pmas(kc,4))**2.gt.rlu(0)) mmix=1
        if(mmix.eq.1) kfs=-kfs
      endif

c...check existence of decay channels. particle/antiparticle rules.
      kca=kc
      if(mdcy(kc,2).gt.0) then
        mdmdcy=mdme(mdcy(kc,2),2)
        if(mdmdcy.gt.80.and.mdmdcy.le.90) kca=mdmdcy
      endif
      if(mdcy(kca,2).le.0.or.mdcy(kca,3).le.0) then
        call luerrm(9,'(ludecy:) no decay channel defined')
        return
      endif
      if(mod(kfa/1000,10).eq.0.and.(kca.eq.85.or.kca.eq.87)) kfs=-kfs
      if(kchg(kc,3).eq.0) then
        kfsp=1
        kfsn=0
        if(rlu(0).gt.0.5) kfs=-kfs
      elseif(kfs.gt.0) then
        kfsp=1
        kfsn=0
      else
        kfsp=0
        kfsn=1
      endif

c...sum branching ratios of allowed decay channels.
  110 nope=0
      brsu=0.
      do 120 idl=mdcy(kca,2),mdcy(kca,2)+mdcy(kca,3)-1
      if(mdme(idl,1).ne.1.and.kfsp*mdme(idl,1).ne.2.and.
     &kfsn*mdme(idl,1).ne.3) goto 120
      if(mdme(idl,2).gt.100) goto 120
      nope=nope+1
      brsu=brsu+brat(idl)
  120 continue
      if(nope.eq.0) then
        call luerrm(2,'(ludecy:) all decay channels closed by user')
        return
      endif

c...select decay channel among allowed ones.
  130 rbr=brsu*rlu(0)
      idl=mdcy(kca,2)-1
  140 idl=idl+1
      if(mdme(idl,1).ne.1.and.kfsp*mdme(idl,1).ne.2.and.
     &kfsn*mdme(idl,1).ne.3) then
        if(idl.lt.mdcy(kca,2)+mdcy(kca,3)-1) goto 140
      elseif(mdme(idl,2).gt.100) then
        if(idl.lt.mdcy(kca,2)+mdcy(kca,3)-1) goto 140
      else
        idc=idl
        rbr=rbr-brat(idl)
        if(idl.lt.mdcy(kca,2)+mdcy(kca,3)-1.and.rbr.gt.0.) goto 140
      endif

c...start readout of decay channel: matrix element, reset counters.
      mmat=mdme(idc,2)
  150 ntry=ntry+1
      if(ntry.gt.1000) then
        call luerrm(14,'(ludecy:) caught in infinite loop')
        if(mstu(21).ge.1) return
      endif
      i=n
      np=0
      nq=0
      mbst=0
      if(mmat.ge.11.and.mmat.ne.46.and.p(ip,4).gt.20.*p(ip,5)) mbst=1
      do 160 j=1,4
      pv(1,j)=0.
  160 if(mbst.eq.0) pv(1,j)=p(ip,j)
      if(mbst.eq.1) pv(1,4)=p(ip,5)
      pv(1,5)=p(ip,5)
      ps=0.
      psq=0.
      mrem=0

c...read out decay products. convert to standard flavour code.
      jtmax=5
      if(mdme(idc+1,2).eq.101) jtmax=10
      do 170 jt=1,jtmax
      if(jt.le.5) kp=kfdp(idc,jt)
      if(jt.ge.6) kp=kfdp(idc+1,jt-5)
      if(kp.eq.0) goto 170
      kpa=iabs(kp)
      kcp=lucomp(kpa)
      if(kchg(kcp,3).eq.0.and.kpa.ne.81.and.kpa.ne.82) then
        kfp=kp
      elseif(kpa.ne.81.and.kpa.ne.82) then
        kfp=kfs*kp
      elseif(kpa.eq.81.and.mod(kfa/1000,10).eq.0) then
        kfp=-kfs*mod(kfa/10,10)
      elseif(kpa.eq.81.and.mod(kfa/100,10).ge.mod(kfa/10,10)) then
        kfp=kfs*(100*mod(kfa/10,100)+3)
      elseif(kpa.eq.81) then
        kfp=kfs*(1000*mod(kfa/10,10)+100*mod(kfa/100,10)+1)
      elseif(kp.eq.82) then
        call lukfdi(-kfs*int(1.+(2.+parj(2))*rlu(0)),0,kfp,kdump)
        if(kfp.eq.0) goto 150
        mstj(93)=1
        if(pv(1,5).lt.parj(32)+2.*ulmass(kfp)) goto 150
      elseif(kp.eq.-82) then
        kfp=-kfp
        if(iabs(kfp).gt.10) kfp=kfp+isign(10000,kfp)
      endif
      if(kpa.eq.81.or.kpa.eq.82) kcp=lucomp(kfp)

c...add decay product to event record or to quark flavour list.
      kfpa=iabs(kfp)
      kqp=kchg(kcp,2)
      if(mmat.ge.11.and.mmat.le.30.and.kqp.ne.0) then
        nq=nq+1
        kflo(nq)=kfp
        mstj(93)=2
        psq=psq+ulmass(kflo(nq))
      elseif(mmat.ge.42.and.mmat.le.43.and.np.eq.3.and.mod(nq,2).eq.1)
     &then
        nq=nq-1
        ps=ps-p(i,5)
        k(i,1)=1
        kfi=k(i,2)
        call lukfdi(kfp,kfi,kfldmp,k(i,2))
        if(k(i,2).eq.0) goto 150
        mstj(93)=1
        p(i,5)=ulmass(k(i,2))
        ps=ps+p(i,5)
      else
        i=i+1
        np=np+1
        if(mmat.ne.33.and.kqp.ne.0) nq=nq+1
        if(mmat.eq.33.and.kqp.ne.0.and.kqp.ne.2) nq=nq+1
        k(i,1)=1+mod(nq,2)
        if(mmat.eq.4.and.jt.le.2.and.kfp.eq.21) k(i,1)=2
        if(mmat.eq.4.and.jt.eq.3) k(i,1)=1
        k(i,2)=kfp
        k(i,3)=ip
        k(i,4)=0
        k(i,5)=0
        p(i,5)=ulmass(kfp)
        if(mmat.eq.45.and.kfpa.eq.89) p(i,5)=parj(32)
        ps=ps+p(i,5)
      endif
  170 continue

c...choose decay multiplicity in phase space model.
  180 if(mmat.ge.11.and.mmat.le.30) then
        psp=ps
        cnde=parj(61)*log(max((pv(1,5)-ps-psq)/parj(62),1.1))
        if(mmat.eq.12) cnde=cnde+parj(63)
  190   ntry=ntry+1
        if(ntry.gt.1000) then
          call luerrm(14,'(ludecy:) caught in infinite loop')
          if(mstu(21).ge.1) return
        endif
        if(mmat.le.20) then
          gauss=sqrt(-2.*cnde*log(max(1e-10,rlu(0))))*
     &    sin(paru(2)*rlu(0))
          nd=0.5+0.5*np+0.25*nq+cnde+gauss
          if(nd.lt.np+nq/2.or.nd.lt.2.or.nd.gt.10) goto 190
          if(mmat.eq.13.and.nd.eq.2) goto 190
          if(mmat.eq.14.and.nd.le.3) goto 190
          if(mmat.eq.15.and.nd.le.4) goto 190
        else
          nd=mmat-20
        endif

c...form hadrons from flavour content.
        do 200 jt=1,4
  200   kfl1(jt)=kflo(jt)
        if(nd.eq.np+nq/2) goto 220
        do 210 i=n+np+1,n+nd-nq/2
        jt=1+int((nq-1)*rlu(0))
        call lukfdi(kfl1(jt),0,kfl2,k(i,2))
        if(k(i,2).eq.0) goto 190
  210   kfl1(jt)=-kfl2
  220   jt=2
        jt2=3
        jt3=4
        if(nq.eq.4.and.rlu(0).lt.parj(66)) jt=4
        if(jt.eq.4.and.isign(1,kfl1(1)*(10-iabs(kfl1(1))))*
     &  isign(1,kfl1(jt)*(10-iabs(kfl1(jt)))).gt.0) jt=3
        if(jt.eq.3) jt2=2
        if(jt.eq.4) jt3=2
        call lukfdi(kfl1(1),kfl1(jt),kfldmp,k(n+nd-nq/2+1,2))
        if(k(n+nd-nq/2+1,2).eq.0) goto 190
        if(nq.eq.4) call lukfdi(kfl1(jt2),kfl1(jt3),kfldmp,k(n+nd,2))
        if(nq.eq.4.and.k(n+nd,2).eq.0) goto 190

c...check that sum of decay product masses not too large.
        ps=psp
        do 230 i=n+np+1,n+nd
        k(i,1)=1
        k(i,3)=ip
        k(i,4)=0
        k(i,5)=0
        p(i,5)=ulmass(k(i,2))
  230   ps=ps+p(i,5)
        if(ps+parj(64).gt.pv(1,5)) goto 190

c...rescale energy to subtract off spectator quark mass.
      elseif((mmat.eq.31.or.mmat.eq.33.or.mmat.eq.44.or.mmat.eq.45).
     &and.np.ge.3) then
        ps=ps-p(n+np,5)
        pqt=(p(n+np,5)+parj(65))/pv(1,5)
        do 240 j=1,5
        p(n+np,j)=pqt*pv(1,j)
  240   pv(1,j)=(1.-pqt)*pv(1,j)
        if(ps+parj(64).gt.pv(1,5)) goto 150
        nd=np-1
        mrem=1

c...phase space factors imposed in w decay.
      elseif(mmat.eq.46) then
        mstj(93)=1
        psmc=ulmass(k(n+1,2))
        mstj(93)=1
        psmc=psmc+ulmass(k(n+2,2))
        if(max(ps,psmc)+parj(32).gt.pv(1,5)) goto 130
        hr1=(p(n+1,5)/pv(1,5))**2
        hr2=(p(n+2,5)/pv(1,5))**2
        if((1.-hr1-hr2)*(2.+hr1+hr2)*sqrt((1.-hr1-hr2)**2-4.*hr1*hr2).
     &  lt.2.*rlu(0)) goto 130
        nd=np

c...fully specified final state: check mass broadening effects.
      else
        if(np.ge.2.and.ps+parj(64).gt.pv(1,5)) goto 150
        nd=np
      endif

c...select w mass in decay q -> w + q, without w propagator.
      if(mmat.eq.45.and.mstj(25).le.0) then
        hlq=(parj(32)/pv(1,5))**2
        huq=(1.-(p(n+2,5)+parj(64))/pv(1,5))**2
        hrq=(p(n+2,5)/pv(1,5))**2
  250   hw=hlq+rlu(0)*(huq-hlq)
        if(hmeps(hw).lt.rlu(0)) goto 250
        p(n+1,5)=pv(1,5)*sqrt(hw)

c...ditto, including w propagator. divide mass range into three regions.
      elseif(mmat.eq.45) then
        hqw=(pv(1,5)/pmas(24,1))**2
        hlw=(parj(32)/pmas(24,1))**2
        huw=((pv(1,5)-p(n+2,5)-parj(64))/pmas(24,1))**2
        hrq=(p(n+2,5)/pv(1,5))**2
        hg=pmas(24,2)/pmas(24,1)
        hatl=atan((hlw-1.)/hg)
        hm=min(1.,huw-0.001)
        hmv1=hmeps(hm/hqw)/((hm-1.)**2+hg**2)
  260   hm=hm-hg
        hmv2=hmeps(hm/hqw)/((hm-1.)**2+hg**2)
        if(hmv2.gt.hmv1.and.hm-hg.gt.hlw) then
          hmv1=hmv2
          goto 260
        endif
        hmv=min(2.*hmv1,hmeps(hm/hqw)/hg**2)
        hm1=1.-sqrt(1./hmv-hg**2)
        if(hm1.gt.hlw.and.hm1.lt.hm) then
          hm=hm1
        elseif(hmv2.le.hmv1) then
          hm=max(hlw,hm-min(0.1,1.-hm))
        endif
        hatm=atan((hm-1.)/hg)
        hwt1=(hatm-hatl)/hg
        hwt2=hmv*(min(1.,huw)-hm)
        hwt3=0.
        if(huw.gt.1.) then
          hatu=atan((huw-1.)/hg)
          hmp1=hmeps(1./hqw)
          hwt3=hmp1*hatu/hg
        endif

c...select mass region and w mass there. accept according to weight.
  270   hreg=rlu(0)*(hwt1+hwt2+hwt3)
        if(hreg.le.hwt1) then
          hw=1.+hg*tan(hatl+rlu(0)*(hatm-hatl))
          hacc=hmeps(hw/hqw)
        elseif(hreg.le.hwt1+hwt2) then
          hw=hm+rlu(0)*(min(1.,huw)-hm)
          hacc=hmeps(hw/hqw)/((hw-1.)**2+hg**2)/hmv
        else
          hw=1.+hg*tan(rlu(0)*hatu)
          hacc=hmeps(hw/hqw)/hmp1
        endif
        if(hacc.lt.rlu(0)) goto 270
        p(n+1,5)=pmas(24,1)*sqrt(hw)
      endif

c...determine position of grandmother, number of sisters, q -> w sign.
      nm=0
      kfas=0
      msgn=0
      if(mmat.eq.3.or.mmat.eq.46) then
        im=k(ip,3)
        if(im.lt.0.or.im.ge.ip) im=0
        if(im.ne.0) kfam=iabs(k(im,2))
        if(im.ne.0.and.mmat.eq.3) then
          do 280 il=max(ip-2,im+1),min(ip+2,n)
          if(k(il,3).eq.im) nm=nm+1
  280     if(k(il,3).eq.im.and.il.ne.ip) isis=il
          if(nm.ne.2.or.kfam.le.100.or.mod(kfam,10).ne.1.or.
     &    mod(kfam/1000,10).ne.0) nm=0
          if(nm.eq.2) then
            kfas=iabs(k(isis,2))
            if((kfas.le.100.or.mod(kfas,10).ne.1.or.
     &      mod(kfas/1000,10).ne.0).and.kfas.ne.22) nm=0
          endif
        elseif(im.ne.0.and.mmat.eq.46) then
          msgn=isign(1,k(im,2)*k(ip,2))
          if(kfam.gt.100.and.mod(kfam/1000,10).eq.0) msgn=
     &    msgn*(-1)**mod(kfam/100,10)
        endif
      endif

c...kinematics of one-particle decays.
      if(nd.eq.1) then
        do 290 j=1,4
  290   p(n+1,j)=p(ip,j)
        goto 520
      endif

c...calculate maximum weight nd-particle decay.
      pv(nd,5)=p(n+nd,5)
      if(nd.ge.3) then
        wtmax=1./wtcor(nd-2)
        pmax=pv(1,5)-ps+p(n+nd,5)
        pmin=0.
        do 300 il=nd-1,1,-1
        pmax=pmax+p(n+il,5)
        pmin=pmin+p(n+il+1,5)
  300   wtmax=wtmax*pawt(pmax,pmin,p(n+il,5))
      endif

c...find virtual gamma mass in dalitz decay.
  310 if(nd.eq.2) then
      elseif(mmat.eq.2) then
        pmes=4.*pmas(11,1)**2
        pmrho2=pmas(131,1)**2
        pgrho2=pmas(131,2)**2
  320   pmst=pmes*(p(ip,5)**2/pmes)**rlu(0)
        wt=(1+0.5*pmes/pmst)*sqrt(max(0.,1.-pmes/pmst))*
     &  (1.-pmst/p(ip,5)**2)**3*(1.+pgrho2/pmrho2)/
     &  ((1.-pmst/pmrho2)**2+pgrho2/pmrho2)
        if(wt.lt.rlu(0)) goto 320
        pv(2,5)=max(2.00001*pmas(11,1),sqrt(pmst))

c...m-generator gives weight. if rejected, try again.
      else
  330   rord(1)=1.
        do 350 il1=2,nd-1
        rsav=rlu(0)
        do 340 il2=il1-1,1,-1
        if(rsav.le.rord(il2)) goto 350
  340   rord(il2+1)=rord(il2)
  350   rord(il2+1)=rsav
        rord(nd)=0.
        wt=1.
        do 360 il=nd-1,1,-1
        pv(il,5)=pv(il+1,5)+p(n+il,5)+(rord(il)-rord(il+1))*(pv(1,5)-ps)
  360   wt=wt*pawt(pv(il,5),pv(il+1,5),p(n+il,5))
        if(wt.lt.rlu(0)*wtmax) goto 330
      endif

c...perform two-particle decays in respective cm frame.
  370 do 390 il=1,nd-1
      pa=pawt(pv(il,5),pv(il+1,5),p(n+il,5))
      ue(3)=2.*rlu(0)-1.
      phi=paru(2)*rlu(0)
      ue(1)=sqrt(1.-ue(3)**2)*cos(phi)
      ue(2)=sqrt(1.-ue(3)**2)*sin(phi)
      do 380 j=1,3
      p(n+il,j)=pa*ue(j)
  380 pv(il+1,j)=-pa*ue(j)
      p(n+il,4)=sqrt(pa**2+p(n+il,5)**2)
  390 pv(il+1,4)=sqrt(pa**2+pv(il+1,5)**2)

c...lorentz transform decay products to lab frame.
      do 400 j=1,4
  400 p(n+nd,j)=pv(nd,j)
      do 430 il=nd-1,1,-1
      do 410 j=1,3
  410 be(j)=pv(il,j)/pv(il,4)
      ga=pv(il,4)/pv(il,5)
      do 430 i=n+il,n+nd
      bep=be(1)*p(i,1)+be(2)*p(i,2)+be(3)*p(i,3)
      do 420 j=1,3
  420 p(i,j)=p(i,j)+ga*(ga*bep/(1.+ga)+p(i,4))*be(j)
  430 p(i,4)=ga*(p(i,4)+bep)

c...check that no infinite loop in matrix element weight.
      ntry=ntry+1
      if(ntry.gt.800) goto 450

c...matrix elements for omega and phi decays.
      if(mmat.eq.1) then
        wt=(p(n+1,5)*p(n+2,5)*p(n+3,5))**2-(p(n+1,5)*four(n+2,n+3))**2
     &  -(p(n+2,5)*four(n+1,n+3))**2-(p(n+3,5)*four(n+1,n+2))**2
     &  +2.*four(n+1,n+2)*four(n+1,n+3)*four(n+2,n+3)
        if(max(wt*wtcor(9)/p(ip,5)**6,0.001).lt.rlu(0)) goto 310

c...matrix elements for pi0 or eta dalitz decay to gamma e+ e-.
      elseif(mmat.eq.2) then
        four12=four(n+1,n+2)
        four13=four(n+1,n+3)
        wt=(pmst-0.5*pmes)*(four12**2+four13**2)+
     &  pmes*(four12*four13+four12**2+four13**2)
        if(wt.lt.rlu(0)*0.25*pmst*(p(ip,5)**2-pmst)**2) goto 370

c...matrix element for s0 -> s1 + v1 -> s1 + s2 + s3 (s scalar,
c...v vector), of form cos**2(theta02) in v1 rest frame, and for
c...s0 -> gamma + v1 -> gamma + s2 + s3, of form sin**2(theta02).
      elseif(mmat.eq.3.and.nm.eq.2) then
        four10=four(ip,im)
        four12=four(ip,n+1)
        four02=four(im,n+1)
        pms1=p(ip,5)**2
        pms0=p(im,5)**2
        pms2=p(n+1,5)**2
        if(kfas.ne.22) hnum=(four10*four12-pms1*four02)**2
        if(kfas.eq.22) hnum=pms1*(2.*four10*four12*four02-
     &  pms1*four02**2-pms0*four12**2-pms2*four10**2+pms1*pms0*pms2)
        hnum=max(1e-6*pms1**2*pms0*pms2,hnum)
        hden=(four10**2-pms1*pms0)*(four12**2-pms1*pms2)
        if(hnum.lt.rlu(0)*hden) goto 370

c...matrix element for "onium" -> g + g + g or gamma + g + g.
      elseif(mmat.eq.4) then
        hx1=2.*four(ip,n+1)/p(ip,5)**2
        hx2=2.*four(ip,n+2)/p(ip,5)**2
        hx3=2.*four(ip,n+3)/p(ip,5)**2
        wt=((1.-hx1)/(hx2*hx3))**2+((1.-hx2)/(hx1*hx3))**2+
     &  ((1.-hx3)/(hx1*hx2))**2
        if(wt.lt.2.*rlu(0)) goto 310
        if(k(ip+1,2).eq.22.and.(1.-hx1)*p(ip,5)**2.lt.4.*parj(32)**2)
     &  goto 310

c...effective matrix element for nu spectrum in tau -> nu + hadrons.
      elseif(mmat.eq.41) then
        hx1=2.*four(ip,n+1)/p(ip,5)**2
        if(8.*hx1*(3.-2.*hx1)/9..lt.rlu(0)) goto 310

c...matrix elements for weak decays (only semileptonic for c and b)
      elseif(mmat.ge.42.and.mmat.le.44.and.nd.eq.3) then
        if(mbst.eq.0) wt=four(ip,n+1)*four(n+2,n+3)
        if(mbst.eq.1) wt=p(ip,5)*p(n+1,4)*four(n+2,n+3)
        if(wt.lt.rlu(0)*p(ip,5)*pv(1,5)**3/wtcor(10)) goto 310
      elseif(mmat.ge.42.and.mmat.le.44) then
        do 440 j=1,4
        p(n+np+1,j)=0.
        do 440 is=n+3,n+np
  440   p(n+np+1,j)=p(n+np+1,j)+p(is,j)
        if(mbst.eq.0) wt=four(ip,n+1)*four(n+2,n+np+1)
        if(mbst.eq.1) wt=p(ip,5)*p(n+1,4)*four(n+2,n+np+1)
        if(wt.lt.rlu(0)*p(ip,5)*pv(1,5)**3/wtcor(10)) goto 310

c...angular distribution in w decay.
      elseif(mmat.eq.46.and.msgn.ne.0) then
        if(msgn.gt.0) wt=four(im,n+1)*four(n+2,ip+1)
        if(msgn.lt.0) wt=four(im,n+2)*four(n+1,ip+1)
        if(wt.lt.rlu(0)*p(im,5)**4/wtcor(10)) goto 370
      endif

c...scale back energy and reattach spectator.
  450 if(mrem.eq.1) then
        do 460 j=1,5
  460   pv(1,j)=pv(1,j)/(1.-pqt)
        nd=nd+1
        mrem=0
      endif

c...low invariant mass for system with spectator quark gives particle,
c...not two jets. readjust momenta accordingly.
      if((mmat.eq.31.or.mmat.eq.45).and.nd.eq.3) then
        mstj(93)=1
        pm2=ulmass(k(n+2,2))
        mstj(93)=1
        pm3=ulmass(k(n+3,2))
        if(p(n+2,5)**2+p(n+3,5)**2+2.*four(n+2,n+3).ge.
     &  (parj(32)+pm2+pm3)**2) goto 520
        k(n+2,1)=1
        kftemp=k(n+2,2)
        call lukfdi(kftemp,k(n+3,2),kfldmp,k(n+2,2))
        if(k(n+2,2).eq.0) goto 150
        p(n+2,5)=ulmass(k(n+2,2))
        ps=p(n+1,5)+p(n+2,5)
        pv(2,5)=p(n+2,5)
        mmat=0
        nd=2
        goto 370
      elseif(mmat.eq.44) then
        mstj(93)=1
        pm3=ulmass(k(n+3,2))
        mstj(93)=1
        pm4=ulmass(k(n+4,2))
        if(p(n+3,5)**2+p(n+4,5)**2+2.*four(n+3,n+4).ge.
     &  (parj(32)+pm3+pm4)**2) goto 490
        k(n+3,1)=1
        kftemp=k(n+3,2)
        call lukfdi(kftemp,k(n+4,2),kfldmp,k(n+3,2))
        if(k(n+3,2).eq.0) goto 150
        p(n+3,5)=ulmass(k(n+3,2))
        do 470 j=1,3
  470   p(n+3,j)=p(n+3,j)+p(n+4,j)
        p(n+3,4)=sqrt(p(n+3,1)**2+p(n+3,2)**2+p(n+3,3)**2+p(n+3,5)**2)
        ha=p(n+1,4)**2-p(n+2,4)**2
        hb=ha-(p(n+1,5)**2-p(n+2,5)**2)
        hc=(p(n+1,1)-p(n+2,1))**2+(p(n+1,2)-p(n+2,2))**2+
     &  (p(n+1,3)-p(n+2,3))**2
        hd=(pv(1,4)-p(n+3,4))**2
        he=ha**2-2.*hd*(p(n+1,4)**2+p(n+2,4)**2)+hd**2
        hf=hd*hc-hb**2
        hg=hd*hc-ha*hb
        hh=(sqrt(hg**2+he*hf)-hg)/(2.*hf)
        do 480 j=1,3
        pcor=hh*(p(n+1,j)-p(n+2,j))
        p(n+1,j)=p(n+1,j)+pcor
  480   p(n+2,j)=p(n+2,j)-pcor
        p(n+1,4)=sqrt(p(n+1,1)**2+p(n+1,2)**2+p(n+1,3)**2+p(n+1,5)**2)
        p(n+2,4)=sqrt(p(n+2,1)**2+p(n+2,2)**2+p(n+2,3)**2+p(n+2,5)**2)
        nd=nd-1
      endif

c...check invariant mass of w jets. may give one particle or start over.
  490 if(mmat.ge.42.and.mmat.le.44.and.iabs(k(n+1,2)).lt.10) then
        pmr=sqrt(max(0.,p(n+1,5)**2+p(n+2,5)**2+2.*four(n+1,n+2)))
        mstj(93)=1
        pm1=ulmass(k(n+1,2))
        mstj(93)=1
        pm2=ulmass(k(n+2,2))
        if(pmr.gt.parj(32)+pm1+pm2) goto 500
        kfldum=int(1.5+rlu(0))
        call lukfdi(k(n+1,2),-isign(kfldum,k(n+1,2)),kfldmp,kf1)
        call lukfdi(k(n+2,2),-isign(kfldum,k(n+2,2)),kfldmp,kf2)
        if(kf1.eq.0.or.kf2.eq.0) goto 150
        psm=ulmass(kf1)+ulmass(kf2)
        if(mmat.eq.42.and.pmr.gt.parj(64)+psm) goto 500
        if(mmat.ge.43.and.pmr.gt.0.2*parj(32)+psm) goto 500
        if(nd.eq.4.or.kfa.eq.15) goto 150
        k(n+1,1)=1
        kftemp=k(n+1,2)
        call lukfdi(kftemp,k(n+2,2),kfldmp,k(n+1,2))
        if(k(n+1,2).eq.0) goto 150
        p(n+1,5)=ulmass(k(n+1,2))
        k(n+2,2)=k(n+3,2)
        p(n+2,5)=p(n+3,5)
        ps=p(n+1,5)+p(n+2,5)
        pv(2,5)=p(n+3,5)
        mmat=0
        nd=2
        goto 370
      endif

c...phase space decay of partons from w decay.
  500 if(mmat.eq.42.and.iabs(k(n+1,2)).lt.10) then
        kflo(1)=k(n+1,2)
        kflo(2)=k(n+2,2)
        k(n+1,1)=k(n+3,1)
        k(n+1,2)=k(n+3,2)
        do 510 j=1,5
        pv(1,j)=p(n+1,j)+p(n+2,j)
  510   p(n+1,j)=p(n+3,j)
        pv(1,5)=pmr
        n=n+1
        np=0
        nq=2
        ps=0.
        mstj(93)=2
        psq=ulmass(kflo(1))
        mstj(93)=2
        psq=psq+ulmass(kflo(2))
        mmat=11
        goto 180
      endif

c...boost back for rapidly moving particle.
  520 n=n+nd
      if(mbst.eq.1) then
        do 530 j=1,3
  530   be(j)=p(ip,j)/p(ip,4)
        ga=p(ip,4)/p(ip,5)
        do 550 i=nsav+1,n
        bep=be(1)*p(i,1)+be(2)*p(i,2)+be(3)*p(i,3)
        do 540 j=1,3
  540   p(i,j)=p(i,j)+ga*(ga*bep/(1.+ga)+p(i,4))*be(j)
  550   p(i,4)=ga*(p(i,4)+bep)
      endif

c...fill in position of decay vertex.
      do 570 i=nsav+1,n
      do 560 j=1,4
  560 v(i,j)=vdcy(j)
  570 v(i,5)=0.

c...set up for parton shower evolution from jets.
      if(mstj(23).ge.1.and.mmat.eq.4.and.k(nsav+1,2).eq.21) then
        k(nsav+1,1)=3
        k(nsav+2,1)=3
        k(nsav+3,1)=3
        k(nsav+1,4)=mstu(5)*(nsav+2)
        k(nsav+1,5)=mstu(5)*(nsav+3)
        k(nsav+2,4)=mstu(5)*(nsav+3)
        k(nsav+2,5)=mstu(5)*(nsav+1)
        k(nsav+3,4)=mstu(5)*(nsav+1)
        k(nsav+3,5)=mstu(5)*(nsav+2)
        mstj(92)=-(nsav+1)
      elseif(mstj(23).ge.1.and.mmat.eq.4) then
        k(nsav+2,1)=3
        k(nsav+3,1)=3
        k(nsav+2,4)=mstu(5)*(nsav+3)
        k(nsav+2,5)=mstu(5)*(nsav+3)
        k(nsav+3,4)=mstu(5)*(nsav+2)
        k(nsav+3,5)=mstu(5)*(nsav+2)
        mstj(92)=nsav+2
      elseif(mstj(23).ge.1.and.(mmat.eq.32.or.mmat.eq.44.or.mmat.eq.46).
     &and.iabs(k(nsav+1,2)).le.10.and.iabs(k(nsav+2,2)).le.10) then
        k(nsav+1,1)=3
        k(nsav+2,1)=3
        k(nsav+1,4)=mstu(5)*(nsav+2)
        k(nsav+1,5)=mstu(5)*(nsav+2)
        k(nsav+2,4)=mstu(5)*(nsav+1)
        k(nsav+2,5)=mstu(5)*(nsav+1)
        mstj(92)=nsav+1
      elseif(mstj(23).ge.1.and.(mmat.eq.32.or.mmat.eq.44.or.mmat.eq.46).
     &and.iabs(k(nsav+1,2)).le.20.and.iabs(k(nsav+2,2)).le.20) then
        mstj(92)=nsav+1
      elseif(mstj(23).ge.1.and.mmat.eq.33.and.iabs(k(nsav+2,2)).eq.21)
     &then
        k(nsav+1,1)=3
        k(nsav+2,1)=3
        k(nsav+3,1)=3
        kcp=lucomp(k(nsav+1,2))
        kqp=kchg(kcp,2)*isign(1,k(nsav+1,2))
        jcon=4
        if(kqp.lt.0) jcon=5
        k(nsav+1,jcon)=mstu(5)*(nsav+2)
        k(nsav+2,9-jcon)=mstu(5)*(nsav+1)
        k(nsav+2,jcon)=mstu(5)*(nsav+3)
        k(nsav+3,9-jcon)=mstu(5)*(nsav+2)
        mstj(92)=nsav+1
      elseif(mstj(23).ge.1.and.mmat.eq.33) then
        k(nsav+1,1)=3
        k(nsav+3,1)=3
        k(nsav+1,4)=mstu(5)*(nsav+3)
        k(nsav+1,5)=mstu(5)*(nsav+3)
        k(nsav+3,4)=mstu(5)*(nsav+1)
        k(nsav+3,5)=mstu(5)*(nsav+1)
        mstj(92)=nsav+1
      endif

c...mark decayed particle; special option for b-b~ mixing.
      if(k(ip,1).eq.5) k(ip,1)=15
      if(k(ip,1).le.10) k(ip,1)=11
      if(mmix.eq.1.and.mstj(26).eq.2.and.k(ip,1).eq.11) k(ip,1)=12
      k(ip,4)=nsav+1
      k(ip,5)=n

      return
      end

c*********************************************************************

      subroutine lukfdi(kfl1,kfl2,kfl3,kf)

c...purpose: to generate a new flavour pair and combine off a hadron.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat1/,/ludat2/

c...default flavour values. input consistency checks.
      kf1a=iabs(kfl1)
      kf2a=iabs(kfl2)
      kfl3=0
      kf=0
      if(kf1a.eq.0) return
      if(kf2a.ne.0) then
        if(kf1a.le.10.and.kf2a.le.10.and.kfl1*kfl2.gt.0) return
        if(kf1a.gt.10.and.kf2a.gt.10) return
        if((kf1a.gt.10.or.kf2a.gt.10).and.kfl1*kfl2.lt.0) return
      endif

c...check if tabulated flavour probabilities are to be used.
      if(mstj(15).eq.1) then
        ktab1=-1
        if(kf1a.ge.1.and.kf1a.le.6) ktab1=kf1a
        kfl1a=mod(kf1a/1000,10)
        kfl1b=mod(kf1a/100,10)
        kfl1s=mod(kf1a,10)
        if(kfl1a.ge.1.and.kfl1a.le.4.and.kfl1b.ge.1.and.kfl1b.le.4)
     &  ktab1=6+kfl1a*(kfl1a-2)+2*kfl1b+(kfl1s-1)/2
        if(kfl1a.ge.1.and.kfl1a.le.4.and.kfl1a.eq.kfl1b) ktab1=ktab1-1
        if(kf1a.ge.1.and.kf1a.le.6) kfl1a=kf1a
        ktab2=0
        if(kf2a.ne.0) then
          ktab2=-1
          if(kf2a.ge.1.and.kf2a.le.6) ktab2=kf2a
          kfl2a=mod(kf2a/1000,10)
          kfl2b=mod(kf2a/100,10)
          kfl2s=mod(kf2a,10)
          if(kfl2a.ge.1.and.kfl2a.le.4.and.kfl2b.ge.1.and.kfl2b.le.4)
     &    ktab2=6+kfl2a*(kfl2a-2)+2*kfl2b+(kfl2s-1)/2
          if(kfl2a.ge.1.and.kfl2a.le.4.and.kfl2a.eq.kfl2b) ktab2=ktab2-1
        endif
        if(ktab1.ge.0.and.ktab2.ge.0) goto 140
      endif

c...parameters and breaking diquark parameter combinations.
  100 par2=parj(2)
      par3=parj(3)
      par4=3.*parj(4)
      if(mstj(12).ge.2) then
        par3m=sqrt(parj(3))
        par4m=1./(3.*sqrt(parj(4)))
        pardm=parj(7)/(parj(7)+par3m*parj(6))
        pars0=parj(5)*(2.+(1.+par2*par3m*parj(7))*(1.+par4m))
        pars1=parj(7)*pars0/(2.*par3m)+parj(5)*(parj(6)*(1.+par4m)+
     &  par2*par3m*parj(6)*parj(7))
        pars2=parj(5)*2.*parj(6)*parj(7)*(par2*parj(7)+(1.+par4m)/par3m)
        parsm=max(pars0,pars1,pars2)
        par4=par4*(1.+parsm)/(1.+parsm/(3.*par4m))
      endif

c...choice of whether to generate meson or baryon.
      mbary=0
      kfda=0
      if(kf1a.le.10) then
        if(kf2a.eq.0.and.mstj(12).ge.1.and.(1.+parj(1))*rlu(0).gt.1.)
     &  mbary=1
        if(kf2a.gt.10) mbary=2
        if(kf2a.gt.10.and.kf2a.le.10000) kfda=kf2a
      else
        mbary=2
        if(kf1a.le.10000) kfda=kf1a
      endif

c...possibility of process diquark -> meson + new diquark.
      if(kfda.ne.0.and.mstj(12).ge.2) then
        kflda=mod(kfda/1000,10)
        kfldb=mod(kfda/100,10)
        kflds=mod(kfda,10)
        wtdq=pars0
        if(max(kflda,kfldb).eq.3) wtdq=pars1
        if(min(kflda,kfldb).eq.3) wtdq=pars2
        if(kflds.eq.1) wtdq=wtdq/(3.*par4m)
        if((1.+wtdq)*rlu(0).gt.1.) mbary=-1
        if(mbary.eq.-1.and.kf2a.ne.0) return
      endif

c...flavour for meson, possibly with new flavour.
      if(mbary.le.0) then
        kfs=isign(1,kfl1)
        if(mbary.eq.0) then
          if(kf2a.eq.0) kfl3=isign(1+int((2.+par2)*rlu(0)),-kfl1)
          kfla=max(kf1a,kf2a+iabs(kfl3))
          kflb=min(kf1a,kf2a+iabs(kfl3))
          if(kfla.ne.kf1a) kfs=-kfs

c...splitting of diquark into meson plus new diquark.
        else
          kfl1a=mod(kf1a/1000,10)
          kfl1b=mod(kf1a/100,10)
  110     kfl1d=kfl1a+int(rlu(0)+0.5)*(kfl1b-kfl1a)
          kfl1e=kfl1a+kfl1b-kfl1d
          if((kfl1d.eq.3.and.rlu(0).gt.pardm).or.(kfl1e.eq.3.and.
     &    rlu(0).lt.pardm)) then
            kfl1d=kfl1a+kfl1b-kfl1d
            kfl1e=kfl1a+kfl1b-kfl1e
          endif
          kfl3a=1+int((2.+par2*par3m*parj(7))*rlu(0))
          if((kfl1e.ne.kfl3a.and.rlu(0).gt.(1.+par4m)/max(2.,1.+par4m)).
     &    or.(kfl1e.eq.kfl3a.and.rlu(0).gt.2./max(2.,1.+par4m)))
     &    goto 110
          kflds=3
          if(kfl1e.ne.kfl3a) kflds=2*int(rlu(0)+1./(1.+par4m))+1
          kfl3=isign(10000+1000*max(kfl1e,kfl3a)+100*min(kfl1e,kfl3a)+
     &    kflds,-kfl1)
          kfla=max(kfl1d,kfl3a)
          kflb=min(kfl1d,kfl3a)
          if(kfla.ne.kfl1d) kfs=-kfs
        endif

c...form meson, with spin and flavour mixing for diagonal states.
        if(kfla.le.2) kmul=int(parj(11)+rlu(0))
        if(kfla.eq.3) kmul=int(parj(12)+rlu(0))
        if(kfla.ge.4) kmul=int(parj(13)+rlu(0))
        if(kmul.eq.0.and.parj(14).gt.0.) then
          if(rlu(0).lt.parj(14)) kmul=2
        elseif(kmul.eq.1.and.parj(15)+parj(16)+parj(17).gt.0.) then
          rmul=rlu(0)
          if(rmul.lt.parj(15)) kmul=3
          if(kmul.eq.1.and.rmul.lt.parj(15)+parj(16)) kmul=4
          if(kmul.eq.1.and.rmul.lt.parj(15)+parj(16)+parj(17)) kmul=5
        endif
        kfls=3
        if(kmul.eq.0.or.kmul.eq.3) kfls=1
        if(kmul.eq.5) kfls=5
        if(kfla.ne.kflb) then
          kf=(100*kfla+10*kflb+kfls)*kfs*(-1)**kfla
        else
          rmix=rlu(0)
          imix=2*kfla+10*kmul
          if(kfla.le.3) kf=110*(1+int(rmix+parf(imix-1))+
     &    int(rmix+parf(imix)))+kfls
          if(kfla.ge.4) kf=110*kfla+kfls
        endif
        if(kmul.eq.2.or.kmul.eq.3) kf=kf+isign(10000,kf)
        if(kmul.eq.4) kf=kf+isign(20000,kf)

c...generate diquark flavour.
      else
  120   if(kf1a.le.10.and.kf2a.eq.0) then
          kfla=kf1a
  130     kflb=1+int((2.+par2*par3)*rlu(0))
          kflc=1+int((2.+par2*par3)*rlu(0))
          kflds=1
          if(kflb.ge.kflc) kflds=3
          if(kflds.eq.1.and.par4*rlu(0).gt.1.) goto 130
          if(kflds.eq.3.and.par4.lt.rlu(0)) goto 130
          kfl3=isign(1000*max(kflb,kflc)+100*min(kflb,kflc)+kflds,kfl1)

c...take diquark flavour from input.
        elseif(kf1a.le.10) then
          kfla=kf1a
          kflb=mod(kf2a/1000,10)
          kflc=mod(kf2a/100,10)
          kflds=mod(kf2a,10)

c...generate (or take from input) quark to go with diquark.
        else
          if(kf2a.eq.0) kfl3=isign(1+int((2.+par2)*rlu(0)),kfl1)
          kfla=kf2a+iabs(kfl3)
          kflb=mod(kf1a/1000,10)
          kflc=mod(kf1a/100,10)
          kflds=mod(kf1a,10)
        endif

c...su(6) factors for formation of baryon. try again if fails.
        kbary=kflds
        if(kflds.eq.3.and.kflb.ne.kflc) kbary=5
        if(kfla.ne.kflb.and.kfla.ne.kflc) kbary=kbary+1
        wt=parf(60+kbary)+parj(18)*parf(70+kbary)
        if(mbary.eq.1.and.mstj(12).ge.2) then
          wtdq=pars0
          if(max(kflb,kflc).eq.3) wtdq=pars1
          if(min(kflb,kflc).eq.3) wtdq=pars2
          if(kflds.eq.1) wtdq=wtdq/(3.*par4m)
          if(kflds.eq.1) wt=wt*(1.+wtdq)/(1.+parsm/(3.*par4m))
          if(kflds.eq.3) wt=wt*(1.+wtdq)/(1.+parsm)
        endif
        if(kf2a.eq.0.and.wt.lt.rlu(0)) goto 120

c...form baryon. distinguish lambda- and sigmalike baryons.
        kfld=max(kfla,kflb,kflc)
        kflf=min(kfla,kflb,kflc)
        kfle=kfla+kflb+kflc-kfld-kflf
        kfls=2
        if((parf(60+kbary)+parj(18)*parf(70+kbary))*rlu(0).gt.
     &  parf(60+kbary)) kfls=4
        kfll=0
        if(kfls.eq.2.and.kfld.gt.kfle.and.kfle.gt.kflf) then
          if(kflds.eq.1.and.kfla.eq.kfld) kfll=1
          if(kflds.eq.1.and.kfla.ne.kfld) kfll=int(0.25+rlu(0))
          if(kflds.eq.3.and.kfla.ne.kfld) kfll=int(0.75+rlu(0))
        endif
        if(kfll.eq.0) kf=isign(1000*kfld+100*kfle+10*kflf+kfls,kfl1)
        if(kfll.eq.1) kf=isign(1000*kfld+100*kflf+10*kfle+kfls,kfl1)
      endif
      return

c...use tabulated probabilities to select new flavour and hadron.
  140 if(ktab2.eq.0.and.mstj(12).le.0) then
        kt3l=1
        kt3u=6
      elseif(ktab2.eq.0.and.ktab1.ge.7.and.mstj(12).le.1) then
        kt3l=1
        kt3u=6
      elseif(ktab2.eq.0) then
        kt3l=1
        kt3u=22
      else
        kt3l=ktab2
        kt3u=ktab2
      endif
      rfl=0.
      do 150 kts=0,2
      do 150 kt3=kt3l,kt3u
      rfl=rfl+parf(120+80*ktab1+25*kts+kt3)
  150 continue
      rfl=rlu(0)*rfl
      do 160 kts=0,2
      ktabs=kts
      do 160 kt3=kt3l,kt3u
      ktab3=kt3
      rfl=rfl-parf(120+80*ktab1+25*kts+kt3)
  160 if(rfl.le.0.) goto 170
  170 continue

c...reconstruct flavour of produced quark/diquark.
      if(ktab3.le.6) then
        kfl3a=ktab3
        kfl3b=0
        kfl3=isign(kfl3a,kfl1*(2*ktab1-13))
      else
        kfl3a=1
        if(ktab3.ge.8) kfl3a=2
        if(ktab3.ge.11) kfl3a=3
        if(ktab3.ge.16) kfl3a=4
        kfl3b=(ktab3-6-kfl3a*(kfl3a-2))/2
        kfl3=1000*kfl3a+100*kfl3b+1
        if(kfl3a.eq.kfl3b.or.ktab3.ne.6+kfl3a*(kfl3a-2)+2*kfl3b) kfl3=
     &  kfl3+2
        kfl3=isign(kfl3,kfl1*(13-2*ktab1))
      endif

c...reconstruct meson code.
      if(kfl3a.eq.kfl1a.and.kfl3b.eq.kfl1b.and.(kfl3a.le.3.or.
     &kfl3b.ne.0)) then
        rfl=rlu(0)*(parf(143+80*ktab1+25*ktabs)+parf(144+80*ktab1+
     &  25*ktabs)+parf(145+80*ktab1+25*ktabs))
        kf=110+2*ktabs+1
        if(rfl.gt.parf(143+80*ktab1+25*ktabs)) kf=220+2*ktabs+1
        if(rfl.gt.parf(143+80*ktab1+25*ktabs)+parf(144+80*ktab1+
     &  25*ktabs)) kf=330+2*ktabs+1
      elseif(ktab1.le.6.and.ktab3.le.6) then
        kfla=max(ktab1,ktab3)
        kflb=min(ktab1,ktab3)
        kfs=isign(1,kfl1)
        if(kfla.ne.kf1a) kfs=-kfs
        kf=(100*kfla+10*kflb+2*ktabs+1)*kfs*(-1)**kfla
      elseif(ktab1.ge.7.and.ktab3.ge.7) then
        kfs=isign(1,kfl1)
        if(kfl1a.eq.kfl3a) then
          kfla=max(kfl1b,kfl3b)
          kflb=min(kfl1b,kfl3b)
          if(kfla.ne.kfl1b) kfs=-kfs
        elseif(kfl1a.eq.kfl3b) then
          kfla=kfl3a
          kflb=kfl1b
          kfs=-kfs
        elseif(kfl1b.eq.kfl3a) then
          kfla=kfl1a
          kflb=kfl3b
        elseif(kfl1b.eq.kfl3b) then
          kfla=max(kfl1a,kfl3a)
          kflb=min(kfl1a,kfl3a)
          if(kfla.ne.kfl1a) kfs=-kfs
        else
          call luerrm(2,'(lukfdi:) no matching flavours for qq -> qq')
          goto 100
        endif
        kf=(100*kfla+10*kflb+2*ktabs+1)*kfs*(-1)**kfla

c...reconstruct baryon code.
      else
        if(ktab1.ge.7) then
          kfla=kfl3a
          kflb=kfl1a
          kflc=kfl1b
        else
          kfla=kfl1a
          kflb=kfl3a
          kflc=kfl3b
        endif
        kfld=max(kfla,kflb,kflc)
        kflf=min(kfla,kflb,kflc)
        kfle=kfla+kflb+kflc-kfld-kflf
        if(ktabs.eq.0) kf=isign(1000*kfld+100*kflf+10*kfle+2,kfl1)
        if(ktabs.ge.1) kf=isign(1000*kfld+100*kfle+10*kflf+2*ktabs,kfl1)
      endif

c...check that constructed flavour code is an allowed one.
      if(kfl2.ne.0) kfl3=0
      kc=lucomp(kf)
      if(kc.eq.0) then
        call luerrm(2,'(lukfdi:) user-defined flavour probabilities '//
     &  'failed')
        goto 100
      endif

      return
      end

c*********************************************************************

      subroutine luptdi(kfl,px,py)

c...purpose: to generate transverse momentum according to a gaussian.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/

c...generate p_t and azimuthal angle, gives p_x and p_y.
      kfla=iabs(kfl)
      pt=parj(21)*sqrt(-log(max(1e-10,rlu(0))))
      if(mstj(91).eq.1) pt=parj(22)*pt
      if(kfla.eq.0.and.mstj(13).le.0) pt=0.
      phi=paru(2)*rlu(0)
      px=pt*cos(phi)
      py=pt*sin(phi)

      return
      end

c*********************************************************************

      subroutine luzdis(kfl1,kfl2,pr,z)

c...purpose: to generate the longitudinal splitting variable z.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat1/,/ludat2/

c...check if heavy flavour fragmentation.
      kfla=iabs(kfl1)
      kflb=iabs(kfl2)
      kflh=kfla
      if(kfla.ge.10) kflh=mod(kfla/1000,10)

c...lund symmetric scaling function: determine parameters of shape.
      if(mstj(11).eq.1.or.(mstj(11).eq.3.and.kflh.le.3).or.
     &mstj(11).ge.4) then
        fa=parj(41)
        if(mstj(91).eq.1) fa=parj(43)
        if(kflb.ge.10) fa=fa+parj(45)
        fbb=parj(42)
        if(mstj(91).eq.1) fbb=parj(44)
        fb=fbb*pr
        fc=1.
        if(kfla.ge.10) fc=fc-parj(45)
        if(kflb.ge.10) fc=fc+parj(45)
        if(mstj(11).ge.4.and.kflh.ge.4.and.kflh.le.5) then
          fred=parj(46)
          if(mstj(11).eq.5.and.kflh.eq.5) fred=parj(47)
          fc=fc+fred*fbb*parf(100+kflh)**2
        elseif(mstj(11).ge.4.and.kflh.ge.6.and.kflh.le.8) then
          fred=parj(46)
          if(mstj(11).eq.5) fred=parj(48)
          fc=fc+fred*fbb*pmas(kflh,1)**2
        endif
        mc=1
        if(abs(fc-1.).gt.0.01) mc=2

c...determine position of maximum. special cases for a = 0 or a = c.
        if(fa.lt.0.02) then
          ma=1
          zmax=1.
          if(fc.gt.fb) zmax=fb/fc
        elseif(abs(fc-fa).lt.0.01) then
          ma=2
          zmax=fb/(fb+fc)
        else
          ma=3
          zmax=0.5*(fb+fc-sqrt((fb-fc)**2+4.*fa*fb))/(fc-fa)
          if(zmax.gt.0.9999.and.fb.gt.100.) zmax=min(zmax,1.-fa/fb)
        endif

c...subdivide z range if distribution very peaked near endpoint.
        mmax=2
        if(zmax.lt.0.1) then
          mmax=1
          zdiv=2.75*zmax
          if(mc.eq.1) then
            fint=1.-log(zdiv)
          else
            zdivc=zdiv**(1.-fc)
            fint=1.+(1.-1./zdivc)/(fc-1.)
          endif
        elseif(zmax.gt.0.85.and.fb.gt.1.) then
          mmax=3
          fscb=sqrt(4.+(fc/fb)**2)
          zdiv=fscb-1./zmax-(fc/fb)*log(zmax*0.5*(fscb+fc/fb))
          if(ma.ge.2) zdiv=zdiv+(fa/fb)*log(1.-zmax)
          zdiv=min(zmax,max(0.,zdiv))
          fint=1.+fb*(1.-zdiv)
        endif

c...choice of z, preweighted for peaks at low or high z.
  100   z=rlu(0)
        fpre=1.
        if(mmax.eq.1) then
          if(fint*rlu(0).le.1.) then
            z=zdiv*z
          elseif(mc.eq.1) then
            z=zdiv**z
            fpre=zdiv/z
          else
            z=1./(zdivc+z*(1.-zdivc))**(1./(1.-fc))
            fpre=(zdiv/z)**fc
          endif
        elseif(mmax.eq.3) then
          if(fint*rlu(0).le.1.) then
            z=zdiv+log(z)/fb
            fpre=exp(fb*(z-zdiv))
          else
            z=zdiv+z*(1.-zdiv)
          endif
        endif

c...weighting according to correct formula.
        if(z.le.0..or.z.ge.1.) goto 100
        fexp=fc*log(zmax/z)+fb*(1./zmax-1./z)
        if(ma.ge.2) fexp=fexp+fa*log((1.-z)/(1.-zmax))
        fval=exp(max(-50.,fexp))
        if(fval.lt.rlu(0)*fpre) goto 100

c...generate z according to field-feynman, slac, (1-z)**c or z**c.
      else
        fc=parj(50+max(1,kflh))
        if(mstj(91).eq.1) fc=parj(59)
  110   z=rlu(0)
        if(fc.ge.0..and.fc.le.1.) then
          if(fc.gt.rlu(0)) z=1.-z**(1./3.)
        elseif(fc.gt.-1.) then
          if(-4.*fc*z*(1.-z)**2.lt.rlu(0)*((1.-z)**2-fc*z)**2) goto 110
        else
          if(fc.gt.0.) z=1.-z**(1./fc)
          if(fc.lt.0.) z=z**(-1./fc)
        endif
      endif

      return
      end

c*********************************************************************

      subroutine lushow(ip1,ip2,qmax)

c...purpose: to generate timelike parton showers from given partons.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension pmth(5,40),ps(5),pma(4),pmsd(4),iep(4),ipa(4),
     &kfla(4),kfld(4),kfl(4),itry(4),isi(4),isl(4),dp(4),dpt(5,4),
     &ksh(0:40)

c...initialization of cutoff masses etc.
      if(mstj(41).le.0.or.(mstj(41).eq.1.and.qmax.le.parj(82)).or.
     &qmax.le.min(parj(82),parj(83)).or.mstj(41).ge.3) return
      do 101 if=0,40
  101 ksh(if)=0
      ksh(21)=1
      pmth(1,21)=ulmass(21)
      pmth(2,21)=sqrt(pmth(1,21)**2+0.25*parj(82)**2)
      pmth(3,21)=2.*pmth(2,21)
      pmth(4,21)=pmth(3,21)
      pmth(5,21)=pmth(3,21)
      pmth(1,22)=ulmass(22)
      pmth(2,22)=sqrt(pmth(1,22)**2+0.25*parj(83)**2)
      pmth(3,22)=2.*pmth(2,22)
      pmth(4,22)=pmth(3,22)
      pmth(5,22)=pmth(3,22)
      pmqth1=parj(82)
      if(mstj(41).eq.2) pmqth1=min(parj(82),parj(83))
      pmqth2=pmth(2,21)
      if(mstj(41).eq.2) pmqth2=min(pmth(2,21),pmth(2,22))
      do 100 if=1,8
      ksh(if)=1
      pmth(1,if)=ulmass(if)
      pmth(2,if)=sqrt(pmth(1,if)**2+0.25*pmqth1**2)
      pmth(3,if)=pmth(2,if)+pmqth2
      pmth(4,if)=sqrt(pmth(1,if)**2+0.25*parj(82)**2)+pmth(2,21)
  100 pmth(5,if)=sqrt(pmth(1,if)**2+0.25*parj(83)**2)+pmth(2,22)
      do 105 if=11,17,2
      if(mstj(41).eq.2) ksh(if)=1
      pmth(1,if)=ulmass(if)
      pmth(2,if)=sqrt(pmth(1,if)**2+0.25*parj(83)**2)
      pmth(3,if)=pmth(2,if)+pmth(2,22)
      pmth(4,if)=pmth(3,if)
  105 pmth(5,if)=pmth(3,if)
      pt2min=max(0.5*parj(82),1.1*parj(81))**2
      alams=parj(81)**2
      alfm=log(pt2min/alams)

c...store positions of shower initiating partons.
      m3jc=0
      if(ip1.gt.0.and.ip1.le.min(n,mstu(4)-mstu(32)).and.ip2.eq.0) then
        npa=1
        ipa(1)=ip1
      elseif(min(ip1,ip2).gt.0.and.max(ip1,ip2).le.min(n,mstu(4)-
     &mstu(32))) then
        npa=2
        ipa(1)=ip1
        ipa(2)=ip2
      elseif(ip1.gt.0.and.ip1.le.min(n,mstu(4)-mstu(32)).and.ip2.lt.0.
     &and.ip2.ge.-3) then
        npa=iabs(ip2)
        do 110 i=1,npa
  110   ipa(i)=ip1+i-1
      else
        call luerrm(12,
     &  '(lushow:) failed to reconstruct showering system')
        if(mstu(21).ge.1) return
      endif

c...check on phase space available for emission.
      irej=0
      do 120 j=1,5
  120 ps(j)=0.
      pm=0.
      do 130 i=1,npa
      kfla(i)=iabs(k(ipa(i),2))
      pma(i)=p(ipa(i),5)
      if(kfla(i).le.40) then
        if(ksh(kfla(i)).eq.1) pma(i)=pmth(3,kfla(i))
      endif
      pm=pm+pma(i)
      if(kfla(i).gt.40) then
        irej=irej+1
      else
        if(ksh(kfla(i)).eq.0.or.pma(i).gt.qmax) irej=irej+1
      endif
      do 130 j=1,4
  130 ps(j)=ps(j)+p(ipa(i),j)
      if(irej.eq.npa) return
      ps(5)=sqrt(max(0.,ps(4)**2-ps(1)**2-ps(2)**2-ps(3)**2))
      if(npa.eq.1) ps(5)=ps(4)
      if(ps(5).le.pm+pmqth1) return
      if(npa.eq.2.and.mstj(47).ge.1) then
        if(kfla(1).ge.1.and.kfla(1).le.8.and.kfla(2).ge.1.and.
     &  kfla(2).le.8) m3jc=1
        if((kfla(1).eq.11.or.kfla(1).eq.13.or.kfla(1).eq.15.or.
     &  kfla(1).eq.17).and.kfla(2).eq.kfla(1)) m3jc=1
        if((kfla(1).eq.11.or.kfla(1).eq.13.or.kfla(1).eq.15.or.
     &  kfla(1).eq.17).and.kfla(2).eq.kfla(1)+1) m3jc=1
        if((kfla(1).eq.12.or.kfla(1).eq.14.or.kfla(1).eq.16.or.
     &  kfla(1).eq.18).and.kfla(2).eq.kfla(1)-1) m3jc=1
        if(mstj(47).ge.2) m3jc=1
      endif

c...define imagined single initiator of shower for parton system.
      ns=n
      if(n.gt.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lushow:) no more memory left in lujets')
        if(mstu(21).ge.1) return
      endif
      if(npa.ge.2) then
        k(n+1,1)=11
        k(n+1,2)=21
        k(n+1,3)=0
        k(n+1,4)=0
        k(n+1,5)=0
        p(n+1,1)=0.
        p(n+1,2)=0.
        p(n+1,3)=0.
        p(n+1,4)=ps(5)
        p(n+1,5)=ps(5)
        v(n+1,5)=ps(5)**2
        n=n+1
      endif

c...loop over partons that may branch.
      nep=npa
      im=ns
      if(npa.eq.1) im=ns-1
  140 im=im+1
      if(n.gt.ns) then
        if(im.gt.n) goto 380
        kflm=iabs(k(im,2))
        if(kflm.gt.40) goto 140
        if(ksh(kflm).eq.0) goto 140
        if(p(im,5).lt.pmth(2,kflm)) goto 140
        igm=k(im,3)
      else
        igm=-1
      endif
      if(n+nep.gt.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lushow:) no more memory left in lujets')
        if(mstu(21).ge.1) return
      endif

c...position of aunt (sister to branching parton).
c...origin and flavour of daughters.
      iau=0
      if(igm.gt.0) then
        if(k(im-1,3).eq.igm) iau=im-1
        if(n.ge.im+1.and.k(im+1,3).eq.igm) iau=im+1
      endif
      if(igm.ge.0) then
        k(im,4)=n+1
        do 150 i=1,nep
  150   k(n+i,3)=im
      else
        k(n+1,3)=ipa(1)
      endif
      if(igm.le.0) then
        do 160 i=1,nep
  160   k(n+i,2)=k(ipa(i),2)
      elseif(kflm.ne.21) then
        k(n+1,2)=k(im,2)
        k(n+2,2)=k(im,5)
      elseif(k(im,5).eq.21) then
        k(n+1,2)=21
        k(n+2,2)=21
      else
        k(n+1,2)=k(im,5)
        k(n+2,2)=-k(im,5)
      endif

c...reset flags on daughers and tries made.
      do 170 ip=1,nep
      k(n+ip,1)=3
      k(n+ip,4)=0
      k(n+ip,5)=0
      kfld(ip)=iabs(k(n+ip,2))
      if(kchg(lucomp(kfld(ip)),2).eq.0) k(n+ip,1)=1
      itry(ip)=0
      isl(ip)=0
      isi(ip)=0
      if(kfld(ip).le.40) then
        if(ksh(kfld(ip)).eq.1) isi(ip)=1
      endif
  170 continue
      islm=0

c...maximum virtuality of daughters.
      if(igm.le.0) then
        do 180 i=1,npa
        if(npa.ge.3) p(n+i,4)=(ps(4)*p(ipa(i),4)-ps(1)*p(ipa(i),1)-
     &  ps(2)*p(ipa(i),2)-ps(3)*p(ipa(i),3))/ps(5)
        p(n+i,5)=min(qmax,ps(5))
        if(npa.ge.3) p(n+i,5)=min(p(n+i,5),p(n+i,4))
  180   if(isi(i).eq.0) p(n+i,5)=p(ipa(i),5)
      else
        if(mstj(43).le.2) pem=v(im,2)
        if(mstj(43).ge.3) pem=p(im,4)
        p(n+1,5)=min(p(im,5),v(im,1)*pem)
        p(n+2,5)=min(p(im,5),(1.-v(im,1))*pem)
        if(k(n+2,2).eq.22) p(n+2,5)=pmth(1,22)
      endif
      do 190 i=1,nep
      pmsd(i)=p(n+i,5)
      if(isi(i).eq.1) then
        if(p(n+i,5).le.pmth(3,kfld(i))) p(n+i,5)=pmth(1,kfld(i))
      endif
  190 v(n+i,5)=p(n+i,5)**2

c...choose one of the daughters for evolution.
  200 inum=0
      if(nep.eq.1) inum=1
      do 210 i=1,nep
  210 if(inum.eq.0.and.isl(i).eq.1) inum=i
      do 220 i=1,nep
      if(inum.eq.0.and.itry(i).eq.0.and.isi(i).eq.1) then
        if(p(n+i,5).ge.pmth(2,kfld(i))) inum=i
      endif
  220 continue
      if(inum.eq.0) then
        rmax=0.
        do 230 i=1,nep
        if(isi(i).eq.1.and.pmsd(i).ge.pmqth2) then
          rpm=p(n+i,5)/pmsd(i)
          if(rpm.gt.rmax.and.p(n+i,5).ge.pmth(2,kfld(i))) then
            rmax=rpm
            inum=i
          endif
        endif
  230   continue
      endif

c...store information on choice of evolving daughter.
      inum=max(1,inum)
      iep(1)=n+inum
      do 240 i=2,nep
      iep(i)=iep(i-1)+1
  240 if(iep(i).gt.n+nep) iep(i)=n+1
      do 250 i=1,nep
  250 kfl(i)=iabs(k(iep(i),2))
      itry(inum)=itry(inum)+1
      if(itry(inum).gt.200) then
        call luerrm(14,'(lushow:) caught in infinite loop')
        if(mstu(21).ge.1) return
      endif
      z=0.5
      if(kfl(1).gt.40) goto 300
      if(ksh(kfl(1)).eq.0) goto 300
      if(p(iep(1),5).lt.pmth(2,kfl(1))) goto 300

c...calculate allowed z range.
      if(nep.eq.1) then
        pmed=ps(4)
      elseif(igm.eq.0.or.mstj(43).le.2) then
        pmed=p(im,5)
      else
        if(inum.eq.1) pmed=v(im,1)*pem
        if(inum.eq.2) pmed=(1.-v(im,1))*pem
      endif
      if(mod(mstj(43),2).eq.1) then
        zc=pmth(2,21)/pmed
        zce=pmth(2,22)/pmed
      else
        zc=0.5*(1.-sqrt(max(0.,1.-(2.*pmth(2,21)/pmed)**2)))
        if(zc.lt.1e-4) zc=(pmth(2,21)/pmed)**2
        zce=0.5*(1.-sqrt(max(0.,1.-(2.*pmth(2,22)/pmed)**2)))
        if(zce.lt.1e-4) zce=(pmth(2,22)/pmed)**2
      endif
      zc=min(zc,0.491)
      zce=min(zce,0.491)
      if((mstj(41).eq.1.and.zc.gt.0.49).or.(mstj(41).eq.2.and.
     &min(zc,zce).gt.0.49)) then
        p(iep(1),5)=pmth(1,kfl(1))
        v(iep(1),5)=p(iep(1),5)**2
        goto 300
      endif

c...integral of altarelli-parisi z kernel for qcd.
      if(mstj(49).eq.0.and.kfl(1).eq.21) then
        fbr=6.*log((1.-zc)/zc)+mstj(45)*(0.5-zc)
      elseif(mstj(49).eq.0) then
        fbr=(8./3.)*log((1.-zc)/zc)

c...integral of altarelli-parisi z kernel for scalar gluon.
      elseif(mstj(49).eq.1.and.kfl(1).eq.21) then
        fbr=(parj(87)+mstj(45)*parj(88))*(1.-2.*zc)
      elseif(mstj(49).eq.1) then
        fbr=(1.-2.*zc)/3.
        if(igm.eq.0.and.m3jc.eq.1) fbr=4.*fbr

c...integral of altarelli-parisi z kernel for abelian vector gluon.
      elseif(kfl(1).eq.21) then
        fbr=6.*mstj(45)*(0.5-zc)
      else
        fbr=2.*log((1.-zc)/zc)
      endif

c...reset qcd probability for lepton.
      if(kfl(1).ge.11.and.kfl(1).le.18) fbr=0.

c...integral of altarelli-parisi kernel for photon emission.
      if(mstj(41).eq.2.and.kfl(1).ge.1.and.kfl(1).le.18)
     &fbre=(kchg(kfl(1),1)/3.)**2*2.*log((1.-zce)/zce)

c...inner veto algorithm starts. find maximum mass for evolution.
  260 pms=v(iep(1),5)
      if(igm.ge.0) then
        pm2=0.
        do 270 i=2,nep
        pm=p(iep(i),5)
        if(kfl(i).le.40) then
          if(ksh(kfl(i)).eq.1) pm=pmth(2,kfl(i))
        endif
  270   pm2=pm2+pm
        pms=min(pms,(p(im,5)-pm2)**2)
      endif

c...select mass for daughter in qcd evolution.
      b0=27./6.
      do 280 if=4,mstj(45)
  280 if(pms.gt.4.*pmth(2,if)**2) b0=(33.-2.*if)/6.
      if(fbr.lt.1e-3) then
        pmsqcd=0.
      elseif(mstj(44).le.0) then
        pmsqcd=pms*exp(max(-50.,log(rlu(0))*paru(2)/(paru(111)*fbr)))
      elseif(mstj(44).eq.1) then
        pmsqcd=4.*alams*(0.25*pms/alams)**(rlu(0)**(b0/fbr))
      else
        pmsqcd=pms*exp(max(-50.,alfm*b0*log(rlu(0))/fbr))
      endif
      if(zc.gt.0.49.or.pmsqcd.le.pmth(4,kfl(1))**2) pmsqcd=
     &pmth(2,kfl(1))**2
      v(iep(1),5)=pmsqcd
      mce=1

c...select mass for daughter in qed evolution.
      if(mstj(41).eq.2.and.kfl(1).ge.1.and.kfl(1).le.18) then
        pmsqed=pms*exp(max(-80.,log(rlu(0))*paru(2)/(paru(101)*fbre)))
        if(zce.gt.0.49.or.pmsqed.le.pmth(5,kfl(1))**2) pmsqed=
     &  pmth(2,kfl(1))**2
        if(pmsqed.gt.pmsqcd) then
          v(iep(1),5)=pmsqed
          mce=2
        endif
      endif

c...check whether daughter mass below cutoff.
      p(iep(1),5)=sqrt(v(iep(1),5))
      if(p(iep(1),5).le.pmth(3,kfl(1))) then
        p(iep(1),5)=pmth(1,kfl(1))
        v(iep(1),5)=p(iep(1),5)**2
        goto 300
      endif

c...select z value of branching: q -> qgamma.
      if(mce.eq.2) then
        z=1.-(1.-zce)*(zce/(1.-zce))**rlu(0)
        if(1.+z**2.lt.2.*rlu(0)) goto 260
        k(iep(1),5)=22

c...select z value of branching: q -> qg, g -> gg, g -> qqbar.
      elseif(mstj(49).ne.1.and.kfl(1).ne.21) then
        z=1.-(1.-zc)*(zc/(1.-zc))**rlu(0)
        if(1.+z**2.lt.2.*rlu(0)) goto 260
        k(iep(1),5)=21
      elseif(mstj(49).eq.0.and.mstj(45)*(0.5-zc).lt.rlu(0)*fbr) then
        z=(1.-zc)*(zc/(1.-zc))**rlu(0)
        if(rlu(0).gt.0.5) z=1.-z
        if((1.-z*(1.-z))**2.lt.rlu(0)) goto 260
        k(iep(1),5)=21
      elseif(mstj(49).ne.1) then
        z=zc+(1.-2.*zc)*rlu(0)
        if(z**2+(1.-z)**2.lt.rlu(0)) goto 260
        kflb=1+int(mstj(45)*rlu(0))
        pmq=4.*pmth(2,kflb)**2/v(iep(1),5)
        if(pmq.ge.1.) goto 260
        pmq0=4.*pmth(2,21)**2/v(iep(1),5)
        if(mod(mstj(43),2).eq.0.and.(1.+0.5*pmq)*sqrt(1.-pmq).lt.
     &  rlu(0)*(1.+0.5*pmq0)*sqrt(1.-pmq0)) goto 260
        k(iep(1),5)=kflb

c...ditto for scalar gluon model.
      elseif(kfl(1).ne.21) then
        z=1.-sqrt(zc**2+rlu(0)*(1.-2.*zc))
        k(iep(1),5)=21
      elseif(rlu(0)*(parj(87)+mstj(45)*parj(88)).le.parj(87)) then
        z=zc+(1.-2.*zc)*rlu(0)
        k(iep(1),5)=21
      else
        z=zc+(1.-2.*zc)*rlu(0)
        kflb=1+int(mstj(45)*rlu(0))
        pmq=4.*pmth(2,kflb)**2/v(iep(1),5)
        if(pmq.ge.1.) goto 260
        k(iep(1),5)=kflb
      endif
      if(mce.eq.1.and.mstj(44).ge.2) then
        if(z*(1.-z)*v(iep(1),5).lt.pt2min) goto 260
        if(alfm/log(v(iep(1),5)*z*(1.-z)/alams).lt.rlu(0)) goto 260
      endif

c...check if z consistent with chosen m.
      if(kfl(1).eq.21) then
        kflgd1=iabs(k(iep(1),5))
        kflgd2=kflgd1
      else
        kflgd1=kfl(1)
        kflgd2=iabs(k(iep(1),5))
      endif
      if(nep.eq.1) then
        ped=ps(4)
      elseif(nep.ge.3) then
        ped=p(iep(1),4)
      elseif(igm.eq.0.or.mstj(43).le.2) then
        ped=0.5*(v(im,5)+v(iep(1),5)-pm2**2)/p(im,5)
      else
        if(iep(1).eq.n+1) ped=v(im,1)*pem
        if(iep(1).eq.n+2) ped=(1.-v(im,1))*pem
      endif
      if(mod(mstj(43),2).eq.1) then
        pmqth3=0.5*parj(82)
        if(kflgd2.eq.22) pmqth3=0.5*parj(83)
        pmq1=(pmth(1,kflgd1)**2+pmqth3**2)/v(iep(1),5)
        pmq2=(pmth(1,kflgd2)**2+pmqth3**2)/v(iep(1),5)
        zd=sqrt(max(0.,(1.-v(iep(1),5)/ped**2)*((1.-pmq1-pmq2)**2-
     &  4.*pmq1*pmq2)))
        zh=1.+pmq1-pmq2
      else
        zd=sqrt(max(0.,1.-v(iep(1),5)/ped**2))
        zh=1.
      endif
      zl=0.5*(zh-zd)
      zu=0.5*(zh+zd)
      if(z.lt.zl.or.z.gt.zu) goto 260
      if(kfl(1).eq.21) v(iep(1),3)=log(zu*(1.-zl)/max(1e-20,zl*
     &(1.-zu)))
      if(kfl(1).ne.21) v(iep(1),3)=log((1.-zl)/max(1e-10,1.-zu))

c...three-jet matrix element correction.
      if(igm.eq.0.and.m3jc.eq.1) then
        x1=z*(1.+v(iep(1),5)/v(ns+1,5))
        x2=1.-v(iep(1),5)/v(ns+1,5)
        x3=(1.-x1)+(1.-x2)
        if(mce.eq.2) then
          ki1=k(ipa(inum),2)
          ki2=k(ipa(3-inum),2)
          qf1=kchg(iabs(ki1),1)*isign(1,ki1)/3.
          qf2=kchg(iabs(ki2),1)*isign(1,ki2)/3.
          wshow=qf1**2*(1.-x1)/x3*(1.+(x1/(2.-x2))**2)+
     &    qf2**2*(1.-x2)/x3*(1.+(x2/(2.-x1))**2)
          wme=(qf1*(1.-x1)/x3-qf2*(1.-x2)/x3)**2*(x1**2+x2**2)
        elseif(mstj(49).ne.1) then
          wshow=1.+(1.-x1)/x3*(x1/(2.-x2))**2+
     &    (1.-x2)/x3*(x2/(2.-x1))**2
          wme=x1**2+x2**2
        else
          wshow=4.*x3*((1.-x1)/(2.-x2)**2+(1.-x2)/(2.-x1)**2)
          wme=x3**2
          if(mstj(102).ge.2) wme=x3**2-2.*(1.+x3)*(1.-x1)*(1.-x2)*
     &    parj(171)
        endif
        if(wme.lt.rlu(0)*wshow) goto 260

c...impose angular ordering by rejection of nonordered emission.
      elseif(mce.eq.1.and.igm.gt.0.and.mstj(42).ge.2) then
        maom=1
        zm=v(im,1)
        if(iep(1).eq.n+2) zm=1.-v(im,1)
        the2id=z*(1.-z)*(zm*p(im,4))**2/v(iep(1),5)
        iaom=im
  290   if(k(iaom,5).eq.22) then
          iaom=k(iaom,3)
          if(k(iaom,3).le.ns) maom=0
          if(maom.eq.1) goto 290
        endif
        if(maom.eq.1) then
          the2im=v(iaom,1)*(1.-v(iaom,1))*p(iaom,4)**2/v(iaom,5)
          if(the2id.lt.the2im) goto 260
        endif
      endif

c...impose user-defined maximum angle at first branching.
      if(mstj(48).eq.1) then
        if(nep.eq.1.and.im.eq.ns) then
          the2id=z*(1.-z)*ps(4)**2/v(iep(1),5)
          if(the2id.lt.1./parj(85)**2) goto 260
        elseif(nep.eq.2.and.iep(1).eq.ns+2) then
          the2id=z*(1.-z)*(0.5*p(im,4))**2/v(iep(1),5)
          if(the2id.lt.1./parj(85)**2) goto 260
        elseif(nep.eq.2.and.iep(1).eq.ns+3) then
          the2id=z*(1.-z)*(0.5*p(im,4))**2/v(iep(1),5)
          if(the2id.lt.1./parj(86)**2) goto 260
        endif
      endif

c...end of inner veto algorithm. check if only one leg evolved so far.
  300 v(iep(1),1)=z
      isl(1)=0
      isl(2)=0
      if(nep.eq.1) goto 330
      if(nep.eq.2.and.p(iep(1),5)+p(iep(2),5).ge.p(im,5)) goto 200
      do 310 i=1,nep
      if(itry(i).eq.0.and.kfld(i).le.40) then
        if(ksh(kfld(i)).eq.1) then
          if(p(n+i,5).ge.pmth(2,kfld(i))) goto 200
        endif
      endif
  310 continue

c...check if chosen multiplet m1,m2,z1,z2 is physical.
      if(nep.eq.3) then
        pa1s=(p(n+1,4)+p(n+1,5))*(p(n+1,4)-p(n+1,5))
        pa2s=(p(n+2,4)+p(n+2,5))*(p(n+2,4)-p(n+2,5))
        pa3s=(p(n+3,4)+p(n+3,5))*(p(n+3,4)-p(n+3,5))
        pts=0.25*(2.*pa1s*pa2s+2.*pa1s*pa3s+2.*pa2s*pa3s-
     &  pa1s**2-pa2s**2-pa3s**2)/pa1s
        if(pts.le.0.) goto 200
      elseif(igm.eq.0.or.mstj(43).le.2.or.mod(mstj(43),2).eq.0) then
        do 320 i1=n+1,n+2
        kflda=iabs(k(i1,2))
        if(kflda.gt.40) goto 320
        if(ksh(kflda).eq.0) goto 320
        if(p(i1,5).lt.pmth(2,kflda)) goto 320
        if(kflda.eq.21) then
          kflgd1=iabs(k(i1,5))
          kflgd2=kflgd1
        else
          kflgd1=kflda
          kflgd2=iabs(k(i1,5))
        endif
        i2=2*n+3-i1
        if(igm.eq.0.or.mstj(43).le.2) then
          ped=0.5*(v(im,5)+v(i1,5)-v(i2,5))/p(im,5)
        else
          if(i1.eq.n+1) zm=v(im,1)
          if(i1.eq.n+2) zm=1.-v(im,1)
          pml=sqrt((v(im,5)-v(n+1,5)-v(n+2,5))**2-
     &    4.*v(n+1,5)*v(n+2,5))
          ped=pem*(0.5*(v(im,5)-pml+v(i1,5)-v(i2,5))+pml*zm)/v(im,5)
        endif
        if(mod(mstj(43),2).eq.1) then
          pmqth3=0.5*parj(82)
          if(kflgd2.eq.22) pmqth3=0.5*parj(83)
          pmq1=(pmth(1,kflgd1)**2+pmqth3**2)/v(i1,5)
          pmq2=(pmth(1,kflgd2)**2+pmqth3**2)/v(i1,5)
          zd=sqrt(max(0.,(1.-v(i1,5)/ped**2)*((1.-pmq1-pmq2)**2-
     &    4.*pmq1*pmq2)))
          zh=1.+pmq1-pmq2
        else
          zd=sqrt(max(0.,1.-v(i1,5)/ped**2))
          zh=1.
        endif
        zl=0.5*(zh-zd)
        zu=0.5*(zh+zd)
        if(i1.eq.n+1.and.(v(i1,1).lt.zl.or.v(i1,1).gt.zu)) isl(1)=1
        if(i1.eq.n+2.and.(v(i1,1).lt.zl.or.v(i1,1).gt.zu)) isl(2)=1
        if(kflda.eq.21) v(i1,4)=log(zu*(1.-zl)/max(1e-20,zl*(1.-zu)))
        if(kflda.ne.21) v(i1,4)=log((1.-zl)/max(1e-10,1.-zu))
  320   continue
        if(isl(1).eq.1.and.isl(2).eq.1.and.islm.ne.0) then
          isl(3-islm)=0
          islm=3-islm
        elseif(isl(1).eq.1.and.isl(2).eq.1) then
          zdr1=max(0.,v(n+1,3)/max(1e-6,v(n+1,4))-1.)
          zdr2=max(0.,v(n+2,3)/max(1e-6,v(n+2,4))-1.)
          if(zdr2.gt.rlu(0)*(zdr1+zdr2)) isl(1)=0
          if(isl(1).eq.1) isl(2)=0
          if(isl(1).eq.0) islm=1
          if(isl(2).eq.0) islm=2
        endif
        if(isl(1).eq.1.or.isl(2).eq.1) goto 200
      endif
      if(igm.gt.0.and.mod(mstj(43),2).eq.1.and.(p(n+1,5).ge.
     &pmth(2,kfld(1)).or.p(n+2,5).ge.pmth(2,kfld(2)))) then
        pmq1=v(n+1,5)/v(im,5)
        pmq2=v(n+2,5)/v(im,5)
        zd=sqrt(max(0.,(1.-v(im,5)/pem**2)*((1.-pmq1-pmq2)**2-
     &  4.*pmq1*pmq2)))
        zh=1.+pmq1-pmq2
        zl=0.5*(zh-zd)
        zu=0.5*(zh+zd)
        if(v(im,1).lt.zl.or.v(im,1).gt.zu) goto 200
      endif

c...accepted branch. construct four-momentum for initial partons.
  330 mazip=0
      mazic=0
      if(nep.eq.1) then
        p(n+1,1)=0.
        p(n+1,2)=0.
        p(n+1,3)=sqrt(max(0.,(p(ipa(1),4)+p(n+1,5))*(p(ipa(1),4)-
     &  p(n+1,5))))
        p(n+1,4)=p(ipa(1),4)
        v(n+1,2)=p(n+1,4)
      elseif(igm.eq.0.and.nep.eq.2) then
        ped1=0.5*(v(im,5)+v(n+1,5)-v(n+2,5))/p(im,5)
        p(n+1,1)=0.
        p(n+1,2)=0.
        p(n+1,3)=sqrt(max(0.,(ped1+p(n+1,5))*(ped1-p(n+1,5))))
        p(n+1,4)=ped1
        p(n+2,1)=0.
        p(n+2,2)=0.
        p(n+2,3)=-p(n+1,3)
        p(n+2,4)=p(im,5)-ped1
        v(n+1,2)=p(n+1,4)
        v(n+2,2)=p(n+2,4)
      elseif(nep.eq.3) then
        p(n+1,1)=0.
        p(n+1,2)=0.
        p(n+1,3)=sqrt(max(0.,pa1s))
        p(n+2,1)=sqrt(pts)
        p(n+2,2)=0.
        p(n+2,3)=0.5*(pa3s-pa2s-pa1s)/p(n+1,3)
        p(n+3,1)=-p(n+2,1)
        p(n+3,2)=0.
        p(n+3,3)=-(p(n+1,3)+p(n+2,3))
        v(n+1,2)=p(n+1,4)
        v(n+2,2)=p(n+2,4)
        v(n+3,2)=p(n+3,4)

c...construct transverse momentum for ordinary branching in shower.
      else
        zm=v(im,1)
        pzm=sqrt(max(0.,(pem+p(im,5))*(pem-p(im,5))))
        pmls=(v(im,5)-v(n+1,5)-v(n+2,5))**2-4.*v(n+1,5)*v(n+2,5)
        if(pzm.le.0.) then
          pts=0.
        elseif(mod(mstj(43),2).eq.1) then
          pts=(pem**2*(zm*(1.-zm)*v(im,5)-(1.-zm)*v(n+1,5)-
     &    zm*v(n+2,5))-0.25*pmls)/pzm**2
        else
          pts=pmls*(zm*(1.-zm)*pem**2/v(im,5)-0.25)/pzm**2
        endif
        pt=sqrt(max(0.,pts))

c...find coefficient of azimuthal asymmetry due to gluon polarization.
        hazip=0.
        if(mstj(49).ne.1.and.mod(mstj(46),2).eq.1.and.k(im,2).eq.21.
     &  and.iau.ne.0) then
          if(k(igm,3).ne.0) mazip=1
          zau=v(igm,1)
          if(iau.eq.im+1) zau=1.-v(igm,1)
          if(mazip.eq.0) zau=0.
          if(k(igm,2).ne.21) then
            hazip=2.*zau/(1.+zau**2)
          else
            hazip=(zau/(1.-zau*(1.-zau)))**2
          endif
          if(k(n+1,2).ne.21) then
            hazip=hazip*(-2.*zm*(1.-zm))/(1.-2.*zm*(1.-zm))
          else
            hazip=hazip*(zm*(1.-zm)/(1.-zm*(1.-zm)))**2
          endif
        endif

c...find coefficient of azimuthal asymmetry due to soft gluon
c...interference.
        hazic=0.
        if(mstj(49).ne.2.and.mstj(46).ge.2.and.(k(n+1,2).eq.21.or.
     &  k(n+2,2).eq.21).and.iau.ne.0) then
          if(k(igm,3).ne.0) mazic=n+1
          if(k(igm,3).ne.0.and.k(n+1,2).ne.21) mazic=n+2
          if(k(igm,3).ne.0.and.k(n+1,2).eq.21.and.k(n+2,2).eq.21.and.
     &    zm.gt.0.5) mazic=n+2
          if(k(iau,2).eq.22) mazic=0
          zs=zm
          if(mazic.eq.n+2) zs=1.-zm
          zgm=v(igm,1)
          if(iau.eq.im-1) zgm=1.-v(igm,1)
          if(mazic.eq.0) zgm=1.
          hazic=(p(im,5)/p(igm,5))*sqrt((1.-zs)*(1.-zgm)/(zs*zgm))
          hazic=min(0.95,hazic)
        endif
      endif

c...construct kinematics for ordinary branching in shower.
  340 if(nep.eq.2.and.igm.gt.0) then
        if(mod(mstj(43),2).eq.1) then
          p(n+1,4)=pem*v(im,1)
        else
          p(n+1,4)=pem*(0.5*(v(im,5)-sqrt(pmls)+v(n+1,5)-v(n+2,5))+
     &    sqrt(pmls)*zm)/v(im,5)
        endif
        phi=paru(2)*rlu(0)
        p(n+1,1)=pt*cos(phi)
        p(n+1,2)=pt*sin(phi)
        if(pzm.gt.0.) then
          p(n+1,3)=0.5*(v(n+2,5)-v(n+1,5)-v(im,5)+2.*pem*p(n+1,4))/pzm
        else
          p(n+1,3)=0.
        endif
        p(n+2,1)=-p(n+1,1)
        p(n+2,2)=-p(n+1,2)
        p(n+2,3)=pzm-p(n+1,3)
        p(n+2,4)=pem-p(n+1,4)
        if(mstj(43).le.2) then
          v(n+1,2)=(pem*p(n+1,4)-pzm*p(n+1,3))/p(im,5)
          v(n+2,2)=(pem*p(n+2,4)-pzm*p(n+2,3))/p(im,5)
        endif
      endif

c...rotate and boost daughters.
      if(igm.gt.0) then
        if(mstj(43).le.2) then
          bex=p(igm,1)/p(igm,4)
          bey=p(igm,2)/p(igm,4)
          bez=p(igm,3)/p(igm,4)
          ga=p(igm,4)/p(igm,5)
          gabep=ga*(ga*(bex*p(im,1)+bey*p(im,2)+bez*p(im,3))/(1.+ga)-
     &    p(im,4))
        else
          bex=0.
          bey=0.
          bez=0.
          ga=1.
          gabep=0.
        endif
        the=ulangl(p(im,3)+gabep*bez,sqrt((p(im,1)+gabep*bex)**2+
     &  (p(im,2)+gabep*bey)**2))
        phi=ulangl(p(im,1)+gabep*bex,p(im,2)+gabep*bey)
        do 350 i=n+1,n+2
        dp(1)=cos(the)*cos(phi)*p(i,1)-sin(phi)*p(i,2)+
     &  sin(the)*cos(phi)*p(i,3)
        dp(2)=cos(the)*sin(phi)*p(i,1)+cos(phi)*p(i,2)+
     &  sin(the)*sin(phi)*p(i,3)
        dp(3)=-sin(the)*p(i,1)+cos(the)*p(i,3)
        dp(4)=p(i,4)
        dbp=bex*dp(1)+bey*dp(2)+bez*dp(3)
        dgabp=ga*(ga*dbp/(1d0+ga)+dp(4))
        p(i,1)=dp(1)+dgabp*bex
        p(i,2)=dp(2)+dgabp*bey
        p(i,3)=dp(3)+dgabp*bez
  350   p(i,4)=ga*(dp(4)+dbp)
      endif

c...weight with azimuthal distribution, if required.
      if(mazip.ne.0.or.mazic.ne.0) then
        do 360 j=1,3
        dpt(1,j)=p(im,j)
        dpt(2,j)=p(iau,j)
  360   dpt(3,j)=p(n+1,j)
        dpma=dpt(1,1)*dpt(2,1)+dpt(1,2)*dpt(2,2)+dpt(1,3)*dpt(2,3)
        dpmd=dpt(1,1)*dpt(3,1)+dpt(1,2)*dpt(3,2)+dpt(1,3)*dpt(3,3)
        dpmm=dpt(1,1)**2+dpt(1,2)**2+dpt(1,3)**2
        do 370 j=1,3
        dpt(4,j)=dpt(2,j)-dpma*dpt(1,j)/dpmm
  370   dpt(5,j)=dpt(3,j)-dpmd*dpt(1,j)/dpmm
        dpt(4,4)=sqrt(dpt(4,1)**2+dpt(4,2)**2+dpt(4,3)**2)
        dpt(5,4)=sqrt(dpt(5,1)**2+dpt(5,2)**2+dpt(5,3)**2)
        if(min(dpt(4,4),dpt(5,4)).gt.0.1*parj(82)) then
          cad=(dpt(4,1)*dpt(5,1)+dpt(4,2)*dpt(5,2)+
     &    dpt(4,3)*dpt(5,3))/(dpt(4,4)*dpt(5,4))
          if(mazip.ne.0) then
            if(1.+hazip*(2.*cad**2-1.).lt.rlu(0)*(1.+abs(hazip)))
     &      goto 340
          endif
          if(mazic.ne.0) then
            if(mazic.eq.n+2) cad=-cad
            if((1.-hazic)*(1.-hazic*cad)/(1.+hazic**2-2.*hazic*cad).
     &      lt.rlu(0)) goto 340
          endif
        endif
      endif

c...continue loop over partons that may branch, until none left.
      if(igm.ge.0) k(im,1)=14
      n=n+nep
      nep=2
      if(n.gt.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lushow:) no more memory left in lujets')
        if(mstu(21).ge.1) n=ns
        if(mstu(21).ge.1) return
      endif
      goto 140

c...set information on imagined shower initiator.
  380 if(npa.ge.2) then
        k(ns+1,1)=11
        k(ns+1,2)=94
        k(ns+1,3)=ip1
        if(ip2.gt.0.and.ip2.lt.ip1) k(ns+1,3)=ip2
        k(ns+1,4)=ns+2
        k(ns+1,5)=ns+1+npa
        iim=1
      else
        iim=0
      endif

c...reconstruct string drawing information.
      do 390 i=ns+1+iim,n
      if(k(i,1).le.10.and.k(i,2).eq.22) then
        k(i,1)=1
      elseif(k(i,1).le.10.and.iabs(k(i,2)).ge.11.and.
     &iabs(k(i,2)).le.18) then
        k(i,1)=1
      elseif(k(i,1).le.10) then
        k(i,4)=mstu(5)*(k(i,4)/mstu(5))
        k(i,5)=mstu(5)*(k(i,5)/mstu(5))
      elseif(k(mod(k(i,4),mstu(5))+1,2).ne.22) then
        id1=mod(k(i,4),mstu(5))
        if(k(i,2).ge.1.and.k(i,2).le.8) id1=mod(k(i,4),mstu(5))+1
        id2=2*mod(k(i,4),mstu(5))+1-id1
        k(i,4)=mstu(5)*(k(i,4)/mstu(5))+id1
        k(i,5)=mstu(5)*(k(i,5)/mstu(5))+id2
        k(id1,4)=k(id1,4)+mstu(5)*i
        k(id1,5)=k(id1,5)+mstu(5)*id2
        k(id2,4)=k(id2,4)+mstu(5)*id1
        k(id2,5)=k(id2,5)+mstu(5)*i
      else
        id1=mod(k(i,4),mstu(5))
        id2=id1+1
        k(i,4)=mstu(5)*(k(i,4)/mstu(5))+id1
        k(i,5)=mstu(5)*(k(i,5)/mstu(5))+id1
        k(id1,4)=k(id1,4)+mstu(5)*i
        k(id1,5)=k(id1,5)+mstu(5)*i
        k(id2,4)=0
        k(id2,5)=0
      endif
  390 continue

c...transformation from cm frame.
      if(npa.ge.2) then
        bex=ps(1)/ps(4)
        bey=ps(2)/ps(4)
        bez=ps(3)/ps(4)
        ga=ps(4)/ps(5)
        gabep=ga*(ga*(bex*p(ipa(1),1)+bey*p(ipa(1),2)+bez*p(ipa(1),3))
     &  /(1.+ga)-p(ipa(1),4))
      else
        bex=0.
        bey=0.
        bez=0.
        gabep=0.
      endif
      the=ulangl(p(ipa(1),3)+gabep*bez,sqrt((p(ipa(1),1)
     &+gabep*bex)**2+(p(ipa(1),2)+gabep*bey)**2))
      phi=ulangl(p(ipa(1),1)+gabep*bex,p(ipa(1),2)+gabep*bey)
      if(npa.eq.3) then
        chi=ulangl(cos(the)*cos(phi)*(p(ipa(2),1)+gabep*bex)+cos(the)*
     &  sin(phi)*(p(ipa(2),2)+gabep*bey)-sin(the)*(p(ipa(2),3)+gabep*
     &  bez),-sin(phi)*(p(ipa(2),1)+gabep*bex)+cos(phi)*(p(ipa(2),2)+
     &  gabep*bey))
        mstu(33)=1
        call ludbrb(ns+1,n,0.,chi,0d0,0d0,0d0)
      endif
      dbex=dble(bex)
      dbey=dble(bey)
      dbez=dble(bez)
      mstu(33)=1
      call ludbrb(ns+1,n,the,phi,dbex,dbey,dbez)

c...decay vertex of shower.
      do 400 i=ns+1,n
      do 400 j=1,5
  400 v(i,j)=v(ip1,j)

c...delete trivial shower, else connect initiators.
      if(n.eq.ns+npa+iim) then
        n=ns
      else
        do 410 ip=1,npa
        k(ipa(ip),1)=14
        k(ipa(ip),4)=k(ipa(ip),4)+ns+iim+ip
        k(ipa(ip),5)=k(ipa(ip),5)+ns+iim+ip
        k(ns+iim+ip,3)=ipa(ip)
        if(iim.eq.1.and.mstu(16).ne.2) k(ns+iim+ip,3)=ns+1
        k(ns+iim+ip,4)=mstu(5)*ipa(ip)+k(ns+iim+ip,4)
  410   k(ns+iim+ip,5)=mstu(5)*ipa(ip)+k(ns+iim+ip,5)
      endif

      return
      end

c*********************************************************************

      subroutine luboei(nsav)

c...purpose: to modify event so as to approximately take into account
c...bose-einstein effects according to a simple phenomenological
c...parametrization.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /lujets/,/ludat1/
      dimension dps(4),kfbe(9),nbe(0:9),bei(100)
      data kfbe/211,-211,111,321,-321,130,310,221,331/

c...boost event to overall cm frame. calculate cm energy.
      if((mstj(51).ne.1.and.mstj(51).ne.2).or.n-nsav.le.1) return
      do 100 j=1,4
  100 dps(j)=0.
      do 120 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 120
      do 110 j=1,4
  110 dps(j)=dps(j)+p(i,j)
  120 continue
      call ludbrb(0,0,0.,0.,-dps(1)/dps(4),-dps(2)/dps(4),
     &-dps(3)/dps(4))
      pecm=0.
      do 130 i=1,n
  130 if(k(i,1).ge.1.and.k(i,1).le.10) pecm=pecm+p(i,4)

c...reserve copy of particles by species at end of record.
      nbe(0)=n+mstu(3)
      do 160 ibe=1,min(9,mstj(52))
      nbe(ibe)=nbe(ibe-1)
      do 150 i=nsav+1,n
      if(k(i,2).ne.kfbe(ibe)) goto 150
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 150
      if(nbe(ibe).ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(luboei:) no more memory left in lujets')
        return
      endif
      nbe(ibe)=nbe(ibe)+1
      k(nbe(ibe),1)=i
      do 140 j=1,3
  140 p(nbe(ibe),j)=0.
  150 continue
  160 continue

c...tabulate integral for subsequent momentum shift.
      do 210 ibe=1,min(9,mstj(52))
      if(ibe.ne.1.and.ibe.ne.4.and.ibe.le.7) goto 180
      if(ibe.eq.1.and.max(nbe(1)-nbe(0),nbe(2)-nbe(1),nbe(3)-nbe(2)).
     &le.1) goto 180
      if(ibe.eq.4.and.max(nbe(4)-nbe(3),nbe(5)-nbe(4),nbe(6)-nbe(5),
     &nbe(7)-nbe(6)).le.1) goto 180
      if(ibe.ge.8.and.nbe(ibe)-nbe(ibe-1).le.1) goto 180
      if(ibe.eq.1) pmhq=2.*ulmass(211)
      if(ibe.eq.4) pmhq=2.*ulmass(321)
      if(ibe.eq.8) pmhq=2.*ulmass(221)
      if(ibe.eq.9) pmhq=2.*ulmass(331)
      qdel=0.1*min(pmhq,parj(93))
      if(mstj(51).eq.1) then
        nbin=min(100,nint(9.*parj(93)/qdel))
        beex=exp(0.5*qdel/parj(93))
        bert=exp(-qdel/parj(93))
      else
        nbin=min(100,nint(3.*parj(93)/qdel))
      endif
      do 170 ibin=1,nbin
      qbin=qdel*(ibin-0.5)
      bei(ibin)=qdel*(qbin**2+qdel**2/12.)/sqrt(qbin**2+pmhq**2)
      if(mstj(51).eq.1) then
        beex=beex*bert
        bei(ibin)=bei(ibin)*beex
      else
        bei(ibin)=bei(ibin)*exp(-(qbin/parj(93))**2)
      endif
  170 if(ibin.ge.2) bei(ibin)=bei(ibin)+bei(ibin-1)

c...loop through particle pairs and find old relative momentum.
  180 do 200 i1m=nbe(ibe-1)+1,nbe(ibe)-1
      i1=k(i1m,1)
      do 200 i2m=i1m+1,nbe(ibe)
      i2=k(i2m,1)
      q2old=max(0.,(p(i1,4)+p(i2,4))**2-(p(i1,1)+p(i2,1))**2-(p(i1,2)+
     &p(i2,2))**2-(p(i1,3)+p(i2,3))**2-(p(i1,5)+p(i2,5))**2)
      qold=sqrt(q2old)

c...calculate new relative momentum.
      if(qold.lt.0.5*qdel) then
        qmov=qold/3.
      elseif(qold.lt.(nbin-0.1)*qdel) then
        rbin=qold/qdel
        ibin=rbin
        rinp=(rbin**3-ibin**3)/(3*ibin*(ibin+1)+1)
        qmov=(bei(ibin)+rinp*(bei(ibin+1)-bei(ibin)))*
     &  sqrt(q2old+pmhq**2)/q2old
      else
        qmov=bei(nbin)*sqrt(q2old+pmhq**2)/q2old
      endif
      q2new=q2old*(qold/(qold+3.*parj(92)*qmov))**(2./3.)

c...calculate and save shift to be performed on three-momenta.
      hc1=(p(i1,4)+p(i2,4))**2-(q2old-q2new)
      hc2=(q2old-q2new)*(p(i1,4)-p(i2,4))**2
      ha=0.5*(1.-sqrt(hc1*q2new/(hc1*q2old-hc2)))
      do 190 j=1,3
      pd=ha*(p(i2,j)-p(i1,j))
      p(i1m,j)=p(i1m,j)+pd
  190 p(i2m,j)=p(i2m,j)-pd
  200 continue
  210 continue

c...shift momenta and recalculate energies.
      do 230 im=nbe(0)+1,nbe(min(9,mstj(52)))
      i=k(im,1)
      do 220 j=1,3
  220 p(i,j)=p(i,j)+p(im,j)
  230 p(i,4)=sqrt(p(i,5)**2+p(i,1)**2+p(i,2)**2+p(i,3)**2)

c...rescale all momenta for energy conservation.
      pes=0.
      pqs=0.
      do 240 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 240
      pes=pes+p(i,4)
      pqs=pqs+p(i,5)**2/p(i,4)
  240 continue
      fac=(pecm-pqs)/(pes-pqs)
      do 260 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 260
      do 250 j=1,3
  250 p(i,j)=fac*p(i,j)
      p(i,4)=sqrt(p(i,5)**2+p(i,1)**2+p(i,2)**2+p(i,3)**2)
  260 continue

c...boost back to correct reference frame.
      call ludbrb(0,0,0.,0.,dps(1)/dps(4),dps(2)/dps(4),dps(3)/dps(4))

      return
      end

c*********************************************************************

      function ulmass(kf)

c...purpose: to give the mass of a particle/parton.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat1/,/ludat2/

c...reset variables. compressed code.
      ulmass=0.
      kfa=iabs(kf)
      kc=lucomp(kf)
      if(kc.eq.0) return
      parf(106)=pmas(6,1)
      parf(107)=pmas(7,1)
      parf(108)=pmas(8,1)

c...guarantee use of constituent masses for internal checks.
      if((mstj(93).eq.1.or.mstj(93).eq.2).and.kfa.le.10) then
        ulmass=parf(100+kfa)
        if(mstj(93).eq.2) ulmass=max(0.,ulmass-parf(121))

c...masses that can be read directly off table.
      elseif(kfa.le.100.or.kc.le.80.or.kc.gt.100) then
        ulmass=pmas(kc,1)

c...find constituent partons and their masses.
      else
        kfla=mod(kfa/1000,10)
        kflb=mod(kfa/100,10)
        kflc=mod(kfa/10,10)
        kfls=mod(kfa,10)
        kflr=mod(kfa/10000,10)
        pma=parf(100+kfla)
        pmb=parf(100+kflb)
        pmc=parf(100+kflc)

c...construct masses for various meson, diquark and baryon cases.
        if(kfla.eq.0.and.kflr.eq.0.and.kfls.le.3) then
          if(kfls.eq.1) pmspl=-3./(pmb*pmc)
          if(kfls.ge.3) pmspl=1./(pmb*pmc)
          ulmass=parf(111)+pmb+pmc+parf(113)*parf(101)**2*pmspl
        elseif(kfla.eq.0) then
          kmul=2
          if(kfls.eq.1) kmul=3
          if(kflr.eq.2) kmul=4
          if(kfls.eq.5) kmul=5
          ulmass=parf(113+kmul)+pmb+pmc
        elseif(kflc.eq.0) then
          if(kfls.eq.1) pmspl=-3./(pma*pmb)
          if(kfls.eq.3) pmspl=1./(pma*pmb)
          ulmass=2.*parf(112)/3.+pma+pmb+parf(114)*parf(101)**2*pmspl
          if(mstj(93).eq.1) ulmass=pma+pmb
          if(mstj(93).eq.2) ulmass=max(0.,ulmass-parf(122)-
     &    2.*parf(112)/3.)
        else
          if(kfls.eq.2.and.kfla.eq.kflb) then
            pmspl=1./(pma*pmb)-2./(pma*pmc)-2./(pmb*pmc)
          elseif(kfls.eq.2.and.kflb.ge.kflc) then
            pmspl=-2./(pma*pmb)-2./(pma*pmc)+1./(pmb*pmc)
          elseif(kfls.eq.2) then
            pmspl=-3./(pmb*pmc)
          else
            pmspl=1./(pma*pmb)+1./(pma*pmc)+1./(pmb*pmc)
          endif
          ulmass=parf(112)+pma+pmb+pmc+parf(114)*parf(101)**2*pmspl
        endif
      endif

c...optional mass broadening according to truncated breit-wigner
c...(either in m or in m^2).
      if(mstj(24).ge.1.and.pmas(kc,2).gt.1e-4) then
        if(mstj(24).eq.1.or.(mstj(24).eq.2.and.kfa.gt.100)) then
          ulmass=ulmass+0.5*pmas(kc,2)*tan((2.*rlu(0)-1.)*
     &    atan(2.*pmas(kc,3)/pmas(kc,2)))
        else
          pm0=ulmass
          pmlow=atan((max(0.,pm0-pmas(kc,3))**2-pm0**2)/
     &    (pm0*pmas(kc,2)))
          pmupp=atan(((pm0+pmas(kc,3))**2-pm0**2)/(pm0*pmas(kc,2)))
          ulmass=sqrt(max(0.,pm0**2+pm0*pmas(kc,2)*tan(pmlow+
     &    (pmupp-pmlow)*rlu(0))))
        endif
      endif
      mstj(93)=0

      return
      end

c*********************************************************************

      subroutine luname(kf,chau)

c...purpose: to give the particle/parton name as a character string.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat4/chaf(500)
      character chaf*8
      save /ludat1/,/ludat2/,/ludat4/
      character chau*16

c...initial values. charge. subdivide code.
      chau=' '
      kfa=iabs(kf)
      kc=lucomp(kf)
      if(kc.eq.0) return
      kq=luchge(kf)
      kfla=mod(kfa/1000,10)
      kflb=mod(kfa/100,10)
      kflc=mod(kfa/10,10)
      kfls=mod(kfa,10)
      kflr=mod(kfa/10000,10)

c...read out root name and spin for simple particle.
      if(kfa.le.100.or.(kfa.gt.100.and.kc.gt.100)) then
        chau=chaf(kc)
        len=0
        do 100 lem=1,8
  100   if(chau(lem:lem).ne.' ') len=lem

c...construct root name for diquark. add on spin.
      elseif(kflc.eq.0) then
        chau(1:2)=chaf(kfla)(1:1)//chaf(kflb)(1:1)
        if(kfls.eq.1) chau(3:4)='_0'
        if(kfls.eq.3) chau(3:4)='_1'
        len=4

c...construct root name for heavy meson. add on spin and heavy flavour.
      elseif(kfla.eq.0) then
        if(kflb.eq.5) chau(1:1)='b'
        if(kflb.eq.6) chau(1:1)='t'
        if(kflb.eq.7) chau(1:1)='l'
        if(kflb.eq.8) chau(1:1)='h'
        len=1
        if(kflr.eq.0.and.kfls.eq.1) then
        elseif(kflr.eq.0.and.kfls.eq.3) then
          chau(2:2)='*'
          len=2
        elseif(kflr.eq.1.and.kfls.eq.3) then
          chau(2:3)='_1'
          len=3
        elseif(kflr.eq.1.and.kfls.eq.1) then
          chau(2:4)='*_0'
          len=4
        elseif(kflr.eq.2) then
          chau(2:4)='*_1'
          len=4
        elseif(kfls.eq.5) then
          chau(2:4)='*_2'
          len=4
        endif
        if(kflc.ge.3.and.kflr.eq.0.and.kfls.le.3) then
          chau(len+1:len+2)='_'//chaf(kflc)(1:1)
          len=len+2
        elseif(kflc.ge.3) then
          chau(len+1:len+1)=chaf(kflc)(1:1)
          len=len+1
        endif

c...construct root name and spin for heavy baryon.
      else
        if(kflb.le.2.and.kflc.le.2) then
          chau='sigma '
          if(kflc.gt.kflb) chau='lambda'
          if(kfls.eq.4) chau='sigma*'
          len=5
          if(chau(6:6).ne.' ') len=6
        elseif(kflb.le.2.or.kflc.le.2) then
          chau='xi '
          if(kfla.gt.kflb.and.kflb.gt.kflc) chau='xi'''
          if(kfls.eq.4) chau='xi*'
          len=2
          if(chau(3:3).ne.' ') len=3
        else
          chau='omega '
          if(kfla.gt.kflb.and.kflb.gt.kflc) chau='omega'''
          if(kfls.eq.4) chau='omega*'
          len=5
          if(chau(6:6).ne.' ') len=6
        endif

c...add on heavy flavour content for heavy baryon.
        chau(len+1:len+2)='_'//chaf(kfla)(1:1)
        len=len+2
        if(kflb.ge.kflc.and.kflc.ge.4) then
          chau(len+1:len+2)=chaf(kflb)(1:1)//chaf(kflc)(1:1)
          len=len+2
        elseif(kflb.ge.kflc.and.kflb.ge.4) then
          chau(len+1:len+1)=chaf(kflb)(1:1)
          len=len+1
        elseif(kflc.gt.kflb.and.kflb.ge.4) then
          chau(len+1:len+2)=chaf(kflc)(1:1)//chaf(kflb)(1:1)
          len=len+2
        elseif(kflc.gt.kflb.and.kflc.ge.4) then
          chau(len+1:len+1)=chaf(kflc)(1:1)
          len=len+1
        endif
      endif

c...add on bar sign for antiparticle (where necessary).
      if(kf.gt.0.or.len.eq.0) then
      elseif(kfa.gt.10.and.kfa.le.40.and.kq.ne.0.and.mod(kq,3).eq.0)
     &then
      elseif(kfa.eq.89.or.(kfa.ge.91.and.kfa.le.99)) then
      elseif(kfa.gt.100.and.kfla.eq.0.and.kq.ne.0) then
      elseif(mstu(15).le.1) then
        chau(len+1:len+1)='~'
        len=len+1
      else
        chau(len+1:len+3)='bar'
        len=len+3
      endif

c...add on charge where applicable (conventional cases skipped).
      if(kq.eq.6) chau(len+1:len+2)='++'
      if(kq.eq.-6) chau(len+1:len+2)='--'
      if(kq.eq.3) chau(len+1:len+1)='+'
      if(kq.eq.-3) chau(len+1:len+1)='-'
      if(kq.eq.0.and.(kfa.le.22.or.len.eq.0)) then
      elseif(kq.eq.0.and.(kfa.ge.81.and.kfa.le.100)) then
      elseif(kfa.gt.100.and.kfla.eq.0.and.kflb.eq.kflc.and.
     &kflb.ne.1) then
      elseif(kq.eq.0) then
        chau(len+1:len+1)='0'
      endif

      return
      end

c*********************************************************************

      function luchge(kf)

c...purpose: to give three times the charge for a particle/parton.
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat2/

c...initial values. simple case of direct readout.
      luchge=0
      kfa=iabs(kf)
      kc=lucomp(kfa)
      if(kc.eq.0) then
      elseif(kfa.le.100.or.kc.le.80.or.kc.gt.100) then
        luchge=kchg(kc,1)

c...construction from quark content for heavy meson, diquark, baryon.
      elseif(mod(kfa/1000,10).eq.0) then
        luchge=(kchg(mod(kfa/100,10),1)-kchg(mod(kfa/10,10),1))*
     &  (-1)**mod(kfa/100,10)
      elseif(mod(kfa/10,10).eq.0) then
        luchge=kchg(mod(kfa/1000,10),1)+kchg(mod(kfa/100,10),1)
      else
        luchge=kchg(mod(kfa/1000,10),1)+kchg(mod(kfa/100,10),1)+
     &  kchg(mod(kfa/10,10),1)
      endif

c...add on correct sign.
      luchge=luchge*isign(1,kf)

      return
      end

c*********************************************************************

      function lucomp(kf)

c...purpose: to compress the standard kf codes for use in mass and decay
c...arrays; also to check whether a given code actually is defined.
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat2/

c...subdivide kf code into constituent pieces.
      lucomp=0
      kfa=iabs(kf)
      kfla=mod(kfa/1000,10)
      kflb=mod(kfa/100,10)
      kflc=mod(kfa/10,10)
      kfls=mod(kfa,10)
      kflr=mod(kfa/10000,10)

c...simple cases: direct translation or special codes.
      if(kfa.eq.0.or.kfa.ge.100000) then
      elseif(kfa.le.100) then
        lucomp=kfa
        if(kf.lt.0.and.kchg(kfa,3).eq.0) lucomp=0
      elseif(kfls.eq.0) then
        if(kf.eq.130) lucomp=221
        if(kf.eq.310) lucomp=222
        if(kfa.eq.210) lucomp=281
        if(kfa.eq.2110) lucomp=282
        if(kfa.eq.2210) lucomp=283

c...mesons.
      elseif(kfa-10000*kflr.lt.1000) then
        if(kflb.eq.0.or.kflb.eq.9.or.kflc.eq.0.or.kflc.eq.9) then
        elseif(kflb.lt.kflc) then
        elseif(kf.lt.0.and.kflb.eq.kflc) then
        elseif(kflb.eq.kflc) then
          if(kflr.eq.0.and.kfls.eq.1) then
            lucomp=110+kflb
          elseif(kflr.eq.0.and.kfls.eq.3) then
            lucomp=130+kflb
          elseif(kflr.eq.1.and.kfls.eq.3) then
            lucomp=150+kflb
          elseif(kflr.eq.1.and.kfls.eq.1) then
            lucomp=170+kflb
          elseif(kflr.eq.2.and.kfls.eq.3) then
            lucomp=190+kflb
          elseif(kflr.eq.0.and.kfls.eq.5) then
            lucomp=210+kflb
          endif
        elseif(kflb.le.5.and.kflc.le.3) then
          if(kflr.eq.0.and.kfls.eq.1) then
            lucomp=100+((kflb-1)*(kflb-2))/2+kflc
          elseif(kflr.eq.0.and.kfls.eq.3) then
            lucomp=120+((kflb-1)*(kflb-2))/2+kflc
          elseif(kflr.eq.1.and.kfls.eq.3) then
            lucomp=140+((kflb-1)*(kflb-2))/2+kflc
          elseif(kflr.eq.1.and.kfls.eq.1) then
            lucomp=160+((kflb-1)*(kflb-2))/2+kflc
          elseif(kflr.eq.2.and.kfls.eq.3) then
            lucomp=180+((kflb-1)*(kflb-2))/2+kflc
          elseif(kflr.eq.0.and.kfls.eq.5) then
            lucomp=200+((kflb-1)*(kflb-2))/2+kflc
          endif
        elseif((kfls.eq.1.and.kflr.le.1).or.(kfls.eq.3.and.kflr.le.2).
     &  or.(kfls.eq.5.and.kflr.eq.0)) then
          lucomp=80+kflb
        endif

c...diquarks.
      elseif((kflr.eq.0.or.kflr.eq.1).and.kflc.eq.0) then
        if(kfls.ne.1.and.kfls.ne.3) then
        elseif(kfla.eq.9.or.kflb.eq.0.or.kflb.eq.9) then
        elseif(kfla.lt.kflb) then
        elseif(kfls.eq.1.and.kfla.eq.kflb) then
        else
          lucomp=90
        endif

c...spin 1/2 baryons.
      elseif(kflr.eq.0.and.kfls.eq.2) then
        if(kfla.eq.9.or.kflb.eq.0.or.kflb.eq.9.or.kflc.eq.9) then
        elseif(kfla.le.kflc.or.kfla.lt.kflb) then
        elseif(kfla.ge.6.or.kflb.ge.4.or.kflc.ge.4) then
          lucomp=80+kfla
        elseif(kflb.lt.kflc) then
          lucomp=300+((kfla+1)*kfla*(kfla-1))/6+(kflc*(kflc-1))/2+kflb
        else
          lucomp=330+((kfla+1)*kfla*(kfla-1))/6+(kflb*(kflb-1))/2+kflc
        endif

c...spin 3/2 baryons.
      elseif(kflr.eq.0.and.kfls.eq.4) then
        if(kfla.eq.9.or.kflb.eq.0.or.kflb.eq.9.or.kflc.eq.9) then
        elseif(kfla.lt.kflb.or.kflb.lt.kflc) then
        elseif(kfla.ge.6.or.kflb.ge.4) then
          lucomp=80+kfla
        else
          lucomp=360+((kfla+1)*kfla*(kfla-1))/6+(kflb*(kflb-1))/2+kflc
        endif
      endif

      return
      end

c*********************************************************************

      subroutine luerrm(merr,chmess)

c...purpose: to inform user of errors in program execution.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /lujets/,/ludat1/
      character chmess*(*)

c...write first few warnings, then be silent.
      if(merr.le.10) then
        mstu(27)=mstu(27)+1
        mstu(28)=merr
        if(mstu(25).eq.1.and.mstu(27).le.mstu(26)) write(mstu(11),5000)
     &  merr,mstu(31),chmess

c...write first few errors, then be silent or stop program.
      elseif(merr.le.20) then
        mstu(23)=mstu(23)+1
        mstu(24)=merr-10
        if(mstu(21).ge.1.and.mstu(23).le.mstu(22)) write(mstu(11),5100)
     &  merr-10,mstu(31),chmess
        if(mstu(21).ge.2.and.mstu(23).gt.mstu(22)) then
          write(mstu(11),5100) merr-10,mstu(31),chmess
          write(mstu(11),5200)
          if(merr.ne.17) call lulist(2)
          stop
        endif

c...stop program in case of irreparable error.
      else
        write(mstu(11),5300) merr-20,mstu(31),chmess
        stop
      endif

c...formats for output.
 5000 format(/5x,'advisory warning type',i2,' given after',i6,
     &' luexec calls:'/5x,a)
 5100 format(/5x,'error type',i2,' has occured after',i6,
     &' luexec calls:'/5x,a)
 5200 format(5x,'execution will be stopped after listing of last ',
     &'event!')
 5300 format(/5x,'fatal error type',i2,' has occured after',i6,
     &' luexec calls:'/5x,a/5x,'execution will now be stopped!')

      return
      end

c*********************************************************************

      function ulalem(q2)

c...purpose: to calculate the running alpha_electromagnetic.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/

c...calculate real part of photon vacuum polarization.
c...for leptons simplify by using asymptotic (q^2 >> m^2) expressions.
c...for hadrons use parametrization of h. burkhardt et al.
c...see r. kleiss et al, cern 89-08, vol. 3, pp. 129-131.
      aempi=paru(101)/(3.*paru(1))
      if(mstu(101).le.0.or.q2.lt.2e-6) then
        rpigg=0.
      elseif(q2.lt.0.09) then
        rpigg=aempi*(13.4916+log(q2))+0.00835*log(1.+q2)
      elseif(q2.lt.9.) then
        rpigg=aempi*(16.3200+2.*log(q2))+0.00238*log(1.+3.927*q2)
      elseif(q2.lt.1e4) then
        rpigg=aempi*(13.4955+3.*log(q2))+0.00165+0.00299*log(1.+q2)
      else
        rpigg=aempi*(13.4955+3.*log(q2))+0.00221+0.00293*log(1.+q2)
      endif

c...calculate running alpha_em.
      ulalem=paru(101)/(1.-rpigg)
      paru(108)=ulalem

      return
      end

c*********************************************************************

      function ulalps(q2)

c...purpose: to give the value of alpha_strong.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat1/,/ludat2/

c...constant alpha_strong trivial.
      if(mstu(111).le.0) then
        ulalps=paru(111)
        mstu(118)=mstu(112)
        paru(117)=0.
        paru(118)=paru(111)
        return
      endif

c...find effective q2, number of flavours and lambda.
      q2eff=q2
      if(mstu(115).ge.2) q2eff=max(q2,paru(114))
      nf=mstu(112)
      alam2=paru(112)**2
  100 if(nf.gt.max(2,mstu(113))) then
        q2thr=paru(113)*pmas(nf,1)**2
        if(q2eff.lt.q2thr) then
          nf=nf-1
          alam2=alam2*(q2thr/alam2)**(2./(33.-2.*nf))
          goto 100
        endif
      endif
  110 if(nf.lt.min(8,mstu(114))) then
        q2thr=paru(113)*pmas(nf+1,1)**2
        if(q2eff.gt.q2thr) then
          nf=nf+1
          alam2=alam2*(alam2/q2thr)**(2./(33.-2.*nf))
          goto 110
        endif
      endif
      if(mstu(115).eq.1) q2eff=q2eff+alam2
      paru(117)=sqrt(alam2)

c...evaluate first or second order alpha_strong.
      b0=(33.-2.*nf)/6.
      algq=log(max(1.0001,q2eff/alam2))
      if(mstu(111).eq.1) then
        ulalps=min(paru(115),paru(2)/(b0*algq))
      else
        b1=(153.-19.*nf)/6.
        ulalps=min(paru(115),paru(2)/(b0*algq)*(1.-b1*log(algq)/
     &  (b0**2*algq)))
      endif
      mstu(118)=nf
      paru(118)=ulalps

      return
      end

c*********************************************************************

      function ulangl(x,y)

c...purpose: to reconstruct an angle from given x and y coordinates.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/

      ulangl=0.
      r=sqrt(x**2+y**2)
      if(r.lt.1e-20) return
      if(abs(x)/r.lt.0.8) then
        ulangl=sign(acos(x/r),y)
      else
        ulangl=asin(y/r)
        if(x.lt.0..and.ulangl.ge.0.) then
          ulangl=paru(1)-ulangl
        elseif(x.lt.0.) then
          ulangl=-paru(1)-ulangl
        endif
      endif

      return
      end

c*********************************************************************

      function rlu(idum)

c...purpose: to generate random numbers uniformly distributed between
c...0 and 1, excluding the endpoints.
      common/ludatr/mrlu(6),rrlu(100)
      save /ludatr/
      equivalence (mrlu1,mrlu(1)),(mrlu2,mrlu(2)),(mrlu3,mrlu(3)),
     &(mrlu4,mrlu(4)),(mrlu5,mrlu(5)),(mrlu6,mrlu(6)),
     &(rrlu98,rrlu(98)),(rrlu99,rrlu(99)),(rrlu00,rrlu(100))
      real*8 u
      call rndc(u)
      rlu = u
      end

c*********************************************************************

      subroutine rluget(lfn,move)

c...purpose: to dump the state of the random number generator on a file
c...for subsequent startup from this state onwards.
      common/ludatr/mrlu(6),rrlu(100)
      save /ludatr/
      character cherr*8

c...backspace required number of records (or as many as there are).
      if(move.lt.0) then
        nbck=min(mrlu(6),-move)
        do 100 ibck=1,nbck
  100   backspace(lfn,err=110,iostat=ierr)
        mrlu(6)=mrlu(6)-nbck
      endif

c...unformatted write on unit lfn.
      write(lfn,err=110,iostat=ierr) (mrlu(i1),i1=1,5),
     &(rrlu(i2),i2=1,100)
      mrlu(6)=mrlu(6)+1
      return

c...write error.
  110 write(cherr,'(i8)') ierr
      call luerrm(18,'(rluget:) error when accessing file, iostat ='//
     &cherr)

      return
      end

c*********************************************************************

      subroutine rluset(lfn,move)

c...purpose: to read a state of the random number generator from a file
c...for subsequent generation from this state onwards.
      common/ludatr/mrlu(6),rrlu(100)
      save /ludatr/
      character cherr*8

c...backspace required number of records (or as many as there are).
      if(move.lt.0) then
        nbck=min(mrlu(6),-move)
        do 100 ibck=1,nbck
  100   backspace(lfn,err=120,iostat=ierr)
        mrlu(6)=mrlu(6)-nbck
      endif

c...unformatted read from unit lfn.
      nfor=1+max(0,move)
      do 110 ifor=1,nfor
  110 read(lfn,err=120,iostat=ierr) (mrlu(i1),i1=1,5),
     &(rrlu(i2),i2=1,100)
      mrlu(6)=mrlu(6)+nfor
      return

c...write error.
  120 write(cherr,'(i8)') ierr
      call luerrm(18,'(rluset:) error when accessing file, iostat ='//
     &cherr)

      return
      end

c*********************************************************************

      subroutine lurobo(the,phi,bex,bey,bez)

c...purpose: to perform rotations and boosts.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /lujets/,/ludat1/
      dimension rot(3,3),pr(3),vr(3),dp(4),dv(4)

c...find range of rotation/boost. convert boost to double precision.
      imin=1
      if(mstu(1).gt.0) imin=mstu(1)
      imax=n
      if(mstu(2).gt.0) imax=mstu(2)
      dbx=bex
      dby=bey
      dbz=bez
      goto 110

c...entry for specific range and double precision boost.
      entry ludbrb(imi,ima,the,phi,dbex,dbey,dbez)
      imin=imi
      if(imin.le.0) imin=1
      imax=ima
      if(imax.le.0) imax=n
      dbx=dbex
      dby=dbey
      dbz=dbez

c...optional resetting of v (when not set before.)
      if(mstu(33).ne.0) then
        do 100 i=min(imin,mstu(4)),min(imax,mstu(4))
        do 100 j=1,5
  100   v(i,j)=0.
        mstu(33)=0
      endif

c...check range of rotation/boost.
  110 if(imin.gt.mstu(4).or.imax.gt.mstu(4)) then
        call luerrm(11,'(lurobo:) range outside lujets memory')
        return
      endif

c...rotate, typically from z axis to direction (theta,phi).
      if(the**2+phi**2.gt.1e-20) then
        rot(1,1)=cos(the)*cos(phi)
        rot(1,2)=-sin(phi)
        rot(1,3)=sin(the)*cos(phi)
        rot(2,1)=cos(the)*sin(phi)
        rot(2,2)=cos(phi)
        rot(2,3)=sin(the)*sin(phi)
        rot(3,1)=-sin(the)
        rot(3,2)=0.
        rot(3,3)=cos(the)
        do 140 i=imin,imax
        if(k(i,1).le.0) goto 140
        do 120 j=1,3
        pr(j)=p(i,j)
  120   vr(j)=v(i,j)
        do 130 j=1,3
        p(i,j)=rot(j,1)*pr(1)+rot(j,2)*pr(2)+rot(j,3)*pr(3)
  130   v(i,j)=rot(j,1)*vr(1)+rot(j,2)*vr(2)+rot(j,3)*vr(3)
  140   continue
      endif

c...boost, typically from rest to momentum/energy=beta.
      if(dbx**2+dby**2+dbz**2.gt.1e-20) then
        db=sqrt(dbx**2+dby**2+dbz**2)
        if(db.gt.0.99999999d0) then
c...rescale boost vector if too close to unity.
          call luerrm(3,'(lurobo:) boost vector too large')
          dbx=dbx*(0.99999999d0/db)
          dby=dby*(0.99999999d0/db)
          dbz=dbz*(0.99999999d0/db)
          db=0.99999999d0
        endif
        dga=1d0/sqrt(1d0-db**2)
        do 160 i=imin,imax
        if(k(i,1).le.0) goto 160
        do 150 j=1,4
        dp(j)=p(i,j)
  150   dv(j)=v(i,j)
        dbp=dbx*dp(1)+dby*dp(2)+dbz*dp(3)
        dgabp=dga*(dga*dbp/(1d0+dga)+dp(4))
        p(i,1)=dp(1)+dgabp*dbx
        p(i,2)=dp(2)+dgabp*dby
        p(i,3)=dp(3)+dgabp*dbz
        p(i,4)=dga*(dp(4)+dbp)
        dbv=dbx*dv(1)+dby*dv(2)+dbz*dv(3)
        dgabv=dga*(dga*dbv/(1d0+dga)+dv(4))
        v(i,1)=dv(1)+dgabv*dbx
        v(i,2)=dv(2)+dgabv*dby
        v(i,3)=dv(3)+dgabv*dbz
        v(i,4)=dga*(dv(4)+dbv)
  160   continue
      endif

      return
      end

c*********************************************************************

      subroutine luedit(medit)

c...purpose: to perform global manipulations on the event record,
c...in particular to exclude unstable or undetectable partons/particles.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension ns(2),pts(2),pls(2)

c...remove unwanted partons/particles.
      if((medit.ge.0.and.medit.le.3).or.medit.eq.5) then
        imax=n
        if(mstu(2).gt.0) imax=mstu(2)
        i1=max(1,mstu(1))-1
        do 110 i=max(1,mstu(1)),imax
        if(k(i,1).eq.0.or.k(i,1).gt.20) goto 110
        if(medit.eq.1) then
          if(k(i,1).gt.10) goto 110
        elseif(medit.eq.2) then
          if(k(i,1).gt.10) goto 110
          kc=lucomp(k(i,2))
          if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.kc.eq.18)
     &    goto 110
        elseif(medit.eq.3) then
          if(k(i,1).gt.10) goto 110
          kc=lucomp(k(i,2))
          if(kc.eq.0) goto 110
          if(kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0) goto 110
        elseif(medit.eq.5) then
          if(k(i,1).eq.13.or.k(i,1).eq.14) goto 110
          kc=lucomp(k(i,2))
          if(kc.eq.0) goto 110
          if(k(i,1).ge.11.and.kchg(kc,2).eq.0) goto 110
        endif

c...pack remaining partons/particles. origin no longer known.
        i1=i1+1
        do 100 j=1,5
        k(i1,j)=k(i,j)
        p(i1,j)=p(i,j)
  100   v(i1,j)=v(i,j)
        k(i1,3)=0
  110   continue
        if(i1.lt.n) mstu(3)=0
        if(i1.lt.n) mstu(70)=0
        n=i1

c...selective removal of class of entries. new position of retained.
      elseif(medit.ge.11.and.medit.le.15) then
        i1=0
        do 120 i=1,n
        k(i,3)=mod(k(i,3),mstu(5))
        if(medit.eq.11.and.k(i,1).lt.0) goto 120
        if(medit.eq.12.and.k(i,1).eq.0) goto 120
        if(medit.eq.13.and.(k(i,1).eq.11.or.k(i,1).eq.12.or.
     &  k(i,1).eq.15).and.k(i,2).ne.94) goto 120
        if(medit.eq.14.and.(k(i,1).eq.13.or.k(i,1).eq.14.or.
     &  k(i,2).eq.94)) goto 120
        if(medit.eq.15.and.k(i,1).ge.21) goto 120
        i1=i1+1
        k(i,3)=k(i,3)+mstu(5)*i1
  120   continue

c...find new event history information and replace old.
        do 140 i=1,n
        if(k(i,1).le.0.or.k(i,1).gt.20.or.k(i,3)/mstu(5).eq.0) goto 140
        id=i
  130   im=mod(k(id,3),mstu(5))
        if(medit.eq.13.and.im.gt.0.and.im.le.n) then
          if((k(im,1).eq.11.or.k(im,1).eq.12.or.k(im,1).eq.15).and.
     &    k(im,2).ne.94) then
            id=im
            goto 130
          endif
        elseif(medit.eq.14.and.im.gt.0.and.im.le.n) then
          if(k(im,1).eq.13.or.k(im,1).eq.14.or.k(im,2).eq.94) then
            id=im
            goto 130
          endif
        endif
        k(i,3)=mstu(5)*(k(i,3)/mstu(5))
        if(im.ne.0) k(i,3)=k(i,3)+k(im,3)/mstu(5)
        if(k(i,1).ne.3.and.k(i,1).ne.13.and.k(i,1).ne.14) then
          if(k(i,4).gt.0.and.k(i,4).le.mstu(4)) k(i,4)=
     &    k(k(i,4),3)/mstu(5)
          if(k(i,5).gt.0.and.k(i,5).le.mstu(4)) k(i,5)=
     &    k(k(i,5),3)/mstu(5)
        else
          kcm=mod(k(i,4)/mstu(5),mstu(5))
          if(kcm.gt.0.and.kcm.le.mstu(4)) kcm=k(kcm,3)/mstu(5)
          kcd=mod(k(i,4),mstu(5))
          if(kcd.gt.0.and.kcd.le.mstu(4)) kcd=k(kcd,3)/mstu(5)
          k(i,4)=mstu(5)**2*(k(i,4)/mstu(5)**2)+mstu(5)*kcm+kcd
          kcm=mod(k(i,5)/mstu(5),mstu(5))
          if(kcm.gt.0.and.kcm.le.mstu(4)) kcm=k(kcm,3)/mstu(5)
          kcd=mod(k(i,5),mstu(5))
          if(kcd.gt.0.and.kcd.le.mstu(4)) kcd=k(kcd,3)/mstu(5)
          k(i,5)=mstu(5)**2*(k(i,5)/mstu(5)**2)+mstu(5)*kcm+kcd
        endif
  140   continue

c...pack remaining entries.
        i1=0
        mstu90=mstu(90)
        mstu(90)=0
        do 170 i=1,n
        if(k(i,3)/mstu(5).eq.0) goto 170
        i1=i1+1
        do 150 j=1,5
        k(i1,j)=k(i,j)
        p(i1,j)=p(i,j)
  150   v(i1,j)=v(i,j)
        k(i1,3)=mod(k(i1,3),mstu(5))
        do 160 iz=1,mstu90
        if(i.eq.mstu(90+iz)) then
          mstu(90)=mstu(90)+1
          mstu(90+mstu(90))=i1
          paru(90+mstu(90))=paru(90+iz)
        endif
  160   continue
  170   continue
        if(i1.lt.n) mstu(3)=0
        if(i1.lt.n) mstu(70)=0
        n=i1

c...save top entries at bottom of lujets commonblock.
      elseif(medit.eq.21) then
        if(2*n.ge.mstu(4)) then
          call luerrm(11,'(luedit:) no more memory left in lujets')
          return
        endif
        do 180 i=1,n
        do 180 j=1,5
        k(mstu(4)-i,j)=k(i,j)
        p(mstu(4)-i,j)=p(i,j)
  180   v(mstu(4)-i,j)=v(i,j)
        mstu(32)=n

c...restore bottom entries of commonblock lujets to top.
      elseif(medit.eq.22) then
        do 190 i=1,mstu(32)
        do 190 j=1,5
        k(i,j)=k(mstu(4)-i,j)
        p(i,j)=p(mstu(4)-i,j)
  190   v(i,j)=v(mstu(4)-i,j)
        n=mstu(32)

c...mark primary entries at top of commonblock lujets as untreated.
      elseif(medit.eq.23) then
        i1=0
        do 200 i=1,n
        kh=k(i,3)
        if(kh.ge.1) then
          if(k(kh,1).gt.20) kh=0
        endif
        if(kh.ne.0) goto 210
        i1=i1+1
  200   if(k(i,1).gt.10.and.k(i,1).le.20) k(i,1)=k(i,1)-10
  210   n=i1

c...place largest axis along z axis and second largest in xy plane.
      elseif(medit.eq.31.or.medit.eq.32) then
        call ludbrb(1,n+mstu(3),0.,-ulangl(p(mstu(61),1),
     &  p(mstu(61),2)),0d0,0d0,0d0)
        call ludbrb(1,n+mstu(3),-ulangl(p(mstu(61),3),
     &  p(mstu(61),1)),0.,0d0,0d0,0d0)
        call ludbrb(1,n+mstu(3),0.,-ulangl(p(mstu(61)+1,1),
     &  p(mstu(61)+1,2)),0d0,0d0,0d0)
        if(medit.eq.31) return

c...rotate to put slim jet along +z axis.
        do 220 is=1,2
        ns(is)=0
        pts(is)=0.
  220   pls(is)=0.
        do 230 i=1,n
        if(k(i,1).le.0.or.k(i,1).gt.10) goto 230
        if(mstu(41).ge.2) then
          kc=lucomp(k(i,2))
          if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &    kc.eq.18) goto 230
          if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &    goto 230
        endif
        is=2.-sign(0.5,p(i,3))
        ns(is)=ns(is)+1
        pts(is)=pts(is)+sqrt(p(i,1)**2+p(i,2)**2)
  230   continue
        if(ns(1)*pts(2)**2.lt.ns(2)*pts(1)**2)
     &  call ludbrb(1,n+mstu(3),paru(1),0.,0d0,0d0,0d0)

c...rotate to put second largest jet into -z,+x quadrant.
        do 240 i=1,n
        if(p(i,3).ge.0.) goto 240
        if(k(i,1).le.0.or.k(i,1).gt.10) goto 240
        if(mstu(41).ge.2) then
          kc=lucomp(k(i,2))
          if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &    kc.eq.18) goto 240
          if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &    goto 240
        endif
        is=2.-sign(0.5,p(i,1))
        pls(is)=pls(is)-p(i,3)
  240   continue
        if(pls(2).gt.pls(1)) call ludbrb(1,n+mstu(3),0.,paru(1),
     &  0d0,0d0,0d0)
      endif

      return
      end

c*********************************************************************

      subroutine lulist(mlist)

c...purpose: to give program heading, or list an event, or particle
c...data, or current parameter values.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      save /lujets/,/ludat1/,/ludat2/,/ludat3/
      character chap*16,chac*16,chan*16,chad(5)*16,chmo(12)*3,chdl(7)*4
      dimension ps(6)
      data chmo/'jan','feb','mar','apr','may','jun','jul','aug','sep',
     &'oct','nov','dec'/,chdl/'(())',' ','()','!!','<>','==','(==)'/

c...initialization printout: version number and date of last change.
      if(mlist.eq.0.or.mstu(12).eq.1) then
        write(mstu(11),5000) mstu(181),mstu(182),mstu(185),
     &  chmo(mstu(184)),mstu(183)
        mstu(12)=0
        if(mlist.eq.0) return
      endif

c...list event data, including additional lines after n.
      if(mlist.ge.1.and.mlist.le.3) then
        if(mlist.eq.1) write(mstu(11),5100)
        if(mlist.eq.2) write(mstu(11),5200)
        if(mlist.eq.3) write(mstu(11),5300)
        lmx=12
        if(mlist.ge.2) lmx=16
        istr=0
        imax=n
        if(mstu(2).gt.0) imax=mstu(2)
        do 120 i=max(1,mstu(1)),max(imax,n+max(0,mstu(3)))
        if((i.gt.imax.and.i.le.n).or.k(i,1).lt.0) goto 120

c...get particle name, pad it and check it is not too long.
        call luname(k(i,2),chap)
        len=0
        do 100 lem=1,16
  100   if(chap(lem:lem).ne.' ') len=lem
        mdl=(k(i,1)+19)/10
        ldl=0
        if(mdl.eq.2.or.mdl.ge.8) then
          chac=chap
          if(len.gt.lmx) chac(lmx:lmx)='?'
        else
          ldl=1
          if(mdl.eq.1.or.mdl.eq.7) ldl=2
          if(len.eq.0) then
            chac=chdl(mdl)(1:2*ldl)//' '
          else
            chac=chdl(mdl)(1:ldl)//chap(1:min(len,lmx-2*ldl))//
     &      chdl(mdl)(ldl+1:2*ldl)//' '
            if(len+2*ldl.gt.lmx) chac(lmx:lmx)='?'
          endif
        endif

c...add information on string connection.
        if(k(i,1).eq.1.or.k(i,1).eq.2.or.k(i,1).eq.11.or.k(i,1).eq.12)
     &  then
          kc=lucomp(k(i,2))
          kcc=0
          if(kc.ne.0) kcc=kchg(kc,2)
          if(iabs(k(i,2)).eq.39) then
            if(len+2*ldl+3.le.lmx) chac(lmx-1:lmx-1)='x'
          elseif(kcc.ne.0.and.istr.eq.0) then
            istr=1
            if(len+2*ldl+3.le.lmx) chac(lmx-1:lmx-1)='a'
          elseif(kcc.ne.0.and.(k(i,1).eq.2.or.k(i,1).eq.12)) then
            if(len+2*ldl+3.le.lmx) chac(lmx-1:lmx-1)='i'
          elseif(kcc.ne.0) then
            istr=0
            if(len+2*ldl+3.le.lmx) chac(lmx-1:lmx-1)='v'
          endif
        endif

c...write data for particle/jet.
        if(mlist.eq.1.and.abs(p(i,4)).lt.9999.) then
          write(mstu(11),5400) i,chac(1:12),(k(i,j1),j1=1,3),
     &    (p(i,j2),j2=1,5)
        elseif(mlist.eq.1.and.abs(p(i,4)).lt.99999.) then
          write(mstu(11),5500) i,chac(1:12),(k(i,j1),j1=1,3),
     &    (p(i,j2),j2=1,5)
        elseif(mlist.eq.1) then
          write(mstu(11),5600) i,chac(1:12),(k(i,j1),j1=1,3),
     &    (p(i,j2),j2=1,5)
        elseif(mstu(5).eq.10000.and.(k(i,1).eq.3.or.k(i,1).eq.13.or.
     &  k(i,1).eq.14)) then
          write(mstu(11),5700) i,chac,(k(i,j1),j1=1,3),
     &    k(i,4)/100000000,mod(k(i,4)/10000,10000),mod(k(i,4),10000),
     &    k(i,5)/100000000,mod(k(i,5)/10000,10000),mod(k(i,5),10000),
     &    (p(i,j2),j2=1,5)
        else
          write(mstu(11),5800) i,chac,(k(i,j1),j1=1,5),(p(i,j2),j2=1,5)
        endif
        if(mlist.eq.3) write(mstu(11),5900) (v(i,j),j=1,5)

c...insert extra separator lines specified by user.
        if(mstu(70).ge.1) then
          isep=0
          do 110 j=1,min(10,mstu(70))
  110     if(i.eq.mstu(70+j)) isep=1
          if(isep.eq.1.and.mlist.eq.1) write(mstu(11),6000)
          if(isep.eq.1.and.mlist.ge.2) write(mstu(11),6100)
        endif
  120   continue

c...sum of charges and momenta.
        do 130 j=1,6
  130   ps(j)=plu(0,j)
        if(mlist.eq.1.and.abs(ps(4)).lt.9999.) then
          write(mstu(11),6200) ps(6),(ps(j),j=1,5)
        elseif(mlist.eq.1.and.abs(ps(4)).lt.99999.) then
          write(mstu(11),6300) ps(6),(ps(j),j=1,5)
        elseif(mlist.eq.1) then
          write(mstu(11),6400) ps(6),(ps(j),j=1,5)
        else
          write(mstu(11),6500) ps(6),(ps(j),j=1,5)
        endif

c...give simple list of kf codes defined in program.
      elseif(mlist.eq.11) then
        write(mstu(11),6600)
        do 140 kf=1,40
        call luname(kf,chap)
        call luname(-kf,chan)
        if(chap.ne.' '.and.chan.eq.' ') write(mstu(11),6700) kf,chap
  140   if(chan.ne.' ') write(mstu(11),6700) kf,chap,-kf,chan
        do 150 kfls=1,3,2
        do 150 kfla=1,8
        do 150 kflb=1,kfla-(3-kfls)/2
        kf=1000*kfla+100*kflb+kfls
        call luname(kf,chap)
        call luname(-kf,chan)
  150   write(mstu(11),6700) kf,chap,-kf,chan
        kf=130
        call luname(kf,chap)
        write(mstu(11),6700) kf,chap
        kf=310
        call luname(kf,chap)
        write(mstu(11),6700) kf,chap
        do 170 kmul=0,5
        kfls=3
        if(kmul.eq.0.or.kmul.eq.3) kfls=1
        if(kmul.eq.5) kfls=5
        kflr=0
        if(kmul.eq.2.or.kmul.eq.3) kflr=1
        if(kmul.eq.4) kflr=2
        do 170 kflb=1,8
        do 160 kflc=1,kflb-1
        kf=10000*kflr+100*kflb+10*kflc+kfls
        call luname(kf,chap)
        call luname(-kf,chan)
  160   write(mstu(11),6700) kf,chap,-kf,chan
        kf=10000*kflr+110*kflb+kfls
        call luname(kf,chap)
  170   write(mstu(11),6700) kf,chap
        do 190 kflsp=1,3
        kfls=2+2*(kflsp/3)
        do 190 kfla=1,8
        do 190 kflb=1,kfla
        do 180 kflc=1,kflb
        if(kflsp.eq.1.and.(kfla.eq.kflb.or.kflb.eq.kflc)) goto 180
        if(kflsp.eq.2.and.kfla.eq.kflc) goto 180
        if(kflsp.eq.1) kf=1000*kfla+100*kflc+10*kflb+kfls
        if(kflsp.ge.2) kf=1000*kfla+100*kflb+10*kflc+kfls
        call luname(kf,chap)
        call luname(-kf,chan)
        write(mstu(11),6700) kf,chap,-kf,chan
  180   continue
  190   continue

c...list parton/particle data table. check whether to be listed.
      elseif(mlist.eq.12) then
        write(mstu(11),6800)
        mstj24=mstj(24)
        mstj(24)=0
        kfmax=20883
        if(mstu(2).ne.0) kfmax=mstu(2)
        do 220 kf=max(1,mstu(1)),kfmax
        kc=lucomp(kf)
        if(kc.eq.0) goto 220
        if(mstu(14).eq.0.and.kf.gt.100.and.kc.le.100) goto 220
        if(mstu(14).gt.0.and.kf.gt.100.and.max(mod(kf/1000,10),
     &  mod(kf/100,10)).gt.mstu(14)) goto 220

c...find particle name and mass. print information.
        call luname(kf,chap)
        if(kf.le.100.and.chap.eq.' '.and.mdcy(kc,2).eq.0) goto 220
        call luname(-kf,chan)
        pm=ulmass(kf)
        write(mstu(11),6900) kf,kc,chap,chan,kchg(kc,1),kchg(kc,2),
     &  kchg(kc,3),pm,pmas(kc,2),pmas(kc,3),pmas(kc,4),mdcy(kc,1)

c...particle decay: channel number, branching ration, matrix element,
c...decay products.
        if(kf.gt.100.and.kc.le.100) goto 220
        do 210 idc=mdcy(kc,2),mdcy(kc,2)+mdcy(kc,3)-1
        do 200 j=1,5
  200   call luname(kfdp(idc,j),chad(j))
  210   write(mstu(11),7000) idc,mdme(idc,1),mdme(idc,2),brat(idc),
     &  (chad(j),j=1,5)
  220   continue
        mstj(24)=mstj24

c...list parameter value table.
      elseif(mlist.eq.13) then
        write(mstu(11),7100)
        do 230 i=1,200
  230   write(mstu(11),7200) i,mstu(i),paru(i),mstj(i),parj(i),parf(i)
      endif

c...format statements for output on unit mstu(11) (by default 6).
 5000 format(///20x,'the lund monte carlo - jetset version ',i1,'.',i1/
     &20x,'**  last date of change:  ',i2,1x,a3,1x,i4,'  **'/)
 5100 format(///28x,'event listing (summary)'//4x,'i  particle/jet ks',
     &5x,'kf orig    p_x      p_y      p_z       e        m'/)
 5200 format(///28x,'event listing (standard)'//4x,'i  particle/jet',
     &'  k(i,1)   k(i,2) k(i,3)     k(i,4)      k(i,5)       p(i,1)',
     &'       p(i,2)       p(i,3)       p(i,4)       p(i,5)'/)
 5300 format(///28x,'event listing (with vertices)'//4x,'i  particle/j',
     &'et  k(i,1)   k(i,2) k(i,3)     k(i,4)      k(i,5)       p(i,1)',
     &'       p(i,2)       p(i,3)       p(i,4)       p(i,5)'/73x,
     &'v(i,1)       v(i,2)       v(i,3)       v(i,4)       v(i,5)'/)
 5400 format(1x,i4,2x,a12,1x,i2,1x,i6,1x,i4,5f9.3)
 5500 format(1x,i4,2x,a12,1x,i2,1x,i6,1x,i4,5f9.2)
 5600 format(1x,i4,2x,a12,1x,i2,1x,i6,1x,i4,5f9.1)
 5700 format(1x,i4,2x,a16,1x,i3,1x,i8,2x,i4,2(3x,i1,2i4),5f13.5)
 5800 format(1x,i4,2x,a16,1x,i3,1x,i8,2x,i4,2(3x,i9),5f13.5)
 5900 format(66x,5(1x,f12.3))
 6000 format(1x,78('='))
 6100 format(1x,130('='))
 6200 format(19x,'sum:',f6.2,5x,5f9.3)
 6300 format(19x,'sum:',f6.2,5x,5f9.2)
 6400 format(19x,'sum:',f6.2,5x,5f9.1)
 6500 format(19x,'sum charge:',f6.2,3x,'sum momentum and inv. mass:',
     &5f13.5)
 6600 format(///20x,'list of kf codes in program'/)
 6700 format(4x,i6,4x,a16,6x,i6,4x,a16)
 6800 format(///30x,'particle/parton data table'//5x,'kf',5x,'kc',4x,
     &'particle',8x,'antiparticle',6x,'chg  col  anti',8x,'mass',7x,
     &'width',7x,'w-cut',5x,'lifetime',1x,'decay'/11x,'idc',1x,'on/off',
     &1x,'me',3x,'br.rat.',4x,'decay products')
 6900 format(/1x,i6,3x,i4,4x,a16,a16,3i5,1x,f12.5,2(1x,f11.5),
     &2x,f12.5,3x,i2)
 7000 format(10x,i4,2x,i3,2x,i3,2x,f8.5,4x,5a16)
 7100 format(///20x,'parameter value table'//4x,'i',3x,'mstu(i)',
     &8x,'paru(i)',3x,'mstj(i)',8x,'parj(i)',8x,'parf(i)')
 7200 format(1x,i4,1x,i9,1x,f14.5,1x,i9,1x,f14.5,1x,f14.5)

      return
      end

c*********************************************************************

      subroutine luupda(mupda,lfn)

c...purpose: to facilitate the updating of particle and decay data.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      common/ludat4/chaf(500)
      character chaf*8
      save /ludat1/,/ludat2/,/ludat3/,/ludat4/
      character chinl*80,chkc*4,chvar(19)*9,chlin*72,
     &chblk(20)*72,chold*12,chtmp*12,chnew*12,chcom*12
      data chvar/ 'kchg(i,1)','kchg(i,2)','kchg(i,3)','pmas(i,1)',
     &'pmas(i,2)','pmas(i,3)','pmas(i,4)','mdcy(i,1)','mdcy(i,2)',
     &'mdcy(i,3)','mdme(i,1)','mdme(i,2)','brat(i)  ','kfdp(i,1)',
     &'kfdp(i,2)','kfdp(i,3)','kfdp(i,4)','kfdp(i,5)','chaf(i)  '/

c...write information on file for editing.
      if(mstu(12).ge.1) call lulist(0)
      if(mupda.eq.1) then
        do 110 kc=1,mstu(6)
        write(lfn,5000) kc,chaf(kc),(kchg(kc,j1),j1=1,3),
     &  (pmas(kc,j2),j2=1,4),mdcy(kc,1)
        do 100 idc=mdcy(kc,2),mdcy(kc,2)+mdcy(kc,3)-1
  100   write(lfn,5100) mdme(idc,1),mdme(idc,2),brat(idc),
     &  (kfdp(idc,j),j=1,5)
  110   continue

c...reset variables and read information from edited file.
      elseif(mupda.eq.2) then
        do 120 i=1,mstu(7)
        mdme(i,1)=1
        mdme(i,2)=0
        brat(i)=0.
        do 120 j=1,5
  120   kfdp(i,j)=0
        kc=0
        idc=0
        ndc=0
  130   read(lfn,5200,end=140) chinl
        if(chinl(2:5).ne.'    ') then
          chkc=chinl(2:5)
          if(kc.ne.0) then
            mdcy(kc,2)=0
            if(ndc.ne.0) mdcy(kc,2)=idc+1-ndc
            mdcy(kc,3)=ndc
          endif
          read(chkc,5300) kc
          if(kc.le.0.or.kc.gt.mstu(6)) call luerrm(27,
     &    '(luupda:) read kc code illegal, kc ='//chkc)
          read(chinl,5000) kcr,chaf(kc),(kchg(kc,j1),j1=1,3),
     &    (pmas(kc,j2),j2=1,4),mdcy(kc,1)
          ndc=0
        else
          idc=idc+1
          ndc=ndc+1
          if(idc.ge.mstu(7)) call luerrm(27,
     &    '(luupda:) decay data arrays full by kc ='//chkc)
          read(chinl,5100) mdme(idc,1),mdme(idc,2),brat(idc),
     &    (kfdp(idc,j),j=1,5)
        endif
        goto 130
  140   mdcy(kc,2)=0
        if(ndc.ne.0) mdcy(kc,2)=idc+1-ndc
        mdcy(kc,3)=ndc

c...perform possible tests that new information is consistent.
        mstj24=mstj(24)
        mstj(24)=0
        do 170 kc=1,mstu(6)
        write(chkc,5300) kc
        if(min(pmas(kc,1),pmas(kc,2),pmas(kc,3),pmas(kc,1)-pmas(kc,3),
     &  pmas(kc,4)).lt.0..or.mdcy(kc,3).lt.0) call luerrm(17,
     &  '(luupda:) mass/width/life/(# channels) wrong for kc ='//chkc)
        brsum=0.
        do 160 idc=mdcy(kc,2),mdcy(kc,2)+mdcy(kc,3)-1
        if(mdme(idc,2).gt.80) goto 160
        kq=kchg(kc,1)
        pms=pmas(kc,1)-pmas(kc,3)-parj(64)
        merr=0
        do 150 j=1,5
        kp=kfdp(idc,j)
        if(kp.eq.0.or.kp.eq.81.or.iabs(kp).eq.82) then
        elseif(lucomp(kp).eq.0) then
          merr=3
        else
          kq=kq-luchge(kp)
          pms=pms-ulmass(kp)
        endif
  150   continue
        if(kq.ne.0) merr=max(2,merr)
        if(kfdp(idc,2).ne.0.and.(kc.le.20.or.kc.gt.40).and.
     &  (kc.le.80.or.kc.gt.100).and.mdme(idc,2).ne.34.and.
     &  mdme(idc,2).ne.61.and.pms.lt.0.) merr=max(1,merr)
        if(merr.eq.3) call luerrm(17,
     &  '(luupda:) unknown particle code in decay of kc ='//chkc)
        if(merr.eq.2) call luerrm(17,
     &  '(luupda:) charge not conserved in decay of kc ='//chkc)
        if(merr.eq.1) call luerrm(7,
     &  '(luupda:) kinematically unallowed decay of kc ='//chkc)
        brsum=brsum+brat(idc)
  160   continue
        write(chtmp,5500) brsum
        if(abs(brsum).gt.0.0005.and.abs(brsum-1.).gt.0.0005) call
     &  luerrm(7,'(luupda:) sum of branching ratios is '//chtmp(5:12)//
     &  ' for kc ='//chkc)
  170   continue
        mstj(24)=mstj24

c...initialize writing of data statements for inclusion in program.
      elseif(mupda.eq.3) then
        do 240 ivar=1,19
        ndim=mstu(6)
        if(ivar.ge.11.and.ivar.le.18) ndim=mstu(7)
        nlin=1
        chlin=' '
        chlin(7:35)='data ('//chvar(ivar)//',i=   1,    )/'
        llin=35
        chold='start'

c...loop through variables for conversion to characters.
        do 220 idim=1,ndim
        if(ivar.eq.1) write(chtmp,5400) kchg(idim,1)
        if(ivar.eq.2) write(chtmp,5400) kchg(idim,2)
        if(ivar.eq.3) write(chtmp,5400) kchg(idim,3)
        if(ivar.eq.4) write(chtmp,5500) pmas(idim,1)
        if(ivar.eq.5) write(chtmp,5500) pmas(idim,2)
        if(ivar.eq.6) write(chtmp,5500) pmas(idim,3)
        if(ivar.eq.7) write(chtmp,5500) pmas(idim,4)
        if(ivar.eq.8) write(chtmp,5400) mdcy(idim,1)
        if(ivar.eq.9) write(chtmp,5400) mdcy(idim,2)
        if(ivar.eq.10) write(chtmp,5400) mdcy(idim,3)
        if(ivar.eq.11) write(chtmp,5400) mdme(idim,1)
        if(ivar.eq.12) write(chtmp,5400) mdme(idim,2)
        if(ivar.eq.13) write(chtmp,5500) brat(idim)
        if(ivar.eq.14) write(chtmp,5400) kfdp(idim,1)
        if(ivar.eq.15) write(chtmp,5400) kfdp(idim,2)
        if(ivar.eq.16) write(chtmp,5400) kfdp(idim,3)
        if(ivar.eq.17) write(chtmp,5400) kfdp(idim,4)
        if(ivar.eq.18) write(chtmp,5400) kfdp(idim,5)
        if(ivar.eq.19) chtmp=chaf(idim)

c...length of variable, trailing decimal zeros, quotation marks.
        llow=1
        lhig=1
        do 180 ll=1,12
        if(chtmp(13-ll:13-ll).ne.' ') llow=13-ll
  180   if(chtmp(ll:ll).ne.' ') lhig=ll
        chnew=chtmp(llow:lhig)//' '
        lnew=1+lhig-llow
        if((ivar.ge.4.and.ivar.le.7).or.ivar.eq.13) then
          lnew=lnew+1
  190     lnew=lnew-1
          if(chnew(lnew:lnew).eq.'0') goto 190
          if(lnew.eq.1) chnew(1:2)='0.'
          if(lnew.eq.1) lnew=2
        elseif(ivar.eq.19) then
          do 200 ll=lnew,1,-1
          if(chnew(ll:ll).eq.'''') then
            chtmp=chnew
            chnew=chtmp(1:ll)//''''//chtmp(ll+1:11)
            lnew=lnew+1
          endif
  200     continue
          chtmp=chnew
          chnew(1:lnew+2)=''''//chtmp(1:lnew)//''''
          lnew=lnew+2
        endif

c...form composite character string, often including repetition counter.
        if(chnew.ne.chold) then
          nrpt=1
          chold=chnew
          chcom=chnew
          lcom=lnew
        else
          lrpt=lnew+1
          if(nrpt.ge.2) lrpt=lnew+3
          if(nrpt.ge.10) lrpt=lnew+4
          if(nrpt.ge.100) lrpt=lnew+5
          if(nrpt.ge.1000) lrpt=lnew+6
          llin=llin-lrpt
          nrpt=nrpt+1
          write(chtmp,5400) nrpt
          lrpt=1
          if(nrpt.ge.10) lrpt=2
          if(nrpt.ge.100) lrpt=3
          if(nrpt.ge.1000) lrpt=4
          chcom(1:lrpt+1+lnew)=chtmp(13-lrpt:12)//'*'//chnew(1:lnew)
          lcom=lrpt+1+lnew
        endif

c...add characters to end of line, to new line (after storing old line),
c...or to new block of lines (after writing old block).
        if(llin+lcom.le.70) then
          chlin(llin+1:llin+lcom+1)=chcom(1:lcom)//','
          llin=llin+lcom+1
        elseif(nlin.le.19) then
          chlin(llin+1:72)=' '
          chblk(nlin)=chlin
          nlin=nlin+1
          chlin(6:6+lcom+1)='&'//chcom(1:lcom)//','
          llin=6+lcom+1
        else
          chlin(llin:72)='/'//' '
          chblk(nlin)=chlin
          write(chtmp,5400) idim-nrpt
          chblk(1)(30:33)=chtmp(9:12)
          do 210 ilin=1,nlin
  210     write(lfn,5600) chblk(ilin)
          nlin=1
          chlin=' '
          chlin(7:35+lcom+1)='data ('//chvar(ivar)//',i=    ,    )/'//
     &    chcom(1:lcom)//','
          write(chtmp,5400) idim-nrpt+1
          chlin(25:28)=chtmp(9:12)
          llin=35+lcom+1
        endif
  220   continue

c...write final block of lines.
        chlin(llin:72)='/'//' '
        chblk(nlin)=chlin
        write(chtmp,5400) ndim
        chblk(1)(30:33)=chtmp(9:12)
        do 230 ilin=1,nlin
  230   write(lfn,5600) chblk(ilin)
  240   continue
      endif

c...formats for reading and writing particle data.
 5000 format(1x,i4,2x,a8,3i3,3f12.5,2x,f12.5,i3)
 5100 format(5x,2i5,f12.5,5i8)
 5200 format(a80)
 5300 format(i4)
 5400 format(i12)
 5500 format(f12.5)
 5600 format(a72)

      return
      end

c*********************************************************************

      function klu(i,j)

c...purpose: to provide various integer-valued event related data.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...default value. for i=0 number of entries, number of stable entries
c...or 3 times total charge.
      klu=0
      if(i.lt.0.or.i.gt.mstu(4).or.j.le.0) then
      elseif(i.eq.0.and.j.eq.1) then
        klu=n
      elseif(i.eq.0.and.(j.eq.2.or.j.eq.6)) then
        do 100 i1=1,n
        if(j.eq.2.and.k(i1,1).ge.1.and.k(i1,1).le.10) klu=klu+1
        if(j.eq.6.and.k(i1,1).ge.1.and.k(i1,1).le.10) klu=klu+
     &  luchge(k(i1,2))
  100   continue
      elseif(i.eq.0) then

c...for i > 0 direct readout of k matrix or charge.
      elseif(j.le.5) then
        klu=k(i,j)
      elseif(j.eq.6) then
        klu=luchge(k(i,2))

c...status (existing/fragmented/decayed), parton/hadron separation.
      elseif(j.le.8) then
        if(k(i,1).ge.1.and.k(i,1).le.10) klu=1
        if(j.eq.8) klu=klu*k(i,2)
      elseif(j.le.12) then
        kfa=iabs(k(i,2))
        kc=lucomp(kfa)
        kq=0
        if(kc.ne.0) kq=kchg(kc,2)
        if(j.eq.9.and.kc.ne.0.and.kq.ne.0) klu=k(i,2)
        if(j.eq.10.and.kc.ne.0.and.kq.eq.0) klu=k(i,2)
        if(j.eq.11) klu=kc
        if(j.eq.12) klu=kq*isign(1,k(i,2))

c...heaviest flavour in hadron/diquark.
      elseif(j.eq.13) then
        kfa=iabs(k(i,2))
        klu=mod(kfa/100,10)*(-1)**mod(kfa/100,10)
        if(kfa.lt.10) klu=kfa
        if(mod(kfa/1000,10).ne.0) klu=mod(kfa/1000,10)
        klu=klu*isign(1,k(i,2))

c...particle history: generation, ancestor, rank.
      elseif(j.le.16) then
        i2=i
        i1=i
  110   klu=klu+1
        i3=i2
        i2=i1
        i1=k(i1,3)
        if(i1.gt.0.and.k(i1,1).gt.0.and.k(i1,1).le.20) goto 110
        if(j.eq.15) klu=i2
        if(j.eq.16) then
          klu=0
          do 120 i1=i2+1,i3
  120     if(k(i1,3).eq.i2.and.k(i1,1).gt.0.and.k(i1,1).le.20) klu=klu+1
        endif

c...particle coming from collapsing jet system or not.
      elseif(j.eq.17) then
        i1=i
  130   klu=klu+1
        i3=i1
        i1=k(i1,3)
        i0=max(1,i1)
        kc=lucomp(k(i0,2))
        if(i1.eq.0.or.k(i0,1).le.0.or.k(i0,1).gt.20.or.kc.eq.0) then
          if(klu.eq.1) klu=-1
          if(klu.gt.1) klu=0
          return
        endif
        if(kchg(kc,2).eq.0) goto 130
        if(k(i1,1).ne.12) klu=0
        if(k(i1,1).ne.12) return
        i2=i1
  140   i2=i2+1
        if(i2.lt.n.and.k(i2,1).ne.11) goto 140
        k3m=k(i3-1,3)
        if(k3m.ge.i1.and.k3m.le.i2) klu=0
        k3p=k(i3+1,3)
        if(i3.lt.n.and.k3p.ge.i1.and.k3p.le.i2) klu=0

c...number of decay products. colour flow.
      elseif(j.eq.18) then
        if(k(i,1).eq.11.or.k(i,1).eq.12) klu=max(0,k(i,5)-k(i,4)+1)
        if(k(i,4).eq.0.or.k(i,5).eq.0) klu=0
      elseif(j.le.22) then
        if(k(i,1).ne.3.and.k(i,1).ne.13.and.k(i,1).ne.14) return
        if(j.eq.19) klu=mod(k(i,4)/mstu(5),mstu(5))
        if(j.eq.20) klu=mod(k(i,5)/mstu(5),mstu(5))
        if(j.eq.21) klu=mod(k(i,4),mstu(5))
        if(j.eq.22) klu=mod(k(i,5),mstu(5))
      else
      endif

      return
      end

c*********************************************************************

      function plu(i,j)

c...purpose: to provide various real-valued event related data.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension psum(4)

c...set default value. for i = 0 sum of momenta or charges,
c...or invariant mass of system.
      plu=0.
      if(i.lt.0.or.i.gt.mstu(4).or.j.le.0) then
      elseif(i.eq.0.and.j.le.4) then
        do 100 i1=1,n
  100   if(k(i1,1).gt.0.and.k(i1,1).le.10) plu=plu+p(i1,j)
      elseif(i.eq.0.and.j.eq.5) then
        do 110 j1=1,4
        psum(j1)=0.
        do 110 i1=1,n
  110   if(k(i1,1).gt.0.and.k(i1,1).le.10) psum(j1)=psum(j1)+p(i1,j1)
        plu=sqrt(max(0.,psum(4)**2-psum(1)**2-psum(2)**2-psum(3)**2))
      elseif(i.eq.0.and.j.eq.6) then
        do 120 i1=1,n
  120   if(k(i1,1).gt.0.and.k(i1,1).le.10) plu=plu+luchge(k(i1,2))/3.
      elseif(i.eq.0) then

c...direct readout of p matrix.
      elseif(j.le.5) then
        plu=p(i,j)

c...charge, total momentum, transverse momentum, transverse mass.
      elseif(j.le.12) then
        if(j.eq.6) plu=luchge(k(i,2))/3.
        if(j.eq.7.or.j.eq.8) plu=p(i,1)**2+p(i,2)**2+p(i,3)**2
        if(j.eq.9.or.j.eq.10) plu=p(i,1)**2+p(i,2)**2
        if(j.eq.11.or.j.eq.12) plu=p(i,5)**2+p(i,1)**2+p(i,2)**2
        if(j.eq.8.or.j.eq.10.or.j.eq.12) plu=sqrt(plu)

c...theta and phi angle in radians or degrees.
      elseif(j.le.16) then
        if(j.le.14) plu=ulangl(p(i,3),sqrt(p(i,1)**2+p(i,2)**2))
        if(j.ge.15) plu=ulangl(p(i,1),p(i,2))
        if(j.eq.14.or.j.eq.16) plu=plu*180./paru(1)

c...true rapidity, rapidity with pion mass, pseudorapidity.
      elseif(j.le.19) then
        pmr=0.
        if(j.eq.17) pmr=p(i,5)
        if(j.eq.18) pmr=ulmass(211)
        pr=max(1e-20,pmr**2+p(i,1)**2+p(i,2)**2)
        plu=sign(log(min((sqrt(pr+p(i,3)**2)+abs(p(i,3)))/sqrt(pr),
     &  1e20)),p(i,3))

c...energy and momentum fractions (only to be used in cm frame).
      elseif(j.le.25) then
        if(j.eq.20) plu=2.*sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)/paru(21)
        if(j.eq.21) plu=2.*p(i,3)/paru(21)
        if(j.eq.22) plu=2.*sqrt(p(i,1)**2+p(i,2)**2)/paru(21)
        if(j.eq.23) plu=2.*p(i,4)/paru(21)
        if(j.eq.24) plu=(p(i,4)+p(i,3))/paru(21)
        if(j.eq.25) plu=(p(i,4)-p(i,3))/paru(21)
      endif

      return
      end

c*********************************************************************

      subroutine lusphe(sph,apl)

c...purpose: to perform sphericity tensor analysis to give sphericity,
c...aplanarity and the related event axes.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension sm(3,3),sv(3,3)

c...calculate matrix to be diagonalized.
      np=0
      do 100 j1=1,3
      do 100 j2=j1,3
  100 sm(j1,j2)=0.
      ps=0.
      do 120 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 120
      if(mstu(41).ge.2) then
        kc=lucomp(k(i,2))
        if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &  kc.eq.18) goto 120
        if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &  goto 120
      endif
      np=np+1
      pa=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)
      pwt=1.
      if(abs(paru(41)-2.).gt.0.001) pwt=max(1e-10,pa)**(paru(41)-2.)
      do 110 j1=1,3
      do 110 j2=j1,3
  110 sm(j1,j2)=sm(j1,j2)+pwt*p(i,j1)*p(i,j2)
      ps=ps+pwt*pa**2
  120 continue

c...very low multiplicities (0 or 1) not considered.
      if(np.le.1) then
        call luerrm(8,'(lusphe:) too few particles for analysis')
        sph=-1.
        apl=-1.
        return
      endif
      do 130 j1=1,3
      do 130 j2=j1,3
  130 sm(j1,j2)=sm(j1,j2)/ps

c...find eigenvalues to matrix (third degree equation).
      sq=(sm(1,1)*sm(2,2)+sm(1,1)*sm(3,3)+sm(2,2)*sm(3,3)-sm(1,2)**2-
     &sm(1,3)**2-sm(2,3)**2)/3.-1./9.
      sr=-0.5*(sq+1./9.+sm(1,1)*sm(2,3)**2+sm(2,2)*sm(1,3)**2+sm(3,3)*
     &sm(1,2)**2-sm(1,1)*sm(2,2)*sm(3,3))+sm(1,2)*sm(1,3)*sm(2,3)+1./27.
      sp=cos(acos(max(min(sr/sqrt(-sq**3),1.),-1.))/3.)
      p(n+1,4)=1./3.+sqrt(-sq)*max(2.*sp,sqrt(3.*(1.-sp**2))-sp)
      p(n+3,4)=1./3.+sqrt(-sq)*min(2.*sp,-sqrt(3.*(1.-sp**2))-sp)
      p(n+2,4)=1.-p(n+1,4)-p(n+3,4)
      if(p(n+2,4).lt.1e-5) then
        call luerrm(8,'(lusphe:) all particles back-to-back')
        sph=-1.
        apl=-1.
        return
      endif

c...find first and last eigenvector by solving equation system.
      do 170 i=1,3,2
      do 140 j1=1,3
      sv(j1,j1)=sm(j1,j1)-p(n+i,4)
      do 140 j2=j1+1,3
      sv(j1,j2)=sm(j1,j2)
  140 sv(j2,j1)=sm(j1,j2)
      smax=0.
      do 150 j1=1,3
      do 150 j2=1,3
      if(abs(sv(j1,j2)).le.smax) goto 150
      ja=j1
      jb=j2
      smax=abs(sv(j1,j2))
  150 continue
      smax=0.
      do 160 j3=ja+1,ja+2
      j1=j3-3*((j3-1)/3)
      rl=sv(j1,jb)/sv(ja,jb)
      do 160 j2=1,3
      sv(j1,j2)=sv(j1,j2)-rl*sv(ja,j2)
      if(abs(sv(j1,j2)).le.smax) goto 160
      jc=j1
      smax=abs(sv(j1,j2))
  160 continue
      jb1=jb+1-3*(jb/3)
      jb2=jb+2-3*((jb+1)/3)
      p(n+i,jb1)=-sv(jc,jb2)
      p(n+i,jb2)=sv(jc,jb1)
      p(n+i,jb)=-(sv(ja,jb1)*p(n+i,jb1)+sv(ja,jb2)*p(n+i,jb2))/
     &sv(ja,jb)
      pa=sqrt(p(n+i,1)**2+p(n+i,2)**2+p(n+i,3)**2)
      sgn=(-1.)**int(rlu(0)+0.5)
      do 170 j=1,3
  170 p(n+i,j)=sgn*p(n+i,j)/pa

c...middle axis orthogonal to other two. fill other codes.
      sgn=(-1.)**int(rlu(0)+0.5)
      p(n+2,1)=sgn*(p(n+1,2)*p(n+3,3)-p(n+1,3)*p(n+3,2))
      p(n+2,2)=sgn*(p(n+1,3)*p(n+3,1)-p(n+1,1)*p(n+3,3))
      p(n+2,3)=sgn*(p(n+1,1)*p(n+3,2)-p(n+1,2)*p(n+3,1))
      do 180 i=1,3
      k(n+i,1)=31
      k(n+i,2)=95
      k(n+i,3)=i
      k(n+i,4)=0
      k(n+i,5)=0
      p(n+i,5)=0.
      do 180 j=1,5
  180 v(i,j)=0.

c...calculate sphericity and aplanarity. select storing option.
      sph=1.5*(p(n+2,4)+p(n+3,4))
      apl=1.5*p(n+3,4)
      mstu(61)=n+1
      mstu(62)=np
      if(mstu(43).le.1) mstu(3)=3
      if(mstu(43).ge.2) n=n+3

      return
      end

c*********************************************************************

      subroutine luthru(thr,obl)

c...purpose: to perform thrust analysis to give thrust, oblateness
c...and the related event axes.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension tdi(3),tpr(3)

c...take copy of particles that are to be considered in thrust analysis.
      np=0
      ps=0.
      do 100 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 100
      if(mstu(41).ge.2) then
        kc=lucomp(k(i,2))
        if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &  kc.eq.18) goto 100
        if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &  goto 100
      endif
      if(n+np+mstu(44)+15.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(luthru:) no more memory left in lujets')
        thr=-2.
        obl=-2.
        return
      endif
      np=np+1
      k(n+np,1)=23
      p(n+np,1)=p(i,1)
      p(n+np,2)=p(i,2)
      p(n+np,3)=p(i,3)
      p(n+np,4)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)
      p(n+np,5)=1.
      if(abs(paru(42)-1.).gt.0.001) p(n+np,5)=p(n+np,4)**(paru(42)-1.)
      ps=ps+p(n+np,4)*p(n+np,5)
  100 continue

c...very low multiplicities (0 or 1) not considered.
      if(np.le.1) then
        call luerrm(8,'(luthru:) too few particles for analysis')
        thr=-1.
        obl=-1.
        return
      endif

c...loop over thrust and major. t axis along z direction in latter case.
      do 280 ild=1,2
      if(ild.eq.2) then
        k(n+np+1,1)=31
        phi=ulangl(p(n+np+1,1),p(n+np+1,2))
        mstu(33)=1
        call ludbrb(n+1,n+np+1,0.,-phi,0d0,0d0,0d0)
        the=ulangl(p(n+np+1,3),p(n+np+1,1))
        call ludbrb(n+1,n+np+1,-the,0.,0d0,0d0,0d0)
      endif

c...find and order particles with highest p (pt for major).
      do 110 ilf=n+np+4,n+np+mstu(44)+4
  110 p(ilf,4)=0.
      do 150 i=n+1,n+np
      if(ild.eq.2) p(i,4)=sqrt(p(i,1)**2+p(i,2)**2)
      do 120 ilf=n+np+mstu(44)+3,n+np+4,-1
      if(p(i,4).le.p(ilf,4)) goto 130
      do 120 j=1,5
  120 p(ilf+1,j)=p(ilf,j)
      ilf=n+np+3
  130 do 140 j=1,5
  140 p(ilf+1,j)=p(i,j)
  150 continue

c...find and order initial axes with highest thrust (major).
      do 160 ilg=n+np+mstu(44)+5,n+np+mstu(44)+15
  160 p(ilg,4)=0.
      nc=2**(min(mstu(44),np)-1)
      do 220 ilc=1,nc
      do 170 j=1,3
  170 tdi(j)=0.
      do 180 ilf=1,min(mstu(44),np)
      sgn=p(n+np+ilf+3,5)
      if(2**ilf*((ilc+2**(ilf-1)-1)/2**ilf).ge.ilc) sgn=-sgn
      do 180 j=1,4-ild
  180 tdi(j)=tdi(j)+sgn*p(n+np+ilf+3,j)
      tds=tdi(1)**2+tdi(2)**2+tdi(3)**2
      do 190 ilg=n+np+mstu(44)+min(ilc,10)+4,n+np+mstu(44)+5,-1
      if(tds.le.p(ilg,4)) goto 200
      do 190 j=1,4
  190 p(ilg+1,j)=p(ilg,j)
      ilg=n+np+mstu(44)+4
  200 do 210 j=1,3
  210 p(ilg+1,j)=tdi(j)
      p(ilg+1,4)=tds
  220 continue

c...iterate direction of axis until stable maximum.
      p(n+np+ild,4)=0.
      ilg=0
  230 ilg=ilg+1
      thp=0.
  240 thps=thp
      do 250 j=1,3
      if(thp.le.1e-10) tdi(j)=p(n+np+mstu(44)+4+ilg,j)
      if(thp.gt.1e-10) tdi(j)=tpr(j)
  250 tpr(j)=0.
      do 260 i=n+1,n+np
      sgn=sign(p(i,5),tdi(1)*p(i,1)+tdi(2)*p(i,2)+tdi(3)*p(i,3))
      do 260 j=1,4-ild
  260 tpr(j)=tpr(j)+sgn*p(i,j)
      thp=sqrt(tpr(1)**2+tpr(2)**2+tpr(3)**2)/ps
      if(thp.ge.thps+paru(48)) goto 240

c...save good axis. try new initial axis until a number of tries agree.
      if(thp.lt.p(n+np+ild,4)-paru(48).and.ilg.lt.min(10,nc)) goto 230
      if(thp.gt.p(n+np+ild,4)+paru(48)) then
        iagr=0
        sgn=(-1.)**int(rlu(0)+0.5)
        do 270 j=1,3
  270   p(n+np+ild,j)=sgn*tpr(j)/(ps*thp)
        p(n+np+ild,4)=thp
        p(n+np+ild,5)=0.
      endif
      iagr=iagr+1
  280 if(iagr.lt.mstu(45).and.ilg.lt.min(10,nc)) goto 230

c...find minor axis and value by orthogonality.
      sgn=(-1.)**int(rlu(0)+0.5)
      p(n+np+3,1)=-sgn*p(n+np+2,2)
      p(n+np+3,2)=sgn*p(n+np+2,1)
      p(n+np+3,3)=0.
      thp=0.
      do 290 i=n+1,n+np
  290 thp=thp+p(i,5)*abs(p(n+np+3,1)*p(i,1)+p(n+np+3,2)*p(i,2))
      p(n+np+3,4)=thp/ps
      p(n+np+3,5)=0.

c...fill axis information. rotate back to original coordinate system.
      do 300 ild=1,3
      k(n+ild,1)=31
      k(n+ild,2)=96
      k(n+ild,3)=ild
      k(n+ild,4)=0
      k(n+ild,5)=0
      do 300 j=1,5
      p(n+ild,j)=p(n+np+ild,j)
  300 v(n+ild,j)=0.
      call ludbrb(n+1,n+3,the,phi,0d0,0d0,0d0)

c...calculate thrust and oblateness. select storing option.
      thr=p(n+1,4)
      obl=p(n+2,4)-p(n+3,4)
      mstu(61)=n+1
      mstu(62)=np
      if(mstu(43).le.1) mstu(3)=3
      if(mstu(43).ge.2) n=n+3

      return
      end

c*********************************************************************

      subroutine luclus(njet)

c...purpose: to subdivide the particle content of an event into
c...jets/clusters.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension ps(5)
      save nsav,np,ps,pss,rinit,npre,nrem

c...functions: distance measure in pt or (pseudo)mass.
      r2t(i1,i2)=(p(i1,5)*p(i2,5)-p(i1,1)*p(i2,1)-p(i1,2)*p(i2,2)-
     &p(i1,3)*p(i2,3))*2.*p(i1,5)*p(i2,5)/(0.0001+p(i1,5)+p(i2,5))**2
      r2m(i1,i2)=2.*p(i1,4)*p(i2,4)*(1.-(p(i1,1)*p(i2,1)+p(i1,2)*
     &p(i2,2)+p(i1,3)*p(i2,3))/(p(i1,5)*p(i2,5)))

c...if first time, reset. if reentering, skip preliminaries.
      if(mstu(48).le.0) then
        np=0
        do 100 j=1,5
  100   ps(j)=0.
        pss=0.
      else
        njet=nsav
        if(mstu(43).ge.2) n=n-njet
        do 110 i=n+1,n+njet
  110   p(i,5)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)
        if(mstu(46).le.3) r2acc=paru(44)**2
        if(mstu(46).ge.4) r2acc=paru(45)*ps(5)**2
        nloop=0
        goto 290
      endif

c...find which particles are to be considered in cluster search.
      do 140 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 140
      if(mstu(41).ge.2) then
        kc=lucomp(k(i,2))
        if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &  kc.eq.18) goto 140
        if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &  goto 140
      endif
      if(n+2*np.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(luclus:) no more memory left in lujets')
        njet=-1
        return
      endif

c...take copy of these particles, with space left for jets later on.
      np=np+1
      k(n+np,3)=i
      do 120 j=1,5
  120 p(n+np,j)=p(i,j)
      if(mstu(42).eq.0) p(n+np,5)=0.
      if(mstu(42).eq.1.and.k(i,2).ne.22) p(n+np,5)=pmas(101,1)
      p(n+np,4)=sqrt(p(n+np,5)**2+p(i,1)**2+p(i,2)**2+p(i,3)**2)
      p(n+np,5)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)
      do 130 j=1,4
  130 ps(j)=ps(j)+p(n+np,j)
      pss=pss+p(n+np,5)
  140 continue
      do 150 i=n+1,n+np
      k(i+np,3)=k(i,3)
      do 150 j=1,5
  150 p(i+np,j)=p(i,j)
      ps(5)=sqrt(max(0.,ps(4)**2-ps(1)**2-ps(2)**2-ps(3)**2))

c...very low multiplicities not considered.
      if(np.lt.mstu(47)) then
        call luerrm(8,'(luclus:) too few particles for analysis')
        njet=-1
        return
      endif

c...find precluster configuration. if too few jets, make harder cuts.
      nloop=0
      if(mstu(46).le.3) r2acc=paru(44)**2
      if(mstu(46).ge.4) r2acc=paru(45)*ps(5)**2
      rinit=1.25*paru(43)
      if(np.le.mstu(47)+2) rinit=0.
  160 rinit=0.8*rinit
      npre=0
      nrem=np
      do 170 i=n+np+1,n+2*np
  170 k(i,4)=0

c...sum up small momentum region. jet if enough absolute momentum.
      if(mstu(46).le.2) then
        do 180 j=1,4
  180   p(n+1,j)=0.
        do 200 i=n+np+1,n+2*np
        if(p(i,5).gt.2.*rinit) goto 200
        nrem=nrem-1
        k(i,4)=1
        do 190 j=1,4
  190   p(n+1,j)=p(n+1,j)+p(i,j)
  200   continue
        p(n+1,5)=sqrt(p(n+1,1)**2+p(n+1,2)**2+p(n+1,3)**2)
        if(p(n+1,5).gt.2.*rinit) npre=1
        if(rinit.ge.0.2*paru(43).and.npre+nrem.lt.mstu(47)) goto 160
        if(nrem.eq.0) goto 160
      endif

c...find fastest remaining particle.
  210 npre=npre+1
      pmax=0.
      do 220 i=n+np+1,n+2*np
      if(k(i,4).ne.0.or.p(i,5).le.pmax) goto 220
      imax=i
      pmax=p(i,5)
  220 continue
      do 230 j=1,5
  230 p(n+npre,j)=p(imax,j)
      nrem=nrem-1
      k(imax,4)=npre

c...sum up precluster around it according to pt separation.
      if(mstu(46).le.2) then
        do 250 i=n+np+1,n+2*np
        if(k(i,4).ne.0) goto 250
        r2=r2t(i,imax)
        if(r2.gt.rinit**2) goto 250
        nrem=nrem-1
        k(i,4)=npre
        do 240 j=1,4
  240   p(n+npre,j)=p(n+npre,j)+p(i,j)
  250   continue
        p(n+npre,5)=sqrt(p(n+npre,1)**2+p(n+npre,2)**2+p(n+npre,3)**2)

c...sum up precluster around it according to mass separation.
      else
  260   imin=0
        r2min=rinit**2
        do 270 i=n+np+1,n+2*np
        if(k(i,4).ne.0) goto 270
        r2=r2m(i,n+npre)
        if(r2.ge.r2min) goto 270
        imin=i
        r2min=r2
  270   continue
        if(imin.ne.0) then
          do 280 j=1,4
  280     p(n+npre,j)=p(n+npre,j)+p(imin,j)
          p(n+npre,5)=sqrt(p(n+npre,1)**2+p(n+npre,2)**2+p(n+npre,3)**2)
          nrem=nrem-1
          k(imin,4)=npre
          goto 260
        endif
      endif

c...check if more preclusters to be found. start over if too few.
      if(rinit.ge.0.2*paru(43).and.npre+nrem.lt.mstu(47)) goto 160
      if(nrem.gt.0) goto 210
      njet=npre

c...reassign all particles to nearest jet. sum up new jet momenta.
  290 tsav=0.
      psjt=0.
  300 if(mstu(46).le.1) then
        do 310 i=n+1,n+njet
        do 310 j=1,4
  310   v(i,j)=0.
        do 340 i=n+np+1,n+2*np
        r2min=pss**2
        do 320 ijet=n+1,n+njet
        if(p(ijet,5).lt.rinit) goto 320
        r2=r2t(i,ijet)
        if(r2.ge.r2min) goto 320
        imin=ijet
        r2min=r2
  320   continue
        k(i,4)=imin-n
        do 330 j=1,4
  330   v(imin,j)=v(imin,j)+p(i,j)
  340   continue
        psjt=0.
        do 360 i=n+1,n+njet
        do 350 j=1,4
  350   p(i,j)=v(i,j)
        p(i,5)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)
  360   psjt=psjt+p(i,5)
      endif

c...find two closest jets.
      r2min=2.*r2acc
      do 370 itry1=n+1,n+njet-1
      do 370 itry2=itry1+1,n+njet
      if(mstu(46).le.2) r2=r2t(itry1,itry2)
      if(mstu(46).ge.3) r2=r2m(itry1,itry2)
      if(r2.ge.r2min) goto 370
      imin1=itry1
      imin2=itry2
      r2min=r2
  370 continue

c...if allowed, join two closest jets and start over.
      if(njet.gt.mstu(47).and.r2min.lt.r2acc) then
        irec=min(imin1,imin2)
        idel=max(imin1,imin2)
        do 380 j=1,4
  380   p(irec,j)=p(imin1,j)+p(imin2,j)
        p(irec,5)=sqrt(p(irec,1)**2+p(irec,2)**2+p(irec,3)**2)
        do 390 i=idel+1,n+njet
        do 390 j=1,5
  390   p(i-1,j)=p(i,j)
        if(mstu(46).ge.2) then
          do 400 i=n+np+1,n+2*np
          iori=n+k(i,4)
          if(iori.eq.idel) k(i,4)=irec-n
  400     if(iori.gt.idel) k(i,4)=k(i,4)-1
        endif
        njet=njet-1
        goto 290

c...divide up broad jet if empty cluster in list of final ones.
      elseif(njet.eq.mstu(47).and.mstu(46).le.1.and.nloop.le.2) then
        do 410 i=n+1,n+njet
  410   k(i,5)=0
        do 420 i=n+np+1,n+2*np
  420   k(n+k(i,4),5)=k(n+k(i,4),5)+1
        iemp=0
        do 430 i=n+1,n+njet
  430   if(k(i,5).eq.0) iemp=i
        if(iemp.ne.0) then
          nloop=nloop+1
          ispl=0
          r2max=0.
          do 440 i=n+np+1,n+2*np
          if(k(n+k(i,4),5).le.1.or.p(i,5).lt.rinit) goto 440
          ijet=n+k(i,4)
          r2=r2t(i,ijet)
          if(r2.le.r2max) goto 440
          ispl=i
          r2max=r2
  440     continue
          if(ispl.ne.0) then
            ijet=n+k(ispl,4)
            do 450 j=1,4
            p(iemp,j)=p(ispl,j)
  450       p(ijet,j)=p(ijet,j)-p(ispl,j)
            p(iemp,5)=p(ispl,5)
            p(ijet,5)=sqrt(p(ijet,1)**2+p(ijet,2)**2+p(ijet,3)**2)
            if(nloop.le.2) goto 290
          endif
        endif
      endif

c...if generalized thrust has not yet converged, continue iteration.
      if(mstu(46).le.1.and.nloop.le.2.and.psjt/pss.gt.tsav+paru(48))
     &then
        tsav=psjt/pss
        goto 300
      endif

c...reorder jets according to energy.
      do 460 i=n+1,n+njet
      do 460 j=1,5
  460 v(i,j)=p(i,j)
      do 490 inew=n+1,n+njet
      pemax=0.
      do 470 itry=n+1,n+njet
      if(v(itry,4).le.pemax) goto 470
      imax=itry
      pemax=v(itry,4)
  470 continue
      k(inew,1)=31
      k(inew,2)=97
      k(inew,3)=inew-n
      k(inew,4)=0
      do 480 j=1,5
  480 p(inew,j)=v(imax,j)
      v(imax,4)=-1.
  490 k(imax,5)=inew

c...clean up particle-jet assignments and jet information.
      do 500 i=n+np+1,n+2*np
      iori=k(n+k(i,4),5)
      k(i,4)=iori-n
      if(k(k(i,3),1).ne.3) k(k(i,3),4)=iori-n
      k(iori,4)=k(iori,4)+1
  500 continue
      iemp=0
      psjt=0.
      do 520 i=n+1,n+njet
      k(i,5)=0
      psjt=psjt+p(i,5)
      p(i,5)=sqrt(max(p(i,4)**2-p(i,5)**2,0.))
      do 510 j=1,5
  510 v(i,j)=0.
  520 if(k(i,4).eq.0) iemp=i

c...select storing option. output variables. check for failure.
      mstu(61)=n+1
      mstu(62)=np
      mstu(63)=npre
      paru(61)=ps(5)
      paru(62)=psjt/pss
      paru(63)=sqrt(r2min)
      if(njet.le.1) paru(63)=0.
      if(iemp.ne.0) then
        call luerrm(8,'(luclus:) failed to reconstruct as requested')
        njet=-1
      endif
      if(mstu(43).le.1) mstu(3)=njet
      if(mstu(43).ge.2) n=n+njet
      nsav=njet

      return
      end

c*********************************************************************

      subroutine lucell(njet)

c...purpose: to provide a simple way of jet finding in an eta-phi-et
c...coordinate frame, as used for calorimeters at hadron colliders.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...loop over all particles. find cell that was hit by given particle.
      ptlrat=1./sinh(paru(51))**2
      np=0
      nc=n
      do 110 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 110
      if(p(i,1)**2+p(i,2)**2.le.ptlrat*p(i,3)**2) goto 110
      if(mstu(41).ge.2) then
        kc=lucomp(k(i,2))
        if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &  kc.eq.18) goto 110
        if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &  goto 110
      endif
      np=np+1
      pt=sqrt(p(i,1)**2+p(i,2)**2)
      eta=sign(log((sqrt(pt**2+p(i,3)**2)+abs(p(i,3)))/pt),p(i,3))
      ieta=max(1,min(mstu(51),1+int(mstu(51)*0.5*(eta/paru(51)+1.))))
      phi=ulangl(p(i,1),p(i,2))
      iphi=max(1,min(mstu(52),1+int(mstu(52)*0.5*(phi/paru(1)+1.))))
      ietph=mstu(52)*ieta+iphi

c...add to cell already hit, or book new cell.
      do 100 ic=n+1,nc
      if(ietph.eq.k(ic,3)) then
        k(ic,4)=k(ic,4)+1
        p(ic,5)=p(ic,5)+pt
        goto 110
      endif
  100 continue
      if(nc.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lucell:) no more memory left in lujets')
        njet=-2
        return
      endif
      nc=nc+1
      k(nc,3)=ietph
      k(nc,4)=1
      k(nc,5)=2
      p(nc,1)=(paru(51)/mstu(51))*(2*ieta-1-mstu(51))
      p(nc,2)=(paru(1)/mstu(52))*(2*iphi-1-mstu(52))
      p(nc,5)=pt
  110 continue

c...smear true bin content by calorimeter resolution.
      if(mstu(53).ge.1) then
        do 130 ic=n+1,nc
        pei=p(ic,5)
        if(mstu(53).eq.2) pei=p(ic,5)/cosh(p(ic,1))
  120   pef=pei+paru(55)*sqrt(-2.*log(max(1e-10,rlu(0)))*pei)*
     &  cos(paru(2)*rlu(0))
        if(pef.lt.0..or.pef.gt.paru(56)*pei) goto 120
        p(ic,5)=pef
  130   if(mstu(53).eq.2) p(ic,5)=pef*cosh(p(ic,1))
      endif

c...find initiator cell: the one with highest pt of not yet used ones.
      nj=nc
  140 etmax=0.
      do 150 ic=n+1,nc
      if(k(ic,5).ne.2) goto 150
      if(p(ic,5).le.etmax) goto 150
      icmax=ic
      eta=p(ic,1)
      phi=p(ic,2)
      etmax=p(ic,5)
  150 continue
      if(etmax.lt.paru(52)) goto 210
      if(nj.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lucell:) no more memory left in lujets')
        njet=-2
        return
      endif
      k(icmax,5)=1
      nj=nj+1
      k(nj,4)=0
      k(nj,5)=1
      p(nj,1)=eta
      p(nj,2)=phi
      p(nj,3)=0.
      p(nj,4)=0.
      p(nj,5)=0.

c...sum up unused cells within required distance of initiator.
      do 160 ic=n+1,nc
      if(k(ic,5).eq.0) goto 160
      if(abs(p(ic,1)-eta).gt.paru(54)) goto 160
      dphia=abs(p(ic,2)-phi)
      if(dphia.gt.paru(54).and.dphia.lt.paru(2)-paru(54)) goto 160
      phic=p(ic,2)
      if(dphia.gt.paru(1)) phic=phic+sign(paru(2),phi)
      if((p(ic,1)-eta)**2+(phic-phi)**2.gt.paru(54)**2) goto 160
      k(ic,5)=-k(ic,5)
      k(nj,4)=k(nj,4)+k(ic,4)
      p(nj,3)=p(nj,3)+p(ic,5)*p(ic,1)
      p(nj,4)=p(nj,4)+p(ic,5)*phic
      p(nj,5)=p(nj,5)+p(ic,5)
  160 continue

c...reject cluster below minimum et, else accept.
      if(p(nj,5).lt.paru(53)) then
        nj=nj-1
        do 170 ic=n+1,nc
  170   if(k(ic,5).lt.0) k(ic,5)=-k(ic,5)
      elseif(mstu(54).le.2) then
        p(nj,3)=p(nj,3)/p(nj,5)
        p(nj,4)=p(nj,4)/p(nj,5)
        if(abs(p(nj,4)).gt.paru(1)) p(nj,4)=p(nj,4)-sign(paru(2),
     &  p(nj,4))
        do 180 ic=n+1,nc
  180   if(k(ic,5).lt.0) k(ic,5)=0
      else
        do 190 j=1,4
  190   p(nj,j)=0.
        do 200 ic=n+1,nc
        if(k(ic,5).ge.0) goto 200
        p(nj,1)=p(nj,1)+p(ic,5)*cos(p(ic,2))
        p(nj,2)=p(nj,2)+p(ic,5)*sin(p(ic,2))
        p(nj,3)=p(nj,3)+p(ic,5)*sinh(p(ic,1))
        p(nj,4)=p(nj,4)+p(ic,5)*cosh(p(ic,1))
        k(ic,5)=0
  200   continue
      endif
      goto 140

c...arrange clusters in falling et sequence.
  210 do 230 i=1,nj-nc
      etmax=0.
      do 220 ij=nc+1,nj
      if(k(ij,5).eq.0) goto 220
      if(p(ij,5).lt.etmax) goto 220
      ijmax=ij
      etmax=p(ij,5)
  220 continue
      k(ijmax,5)=0
      k(n+i,1)=31
      k(n+i,2)=98
      k(n+i,3)=i
      k(n+i,4)=k(ijmax,4)
      k(n+i,5)=0
      do 230 j=1,5
      p(n+i,j)=p(ijmax,j)
  230 v(n+i,j)=0.
      njet=nj-nc

c...convert to massless or massive four-vectors.
      if(mstu(54).eq.2) then
        do 240 i=n+1,n+njet
        eta=p(i,3)
        p(i,1)=p(i,5)*cos(p(i,4))
        p(i,2)=p(i,5)*sin(p(i,4))
        p(i,3)=p(i,5)*sinh(eta)
        p(i,4)=p(i,5)*cosh(eta)
  240   p(i,5)=0.
      elseif(mstu(54).ge.3) then
        do 250 i=n+1,n+njet
  250   p(i,5)=sqrt(max(0.,p(i,4)**2-p(i,1)**2-p(i,2)**2-p(i,3)**2))
      endif

c...information about storage.
      mstu(61)=n+1
      mstu(62)=np
      mstu(63)=nc-n
      if(mstu(43).le.1) mstu(3)=njet
      if(mstu(43).ge.2) n=n+njet

      return
      end

c*********************************************************************

      subroutine lujmas(pmh,pml)

c...purpose: to determine, approximately, the two jet masses that
c...minimize the sum m_h^2 + m_l^2, a la clavelli and wyler.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/
      dimension sm(3,3),sax(3),ps(3,5)

c...reset.
      np=0
      do 110 j1=1,3
      do 100 j2=j1,3
  100 sm(j1,j2)=0.
      do 110 j2=1,4
  110 ps(j1,j2)=0.
      pss=0.

c...take copy of particles that are to be considered in mass analysis.
      do 150 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 150
      if(mstu(41).ge.2) then
        kc=lucomp(k(i,2))
        if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &  kc.eq.18) goto 150
        if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &  goto 150
      endif
      if(n+np+1.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lujmas:) no more memory left in lujets')
        pmh=-2.
        pml=-2.
        return
      endif
      np=np+1
      do 120 j=1,5
  120 p(n+np,j)=p(i,j)
      if(mstu(42).eq.0) p(n+np,5)=0.
      if(mstu(42).eq.1.and.k(i,2).ne.22) p(n+np,5)=pmas(101,1)
      p(n+np,4)=sqrt(p(n+np,5)**2+p(i,1)**2+p(i,2)**2+p(i,3)**2)

c...fill information in sphericity tensor and total momentum vector.
      do 130 j1=1,3
      do 130 j2=j1,3
  130 sm(j1,j2)=sm(j1,j2)+p(i,j1)*p(i,j2)
      pss=pss+(p(i,1)**2+p(i,2)**2+p(i,3)**2)
      do 140 j=1,4
  140 ps(3,j)=ps(3,j)+p(n+np,j)
  150 continue

c...very low multiplicities (0 or 1) not considered.
      if(np.le.1) then
        call luerrm(8,'(lujmas:) too few particles for analysis')
        pmh=-1.
        pml=-1.
        return
      endif
      paru(61)=sqrt(max(0.,ps(3,4)**2-ps(3,1)**2-ps(3,2)**2-ps(3,3)**2))

c...find largest eigenvalue to matrix (third degree equation).
      do 160 j1=1,3
      do 160 j2=j1,3
  160 sm(j1,j2)=sm(j1,j2)/pss
      sq=(sm(1,1)*sm(2,2)+sm(1,1)*sm(3,3)+sm(2,2)*sm(3,3)-sm(1,2)**2-
     &sm(1,3)**2-sm(2,3)**2)/3.-1./9.
      sr=-0.5*(sq+1./9.+sm(1,1)*sm(2,3)**2+sm(2,2)*sm(1,3)**2+sm(3,3)*
     &sm(1,2)**2-sm(1,1)*sm(2,2)*sm(3,3))+sm(1,2)*sm(1,3)*sm(2,3)+1./27.
      sp=cos(acos(max(min(sr/sqrt(-sq**3),1.),-1.))/3.)
      sma=1./3.+sqrt(-sq)*max(2.*sp,sqrt(3.*(1.-sp**2))-sp)

c...find largest eigenvector by solving equation system.
      do 170 j1=1,3
      sm(j1,j1)=sm(j1,j1)-sma
      do 170 j2=j1+1,3
  170 sm(j2,j1)=sm(j1,j2)
      smax=0.
      do 180 j1=1,3
      do 180 j2=1,3
      if(abs(sm(j1,j2)).le.smax) goto 180
      ja=j1
      jb=j2
      smax=abs(sm(j1,j2))
  180 continue
      smax=0.
      do 190 j3=ja+1,ja+2
      j1=j3-3*((j3-1)/3)
      rl=sm(j1,jb)/sm(ja,jb)
      do 190 j2=1,3
      sm(j1,j2)=sm(j1,j2)-rl*sm(ja,j2)
      if(abs(sm(j1,j2)).le.smax) goto 190
      jc=j1
      smax=abs(sm(j1,j2))
  190 continue
      jb1=jb+1-3*(jb/3)
      jb2=jb+2-3*((jb+1)/3)
      sax(jb1)=-sm(jc,jb2)
      sax(jb2)=sm(jc,jb1)
      sax(jb)=-(sm(ja,jb1)*sax(jb1)+sm(ja,jb2)*sax(jb2))/sm(ja,jb)

c...divide particles into two initial clusters by hemisphere.
      do 200 i=n+1,n+np
      psax=p(i,1)*sax(1)+p(i,2)*sax(2)+p(i,3)*sax(3)
      is=1
      if(psax.lt.0.) is=2
      k(i,3)=is
      do 200 j=1,4
  200 ps(is,j)=ps(is,j)+p(i,j)
      pms=max(1e-10,ps(1,4)**2-ps(1,1)**2-ps(1,2)**2-ps(1,3)**2)+
     &max(1e-10,ps(2,4)**2-ps(2,1)**2-ps(2,2)**2-ps(2,3)**2)

c...reassign one particle at a time; find maximum decrease of m^2 sum.
  210 pmd=0.
      im=0
      do 220 j=1,4
  220 ps(3,j)=ps(1,j)-ps(2,j)
      do 230 i=n+1,n+np
      pps=p(i,4)*ps(3,4)-p(i,1)*ps(3,1)-p(i,2)*ps(3,2)-p(i,3)*ps(3,3)
      if(k(i,3).eq.1) pmdi=2.*(p(i,5)**2-pps)
      if(k(i,3).eq.2) pmdi=2.*(p(i,5)**2+pps)
      if(pmdi.lt.pmd) then
        pmd=pmdi
        im=i
      endif
  230 continue

c...loop back if significant reduction in sum of m^2.
      if(pmd.lt.-paru(48)*pms) then
        pms=pms+pmd
        is=k(im,3)
        do 240 j=1,4
        ps(is,j)=ps(is,j)-p(im,j)
  240   ps(3-is,j)=ps(3-is,j)+p(im,j)
        k(im,3)=3-is
        goto 210
      endif

c...final masses and output.
      mstu(61)=n+1
      mstu(62)=np
      ps(1,5)=sqrt(max(0.,ps(1,4)**2-ps(1,1)**2-ps(1,2)**2-ps(1,3)**2))
      ps(2,5)=sqrt(max(0.,ps(2,4)**2-ps(2,1)**2-ps(2,2)**2-ps(2,3)**2))
      pmh=max(ps(1,5),ps(2,5))
      pml=min(ps(1,5),ps(2,5))

      return
      end

c*********************************************************************

      subroutine lufowo(h10,h20,h30,h40)

c...purpose: to calculate the first few fox-wolfram moments.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...copy momenta for particles and calculate h0.
      np=0
      h0=0.
      hd=0.
      do 110 i=1,n
      if(k(i,1).le.0.or.k(i,1).gt.10) goto 110
      if(mstu(41).ge.2) then
        kc=lucomp(k(i,2))
        if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &  kc.eq.18) goto 110
        if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &  goto 110
      endif
      if(n+np.ge.mstu(4)-mstu(32)-5) then
        call luerrm(11,'(lufowo:) no more memory left in lujets')
        h10=-1.
        h20=-1.
        h30=-1.
        h40=-1.
        return
      endif
      np=np+1
      do 100 j=1,3
  100 p(n+np,j)=p(i,j)
      p(n+np,4)=sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2)
      h0=h0+p(n+np,4)
      hd=hd+p(n+np,4)**2
  110 continue
      h0=h0**2

c...very low multiplicities (0 or 1) not considered.
      if(np.le.1) then
        call luerrm(8,'(lufowo:) too few particles for analysis')
        h10=-1.
        h20=-1.
        h30=-1.
        h40=-1.
        return
      endif

c...calculate h1 - h4.
      h10=0.
      h20=0.
      h30=0.
      h40=0.
      do 120 i1=n+1,n+np
      do 120 i2=i1+1,n+np
      cthe=(p(i1,1)*p(i2,1)+p(i1,2)*p(i2,2)+p(i1,3)*p(i2,3))/
     &(p(i1,4)*p(i2,4))
      h10=h10+p(i1,4)*p(i2,4)*cthe
      h20=h20+p(i1,4)*p(i2,4)*(1.5*cthe**2-0.5)
      h30=h30+p(i1,4)*p(i2,4)*(2.5*cthe**3-1.5*cthe)
      h40=h40+p(i1,4)*p(i2,4)*(4.375*cthe**4-3.75*cthe**2+0.375)
  120 continue

c...calculate h1/h0 - h4/h0. output.
      mstu(61)=n+1
      mstu(62)=np
      h10=(hd+2.*h10)/h0
      h20=(hd+2.*h20)/h0
      h30=(hd+2.*h30)/h0
      h40=(hd+2.*h40)/h0

      return
      end

c*********************************************************************

      subroutine lutabu(mtabu)

c...purpose: to evaluate various properties of an event, with
c...statistics accumulated during the course of the run and
c...printed at the end.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      save /lujets/,/ludat1/,/ludat2/,/ludat3/
      dimension kfis(100,2),npis(100,0:10),kffs(400),npfs(400,4),
     &fevfm(10,4),fm1fm(3,10,4),fm2fm(3,10,4),fmoma(4),fmoms(4),
     &fevee(50),fe1ec(50),fe2ec(50),fe1ea(25),fe2ea(25),
     &kfdm(8),kfdc(200,0:8),npdc(200)
      save nevis,nkfis,kfis,npis,nevfs,nprfs,nfifs,nchfs,nkffs,
     &kffs,npfs,nevfm,nmufm,fm1fm,fm2fm,nevee,fe1ec,fe2ec,fe1ea,
     &fe2ea,nevdc,nkfdc,nredc,kfdc,npdc
      character chau*16,chis(2)*12,chdc(8)*12
      data nevis/0/,nkfis/0/,nevfs/0/,nprfs/0/,nfifs/0/,nchfs/0/,
     &nkffs/0/,nevfm/0/,nmufm/0/,fm1fm/120*0./,fm2fm/120*0./,
     &nevee/0/,fe1ec/50*0./,fe2ec/50*0./,fe1ea/25*0./,fe2ea/25*0./,
     &nevdc/0/,nkfdc/0/,nredc/0/

c...reset statistics on initial parton state.
      if(mtabu.eq.10) then
        nevis=0
        nkfis=0

c...identify and order flavour content of initial state.
      elseif(mtabu.eq.11) then
        nevis=nevis+1
        kfm1=2*iabs(mstu(161))
        if(mstu(161).gt.0) kfm1=kfm1-1
        kfm2=2*iabs(mstu(162))
        if(mstu(162).gt.0) kfm2=kfm2-1
        kfmn=min(kfm1,kfm2)
        kfmx=max(kfm1,kfm2)
        do 100 i=1,nkfis
        if(kfmn.eq.kfis(i,1).and.kfmx.eq.kfis(i,2)) then
          ikfis=-i
          goto 110
        elseif(kfmn.lt.kfis(i,1).or.(kfmn.eq.kfis(i,1).and.
     &  kfmx.lt.kfis(i,2))) then
          ikfis=i
          goto 110
        endif
  100   continue
        ikfis=nkfis+1
  110   if(ikfis.lt.0) then
          ikfis=-ikfis
        else
          if(nkfis.ge.100) return
          do 120 i=nkfis,ikfis,-1
          kfis(i+1,1)=kfis(i,1)
          kfis(i+1,2)=kfis(i,2)
          do 120 j=0,10
  120     npis(i+1,j)=npis(i,j)
          nkfis=nkfis+1
          kfis(ikfis,1)=kfmn
          kfis(ikfis,2)=kfmx
          do 130 j=0,10
  130     npis(ikfis,j)=0
        endif
        npis(ikfis,0)=npis(ikfis,0)+1

c...count number of partons in initial state.
        np=0
        do 150 i=1,n
        if(k(i,1).le.0.or.k(i,1).gt.12) then
        elseif(iabs(k(i,2)).gt.80.and.iabs(k(i,2)).le.100) then
        elseif(iabs(k(i,2)).gt.100.and.mod(iabs(k(i,2))/10,10).ne.0)
     &  then
        else
          im=i
  140     im=k(im,3)
          if(im.le.0.or.im.gt.n) then
            np=np+1
          elseif(k(im,1).le.0.or.k(im,1).gt.20) then
            np=np+1
          elseif(iabs(k(im,2)).gt.80.and.iabs(k(im,2)).le.100) then
          elseif(iabs(k(im,2)).gt.100.and.mod(iabs(k(im,2))/10,10).ne.0)
     &    then
          else
            goto 140
          endif
        endif
  150   continue
        npco=max(np,1)
        if(np.ge.6) npco=6
        if(np.ge.8) npco=7
        if(np.ge.11) npco=8
        if(np.ge.16) npco=9
        if(np.ge.26) npco=10
        npis(ikfis,npco)=npis(ikfis,npco)+1
        mstu(62)=np

c...write statistics on initial parton state.
      elseif(mtabu.eq.12) then
        fac=1./max(1,nevis)
        write(mstu(11),5000) nevis
        do 160 i=1,nkfis
        kfmn=kfis(i,1)
        if(kfmn.eq.0) kfmn=kfis(i,2)
        kfm1=(kfmn+1)/2
        if(2*kfm1.eq.kfmn) kfm1=-kfm1
        call luname(kfm1,chau)
        chis(1)=chau(1:12)
        if(chau(13:13).ne.' ') chis(1)(12:12)='?'
        kfmx=kfis(i,2)
        if(kfis(i,1).eq.0) kfmx=0
        kfm2=(kfmx+1)/2
        if(2*kfm2.eq.kfmx) kfm2=-kfm2
        call luname(kfm2,chau)
        chis(2)=chau(1:12)
        if(chau(13:13).ne.' ') chis(2)(12:12)='?'
  160   write(mstu(11),5100) chis(1),chis(2),fac*npis(i,0),
     &  (npis(i,j)/float(npis(i,0)),j=1,10)

c...copy statistics on initial parton state into /lujets/.
      elseif(mtabu.eq.13) then
        fac=1./max(1,nevis)
        do 170 i=1,nkfis
        kfmn=kfis(i,1)
        if(kfmn.eq.0) kfmn=kfis(i,2)
        kfm1=(kfmn+1)/2
        if(2*kfm1.eq.kfmn) kfm1=-kfm1
        kfmx=kfis(i,2)
        if(kfis(i,1).eq.0) kfmx=0
        kfm2=(kfmx+1)/2
        if(2*kfm2.eq.kfmx) kfm2=-kfm2
        k(i,1)=32
        k(i,2)=99
        k(i,3)=kfm1
        k(i,4)=kfm2
        k(i,5)=npis(i,0)
        do 170 j=1,5
        p(i,j)=fac*npis(i,j)
  170   v(i,j)=fac*npis(i,j+5)
        n=nkfis
        do 180 j=1,5
        k(n+1,j)=0
        p(n+1,j)=0.
  180   v(n+1,j)=0.
        k(n+1,1)=32
        k(n+1,2)=99
        k(n+1,5)=nevis
        mstu(3)=1

c...reset statistics on number of particles/partons.
      elseif(mtabu.eq.20) then
        nevfs=0
        nprfs=0
        nfifs=0
        nchfs=0
        nkffs=0

c...identify whether particle/parton is primary or not.
      elseif(mtabu.eq.21) then
        nevfs=nevfs+1
        mstu(62)=0
        do 230 i=1,n
        if(k(i,1).le.0.or.k(i,1).gt.20.or.k(i,1).eq.13) goto 230
        mstu(62)=mstu(62)+1
        kc=lucomp(k(i,2))
        mpri=0
        if(k(i,3).le.0.or.k(i,3).gt.n) then
          mpri=1
        elseif(k(k(i,3),1).le.0.or.k(k(i,3),1).gt.20) then
          mpri=1
        elseif(k(k(i,3),2).ge.91.and.k(k(i,3),2).le.93) then
          mpri=1
        elseif(kc.eq.0) then
        elseif(k(k(i,3),1).eq.13) then
          im=k(k(i,3),3)
          if(im.le.0.or.im.gt.n) then
            mpri=1
          elseif(k(im,1).le.0.or.k(im,1).gt.20) then
            mpri=1
          endif
        elseif(kchg(kc,2).eq.0) then
          kcm=lucomp(k(k(i,3),2))
          if(kcm.ne.0) then
            if(kchg(kcm,2).ne.0) mpri=1
          endif
        endif
        if(kc.ne.0.and.mpri.eq.1) then
          if(kchg(kc,2).eq.0) nprfs=nprfs+1
        endif
        if(k(i,1).le.10) then
          nfifs=nfifs+1
          if(luchge(k(i,2)).ne.0) nchfs=nchfs+1
        endif

c...fill statistics on number of particles/partons in event.
        kfa=iabs(k(i,2))
        kfs=3-isign(1,k(i,2))-mpri
        do 190 ip=1,nkffs
        if(kfa.eq.kffs(ip)) then
          ikffs=-ip
          goto 200
        elseif(kfa.lt.kffs(ip)) then
          ikffs=ip
          goto 200
        endif
  190   continue
        ikffs=nkffs+1
  200   if(ikffs.lt.0) then
          ikffs=-ikffs
        else
          if(nkffs.ge.400) return
          do 210 ip=nkffs,ikffs,-1
          kffs(ip+1)=kffs(ip)
          do 210 j=1,4
  210     npfs(ip+1,j)=npfs(ip,j)
          nkffs=nkffs+1
          kffs(ikffs)=kfa
          do 220 j=1,4
  220     npfs(ikffs,j)=0
        endif
        npfs(ikffs,kfs)=npfs(ikffs,kfs)+1
  230   continue

c...write statistics on particle/parton composition of events.
      elseif(mtabu.eq.22) then
        fac=1./max(1,nevfs)
        write(mstu(11),5200) nevfs,fac*nprfs,fac*nfifs,fac*nchfs
        do 240 i=1,nkffs
        call luname(kffs(i),chau)
        kc=lucomp(kffs(i))
        mdcyf=0
        if(kc.ne.0) mdcyf=mdcy(kc,1)
  240   write(mstu(11),5300) kffs(i),chau,mdcyf,(fac*npfs(i,j),j=1,4),
     &  fac*(npfs(i,1)+npfs(i,2)+npfs(i,3)+npfs(i,4))

c...copy particle/parton composition information into /lujets/.
      elseif(mtabu.eq.23) then
        fac=1./max(1,nevfs)
        do 260 i=1,nkffs
        k(i,1)=32
        k(i,2)=99
        k(i,3)=kffs(i)
        k(i,4)=0
        k(i,5)=npfs(i,1)+npfs(i,2)+npfs(i,3)+npfs(i,4)
        do 250 j=1,4
        p(i,j)=fac*npfs(i,j)
  250   v(i,j)=0.
        p(i,5)=fac*k(i,5)
  260   v(i,5)=0.
        n=nkffs
        do 270 j=1,5
        k(n+1,j)=0
        p(n+1,j)=0.
  270   v(n+1,j)=0.
        k(n+1,1)=32
        k(n+1,2)=99
        k(n+1,5)=nevfs
        p(n+1,1)=fac*nprfs
        p(n+1,2)=fac*nfifs
        p(n+1,3)=fac*nchfs
        mstu(3)=1

c...reset factorial moments statistics.
      elseif(mtabu.eq.30) then
        nevfm=0
        nmufm=0
        do 280 im=1,3
        do 280 ib=1,10
        do 280 ip=1,4
        fm1fm(im,ib,ip)=0.
  280   fm2fm(im,ib,ip)=0.

c...find particles to include, with (pion,pseudo)rapidity and azimuth.
      elseif(mtabu.eq.31) then
        nevfm=nevfm+1
        nlow=n+mstu(3)
        nupp=nlow
        do 360 i=1,n
        if(k(i,1).le.0.or.k(i,1).gt.10) goto 360
        if(mstu(41).ge.2) then
          kc=lucomp(k(i,2))
          if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &    kc.eq.18) goto 360
          if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &    goto 360
        endif
        pmr=0.
        if(mstu(42).eq.1.and.k(i,2).ne.22) pmr=ulmass(211)
        if(mstu(42).ge.2) pmr=p(i,5)
        pr=max(1e-20,pmr**2+p(i,1)**2+p(i,2)**2)
        yeta=sign(log(min((sqrt(pr+p(i,3)**2)+abs(p(i,3)))/sqrt(pr),
     &  1e20)),p(i,3))
        if(abs(yeta).gt.paru(57)) goto 360
        phi=ulangl(p(i,1),p(i,2))
        iyeta=512.*(yeta+paru(57))/(2.*paru(57))
        iyeta=max(0,min(511,iyeta))
        iphi=512.*(phi+paru(1))/paru(2)
        iphi=max(0,min(511,iphi))
        iyep=0
        do 290 ib=0,9
  290   iyep=iyep+4**ib*(2*mod(iyeta/2**ib,2)+mod(iphi/2**ib,2))

c...order particles in (pseudo)rapidity and/or azimuth.
        if(nupp.gt.mstu(4)-5-mstu(32)) then
          call luerrm(11,'(lutabu:) no more memory left in lujets')
          return
        endif
        nupp=nupp+1
        if(nupp.eq.nlow+1) then
          k(nupp,1)=iyeta
          k(nupp,2)=iphi
          k(nupp,3)=iyep
        else
          do 300 i1=nupp-1,nlow+1,-1
          if(iyeta.ge.k(i1,1)) goto 310
  300     k(i1+1,1)=k(i1,1)
  310     k(i1+1,1)=iyeta
          do 320 i1=nupp-1,nlow+1,-1
          if(iphi.ge.k(i1,2)) goto 330
  320     k(i1+1,2)=k(i1,2)
  330     k(i1+1,2)=iphi
          do 340 i1=nupp-1,nlow+1,-1
          if(iyep.ge.k(i1,3)) goto 350
  340     k(i1+1,3)=k(i1,3)
  350     k(i1+1,3)=iyep
        endif
  360   continue
        k(nupp+1,1)=2**10
        k(nupp+1,2)=2**10
        k(nupp+1,3)=4**10

c...calculate sum of factorial moments in event.
        do 400 im=1,3
        do 370 ib=1,10
        do 370 ip=1,4
  370   fevfm(ib,ip)=0.
        do 380 ib=1,10
        if(im.le.2) ibin=2**(10-ib)
        if(im.eq.3) ibin=4**(10-ib)
        iagr=k(nlow+1,im)/ibin
        nagr=1
        do 380 i=nlow+2,nupp+1
        icut=k(i,im)/ibin
        if(icut.eq.iagr) then
          nagr=nagr+1
        else
          if(nagr.eq.1) then
          elseif(nagr.eq.2) then
            fevfm(ib,1)=fevfm(ib,1)+2.
          elseif(nagr.eq.3) then
            fevfm(ib,1)=fevfm(ib,1)+6.
            fevfm(ib,2)=fevfm(ib,2)+6.
          elseif(nagr.eq.4) then
            fevfm(ib,1)=fevfm(ib,1)+12.
            fevfm(ib,2)=fevfm(ib,2)+24.
            fevfm(ib,3)=fevfm(ib,3)+24.
          else
            fevfm(ib,1)=fevfm(ib,1)+nagr*(nagr-1.)
            fevfm(ib,2)=fevfm(ib,2)+nagr*(nagr-1.)*(nagr-2.)
            fevfm(ib,3)=fevfm(ib,3)+nagr*(nagr-1.)*(nagr-2.)*(nagr-3.)
            fevfm(ib,4)=fevfm(ib,4)+nagr*(nagr-1.)*(nagr-2.)*(nagr-3.)*
     &      (nagr-4.)
          endif
          iagr=icut
          nagr=1
        endif
  380   continue

c...add results to total statistics.
        do 390 ib=10,1,-1
        do 390 ip=1,4
        if(fevfm(1,ip).lt.0.5) then
          fevfm(ib,ip)=0.
        elseif(im.le.2) then
          fevfm(ib,ip)=2**((ib-1)*ip)*fevfm(ib,ip)/fevfm(1,ip)
        else
          fevfm(ib,ip)=4**((ib-1)*ip)*fevfm(ib,ip)/fevfm(1,ip)
        endif
        fm1fm(im,ib,ip)=fm1fm(im,ib,ip)+fevfm(ib,ip)
  390   fm2fm(im,ib,ip)=fm2fm(im,ib,ip)+fevfm(ib,ip)**2
  400   continue
        nmufm=nmufm+(nupp-nlow)
        mstu(62)=nupp-nlow

c...write accumulated statistics on factorial moments.
      elseif(mtabu.eq.32) then
        fac=1./max(1,nevfm)
        if(mstu(42).le.0) write(mstu(11),5400) nevfm,'eta'
        if(mstu(42).eq.1) write(mstu(11),5400) nevfm,'ypi'
        if(mstu(42).ge.2) write(mstu(11),5400) nevfm,'y  '
        do 420 im=1,3
        write(mstu(11),5500)
        do 420 ib=1,10
        byeta=2.*paru(57)
        if(im.ne.2) byeta=byeta/2**(ib-1)
        bphi=paru(2)
        if(im.ne.1) bphi=bphi/2**(ib-1)
        if(im.le.2) bnave=fac*nmufm/float(2**(ib-1))
        if(im.eq.3) bnave=fac*nmufm/float(4**(ib-1))
        do 410 ip=1,4
        fmoma(ip)=fac*fm1fm(im,ib,ip)
  410   fmoms(ip)=sqrt(max(0.,fac*(fac*fm2fm(im,ib,ip)-fmoma(ip)**2)))
  420   write(mstu(11),5600) byeta,bphi,bnave,(fmoma(ip),fmoms(ip),
     &  ip=1,4)

c...copy statistics on factorial moments into /lujets/.
      elseif(mtabu.eq.33) then
        fac=1./max(1,nevfm)
        do 430 im=1,3
        do 430 ib=1,10
        i=10*(im-1)+ib
        k(i,1)=32
        k(i,2)=99
        k(i,3)=1
        if(im.ne.2) k(i,3)=2**(ib-1)
        k(i,4)=1
        if(im.ne.1) k(i,4)=2**(ib-1)
        k(i,5)=0
        p(i,1)=2.*paru(57)/k(i,3)
        v(i,1)=paru(2)/k(i,4)
        do 430 ip=1,4
        p(i,ip+1)=fac*fm1fm(im,ib,ip)
  430   v(i,ip+1)=sqrt(max(0.,fac*(fac*fm2fm(im,ib,ip)-p(i,ip+1)**2)))
        n=30
        do 440 j=1,5
        k(n+1,j)=0
        p(n+1,j)=0.
  440   v(n+1,j)=0.
        k(n+1,1)=32
        k(n+1,2)=99
        k(n+1,5)=nevfm
        mstu(3)=1

c...reset statistics on energy-energy correlation.
      elseif(mtabu.eq.40) then
        nevee=0
        do 450 j=1,25
        fe1ec(j)=0.
        fe2ec(j)=0.
        fe1ec(51-j)=0.
        fe2ec(51-j)=0.
        fe1ea(j)=0.
  450   fe2ea(j)=0.

c...find particles to include, with proper assumed mass.
      elseif(mtabu.eq.41) then
        nevee=nevee+1
        nlow=n+mstu(3)
        nupp=nlow
        ecm=0.
        do 460 i=1,n
        if(k(i,1).le.0.or.k(i,1).gt.10) goto 460
        if(mstu(41).ge.2) then
          kc=lucomp(k(i,2))
          if(kc.eq.0.or.kc.eq.12.or.kc.eq.14.or.kc.eq.16.or.
     &    kc.eq.18) goto 460
          if(mstu(41).ge.3.and.kchg(kc,2).eq.0.and.luchge(k(i,2)).eq.0)
     &    goto 460
        endif
        pmr=0.
        if(mstu(42).eq.1.and.k(i,2).ne.22) pmr=ulmass(211)
        if(mstu(42).ge.2) pmr=p(i,5)
        if(nupp.gt.mstu(4)-5-mstu(32)) then
          call luerrm(11,'(lutabu:) no more memory left in lujets')
          return
        endif
        nupp=nupp+1
        p(nupp,1)=p(i,1)
        p(nupp,2)=p(i,2)
        p(nupp,3)=p(i,3)
        p(nupp,4)=sqrt(pmr**2+p(i,1)**2+p(i,2)**2+p(i,3)**2)
        p(nupp,5)=max(1e-10,sqrt(p(i,1)**2+p(i,2)**2+p(i,3)**2))
        ecm=ecm+p(nupp,4)
  460   continue
        if(nupp.eq.nlow) return

c...analyze energy-energy correlation in event.
        fac=(2./ecm**2)*50./paru(1)
        do 470 j=1,50
  470   fevee(j)=0.
        do 480 i1=nlow+2,nupp
        do 480 i2=nlow+1,i1-1
        cthe=(p(i1,1)*p(i2,1)+p(i1,2)*p(i2,2)+p(i1,3)*p(i2,3))/
     &  (p(i1,5)*p(i2,5))
        the=acos(max(-1.,min(1.,cthe)))
        ithe=max(1,min(50,1+int(50.*the/paru(1))))
  480   fevee(ithe)=fevee(ithe)+fac*p(i1,4)*p(i2,4)
        do 490 j=1,25
        fe1ec(j)=fe1ec(j)+fevee(j)
        fe2ec(j)=fe2ec(j)+fevee(j)**2
        fe1ec(51-j)=fe1ec(51-j)+fevee(51-j)
        fe2ec(51-j)=fe2ec(51-j)+fevee(51-j)**2
        fe1ea(j)=fe1ea(j)+(fevee(51-j)-fevee(j))
  490   fe2ea(j)=fe2ea(j)+(fevee(51-j)-fevee(j))**2
        mstu(62)=nupp-nlow

c...write statistics on energy-energy correlation.
      elseif(mtabu.eq.42) then
        fac=1./max(1,nevee)
        write(mstu(11),5700) nevee
        do 500 j=1,25
        feec1=fac*fe1ec(j)
        fees1=sqrt(max(0.,fac*(fac*fe2ec(j)-feec1**2)))
        feec2=fac*fe1ec(51-j)
        fees2=sqrt(max(0.,fac*(fac*fe2ec(51-j)-feec2**2)))
        feeca=fac*fe1ea(j)
        feesa=sqrt(max(0.,fac*(fac*fe2ea(j)-feeca**2)))
  500   write(mstu(11),5800) 3.6*(j-1),3.6*j,feec1,fees1,feec2,fees2,
     &  feeca,feesa

c...copy statistics on energy-energy correlation into /lujets/.
      elseif(mtabu.eq.43) then
        fac=1./max(1,nevee)
        do 510 i=1,25
        k(i,1)=32
        k(i,2)=99
        k(i,3)=0
        k(i,4)=0
        k(i,5)=0
        p(i,1)=fac*fe1ec(i)
        v(i,1)=sqrt(max(0.,fac*(fac*fe2ec(i)-p(i,1)**2)))
        p(i,2)=fac*fe1ec(51-i)
        v(i,2)=sqrt(max(0.,fac*(fac*fe2ec(51-i)-p(i,2)**2)))
        p(i,3)=fac*fe1ea(i)
        v(i,3)=sqrt(max(0.,fac*(fac*fe2ea(i)-p(i,3)**2)))
        p(i,4)=paru(1)*(i-1)/50.
        p(i,5)=paru(1)*i/50.
        v(i,4)=3.6*(i-1)
  510   v(i,5)=3.6*i
        n=25
        do 520 j=1,5
        k(n+1,j)=0
        p(n+1,j)=0.
  520   v(n+1,j)=0.
        k(n+1,1)=32
        k(n+1,2)=99
        k(n+1,5)=nevee
        mstu(3)=1

c...reset statistics on decay channels.
      elseif(mtabu.eq.50) then
        nevdc=0
        nkfdc=0
        nredc=0

c...identify and order flavour content of final state.
      elseif(mtabu.eq.51) then
        nevdc=nevdc+1
        nds=0
        do 550 i=1,n
        if(k(i,1).le.0.or.k(i,1).ge.6) goto 550
        nds=nds+1
        if(nds.gt.8) then
          nredc=nredc+1
          return
        endif
        kfm=2*iabs(k(i,2))
        if(k(i,2).lt.0) kfm=kfm-1
        do 530 ids=nds-1,1,-1
        iin=ids+1
        if(kfm.lt.kfdm(ids)) goto 540
  530   kfdm(ids+1)=kfdm(ids)
        iin=1
  540   kfdm(iin)=kfm
  550   continue

c...find whether old or new final state.
        do 570 idc=1,nkfdc
        if(nds.lt.kfdc(idc,0)) then
          ikfdc=idc
          goto 580
        elseif(nds.eq.kfdc(idc,0)) then
          do 560 i=1,nds
          if(kfdm(i).lt.kfdc(idc,i)) then
            ikfdc=idc
            goto 580
          elseif(kfdm(i).gt.kfdc(idc,i)) then
            goto 570
          endif
  560     continue
          ikfdc=-idc
          goto 580
        endif
  570   continue
        ikfdc=nkfdc+1
  580   if(ikfdc.lt.0) then
          ikfdc=-ikfdc
        elseif(nkfdc.ge.200) then
          nredc=nredc+1
          return
        else
          do 590 idc=nkfdc,ikfdc,-1
          npdc(idc+1)=npdc(idc)
          do 590 i=0,8
  590     kfdc(idc+1,i)=kfdc(idc,i)
          nkfdc=nkfdc+1
          kfdc(ikfdc,0)=nds
          do 600 i=1,nds
  600     kfdc(ikfdc,i)=kfdm(i)
          npdc(ikfdc)=0
        endif
        npdc(ikfdc)=npdc(ikfdc)+1

c...write statistics on decay channels.
      elseif(mtabu.eq.52) then
        fac=1./max(1,nevdc)
        write(mstu(11),5900) nevdc
        do 620 idc=1,nkfdc
        do 610 i=1,kfdc(idc,0)
        kfm=kfdc(idc,i)
        kf=(kfm+1)/2
        if(2*kf.ne.kfm) kf=-kf
        call luname(kf,chau)
        chdc(i)=chau(1:12)
  610   if(chau(13:13).ne.' ') chdc(i)(12:12)='?'
  620   write(mstu(11),6000) fac*npdc(idc),(chdc(i),i=1,kfdc(idc,0))
        if(nredc.ne.0) write(mstu(11),6100) fac*nredc

c...copy statistics on decay channels into /lujets/.
      elseif(mtabu.eq.53) then
        fac=1./max(1,nevdc)
        do 650 idc=1,nkfdc
        k(idc,1)=32
        k(idc,2)=99
        k(idc,3)=0
        k(idc,4)=0
        k(idc,5)=kfdc(idc,0)
        do 630 j=1,5
        p(idc,j)=0.
  630   v(idc,j)=0.
        do 640 i=1,kfdc(idc,0)
        kfm=kfdc(idc,i)
        kf=(kfm+1)/2
        if(2*kf.ne.kfm) kf=-kf
        if(i.le.5) p(idc,i)=kf
  640   if(i.ge.6) v(idc,i-5)=kf
  650   v(idc,5)=fac*npdc(idc)
        n=nkfdc
        do 660 j=1,5
        k(n+1,j)=0
        p(n+1,j)=0.
  660   v(n+1,j)=0.
        k(n+1,1)=32
        k(n+1,2)=99
        k(n+1,5)=nevdc
        v(n+1,5)=fac*nredc
        mstu(3)=1
      endif

c...format statements for output on unit mstu(11) (default 6).
 5000 format(///20x,'event statistics - initial state'/
     &20x,'based on an analysis of ',i6,' events'//
     &3x,'main flavours after',8x,'fraction',4x,'subfractions ',
     &'according to fragmenting system multiplicity'/
     &4x,'hard interaction',24x,'1',7x,'2',7x,'3',7x,'4',7x,'5',
     &6x,'6-7',5x,'8-10',3x,'11-15',3x,'16-25',4x,'>25'/)
 5100 format(3x,a12,1x,a12,f10.5,1x,10f8.4)
 5200 format(///20x,'event statistics - final state'/
     &20x,'based on an analysis of ',i6,' events'//
     &5x,'mean primary multiplicity =',f8.3/
     &5x,'mean final   multiplicity =',f8.3/
     &5x,'mean charged multiplicity =',f8.3//
     &5x,'number of particles produced per event (directly and via ',
     &'decays/branchings)'/
     &5x,'kf    particle/jet  mdcy',8x,'particles',9x,'antiparticles',
     &5x,'total'/34x,'prim      seco      prim      seco'/)
 5300 format(1x,i6,4x,a16,i2,5(1x,f9.4))
 5400 format(///20x,'factorial moments analysis of multiplicity'/
     &20x,'based on an analysis of ',i6,' events'//
     &3x,'delta-',a3,' delta-phi     <n>/bin',10x,'<f2>',18x,'<f3>',
     &18x,'<f4>',18x,'<f5>'/35x,4('     value     error  '))
 5500 format(10x)
 5600 format(2x,2f10.4,f12.4,4(f12.4,f10.4))
 5700 format(///20x,'energy-energy correlation and asymmetry'/
     &20x,'based on an analysis of ',i6,' events'//
     &2x,'theta range',8x,'eec(theta)',8x,'eec(180-theta)',7x,
     &'eeca(theta)'/2x,'in degrees ',3('      value    error')/)
 5800 format(2x,f4.1,' - ',f4.1,3(f11.4,f9.4))
 5900 format(///20x,'decay channel analysis - final state'/
     &20x,'based on an analysis of ',i6,' events'//
     &2x,'probability',10x,'complete final state'/)
 6000 format(2x,f9.5,5x,8(a12,1x))
 6100 format(2x,f9.5,5x,'into other channels (more than 8 particles ',
     &'or table overflow)')

      return
      end

c*********************************************************************

      subroutine lueevt(kfl,ecm)

c...purpose: to handle the generation of an e+e- annihilation jet event.
      implicit double precision(d)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...check input parameters.
      if(mstu(12).ge.1) call lulist(0)
      if(kfl.lt.0.or.kfl.gt.8) then
        call luerrm(16,'(lueevt:) called with unknown flavour code')
        if(mstu(21).ge.1) return
      endif
      if(kfl.le.5) ecmmin=parj(127)+2.02*parf(100+max(1,kfl))
      if(kfl.ge.6) ecmmin=parj(127)+2.02*pmas(kfl,1)
      if(ecm.lt.ecmmin) then
        call luerrm(16,'(lueevt:) called with too small cm energy')
        if(mstu(21).ge.1) return
      endif

c...check consistency of mstj options set.
      if(mstj(109).eq.2.and.mstj(110).ne.1) then
        call luerrm(6,
     &  '(lueevt:) mstj(109) value requires mstj(110) = 1')
        mstj(110)=1
      endif
      if(mstj(109).eq.2.and.mstj(111).ne.0) then
        call luerrm(6,
     &  '(lueevt:) mstj(109) value requires mstj(111) = 0')
        mstj(111)=0
      endif

c...initialize alpha_strong and total cross-section.
      mstu(111)=mstj(108)
      if(mstj(108).eq.2.and.(mstj(101).eq.0.or.mstj(101).eq.1))
     &mstu(111)=1
      paru(112)=parj(121)
      if(mstu(111).eq.2) paru(112)=parj(122)
      if(mstj(116).gt.0.and.(mstj(116).ge.2.or.abs(ecm-parj(151)).ge.
     &parj(139).or.10*mstj(102)+kfl.ne.mstj(119))) call luxtot(kfl,ecm,
     &xtot)
      if(mstj(116).ge.3) mstj(116)=1
      parj(171)=0.

c...add initial e+e- to event record (documentation only).
      ntry=0
  100 ntry=ntry+1
      if(ntry.gt.100) then
        call luerrm(14,'(lueevt:) caught in an infinite loop')
        return
      endif
      mstu(24)=0
      nc=0
      if(mstj(115).ge.2) then
        nc=nc+2
        call lu1ent(nc-1,11,0.5*ecm,0.,0.)
        k(nc-1,1)=21
        call lu1ent(nc,-11,0.5*ecm,paru(1),0.)
        k(nc,1)=21
      endif

c...radiative photon (in initial state).
      mk=0
      ecmc=ecm
      if(mstj(107).ge.1.and.mstj(116).ge.1) call luradk(ecm,mk,pak,
     &thek,phik,alpk)
      if(mk.eq.1) ecmc=sqrt(ecm*(ecm-2.*pak))
      if(mstj(115).ge.1.and.mk.eq.1) then
        nc=nc+1
        call lu1ent(nc,22,pak,thek,phik)
        k(nc,3)=min(mstj(115)/2,1)
      endif

c...virtual exchange boson (gamma or z0).
      if(mstj(115).ge.3) then
        nc=nc+1
        kf=22
        if(mstj(102).eq.2) kf=23
        mstu10=mstu(10)
        mstu(10)=1
        p(nc,5)=ecmc
        call lu1ent(nc,kf,ecmc,0.,0.)
        k(nc,1)=21
        k(nc,3)=1
        mstu(10)=mstu10
      endif

c...choice of flavour and jet configuration.
      call luxkfl(kfl,ecm,ecmc,kflc)
      if(kflc.eq.0) goto 100
      call luxjet(ecmc,njet,cut)
      kfln=21
      if(njet.eq.4) call lux4jt(njet,cut,kflc,ecmc,kfln,x1,x2,x4,
     &x12,x14)
      if(njet.eq.3) call lux3jt(njet,cut,kflc,ecmc,x1,x3)
      if(njet.eq.2) mstj(120)=1

c...fill jet configuration and origin.
      if(njet.eq.2.and.mstj(101).ne.5) call lu2ent(nc+1,kflc,-kflc,ecmc)
      if(njet.eq.2.and.mstj(101).eq.5) call lu2ent(-(nc+1),kflc,-kflc,
     &ecmc)
      if(njet.eq.3) call lu3ent(nc+1,kflc,21,-kflc,ecmc,x1,x3)
      if(njet.eq.4.and.kfln.eq.21) call lu4ent(nc+1,kflc,kfln,kfln,
     &-kflc,ecmc,x1,x2,x4,x12,x14)
      if(njet.eq.4.and.kfln.ne.21) call lu4ent(nc+1,kflc,-kfln,kfln,
     &-kflc,ecmc,x1,x2,x4,x12,x14)
      if(mstu(24).ne.0) goto 100
      do 110 ip=nc+1,n
  110 k(ip,3)=k(ip,3)+min(mstj(115)/2,1)+(mstj(115)/3)*(nc-1)

c...angular orientation according to matrix element.
      if(mstj(106).eq.1) then
        call luxdif(nc,njet,kflc,ecmc,chi,the,phi)
        call ludbrb(nc+1,n,0.,chi,0d0,0d0,0d0)
        call ludbrb(nc+1,n,the,phi,0d0,0d0,0d0)
      endif

c...rotation and boost from radiative photon.
      if(mk.eq.1) then
        dbek=-pak/(ecm-pak)
        nmin=nc+1-mstj(115)/3
        call ludbrb(nmin,n,0.,-phik,0d0,0d0,0d0)
        call ludbrb(nmin,n,alpk,0.,dbek*sin(thek),0d0,dbek*cos(thek))
        call ludbrb(nmin,n,0.,phik,0d0,0d0,0d0)
      endif

c...generate parton shower. rearrange along strings and check.
      if(mstj(101).eq.5) then
        call lushow(n-1,n,ecmc)
        mstj14=mstj(14)
        if(mstj(105).eq.-1) mstj(14)=-1
        if(mstj(105).ge.0) mstu(28)=0
        call luprep(0)
        mstj(14)=mstj14
        if(mstj(105).ge.0.and.mstu(28).ne.0) goto 100
      endif

c...fragmentation/decay generation. information for lutabu.
      if(mstj(105).eq.1) call luexec
      mstu(161)=kflc
      mstu(162)=-kflc

      return
      end

c*********************************************************************

      subroutine luxtot(kfl,ecm,xtot)

c...purpose: to calculate total cross-section, including initial
c...state radiation effects.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat1/,/ludat2/

c...status, (optimized) q^2 scale, alpha_strong.
      parj(151)=ecm
      mstj(119)=10*mstj(102)+kfl
      if(mstj(111).eq.0) then
        q2r=ecm**2
      elseif(mstu(111).eq.0) then
        parj(168)=min(1.,max(parj(128),exp(-12.*paru(1)/
     &  ((33.-2.*mstu(112))*paru(111)))))
        q2r=parj(168)*ecm**2
      else
        parj(168)=min(1.,max(parj(128),paru(112)/ecm,
     &  (2.*paru(112)/ecm)**2))
        q2r=parj(168)*ecm**2
      endif
      alspi=ulalps(q2r)/paru(1)

c...qcd corrections factor in r.
      if(mstj(101).eq.0.or.mstj(109).eq.1) then
        rqcd=1.
      elseif(iabs(mstj(101)).eq.1.and.mstj(109).eq.0) then
        rqcd=1.+alspi
      elseif(mstj(109).eq.0) then
        rqcd=1.+alspi+(1.986-0.115*mstu(118))*alspi**2
        if(mstj(111).eq.1) rqcd=max(1.,rqcd+(33.-2.*mstu(112))/12.*
     &  log(parj(168))*alspi**2)
      elseif(iabs(mstj(101)).eq.1) then
        rqcd=1.+(3./4.)*alspi
      else
        rqcd=1.+(3./4.)*alspi-(3./32.+0.519*mstu(118))*alspi**2
      endif

c...calculate z0 width if default value not acceptable.
      if(mstj(102).ge.3) then
        rva=3.*(3.+(4.*paru(102)-1.)**2)+6.*rqcd*(2.+(1.-8.*paru(102)/
     &  3.)**2+(4.*paru(102)/3.-1.)**2)
        do 100 kflc=5,6
        vq=1.
        if(mod(mstj(103),2).eq.1) vq=sqrt(max(0.,1.-(2.*ulmass(kflc)/
     &  ecm)**2))
        if(kflc.eq.5) vf=4.*paru(102)/3.-1.
        if(kflc.eq.6) vf=1.-8.*paru(102)/3.
  100   rva=rva+3.*rqcd*(0.5*vq*(3.-vq**2)*vf**2+vq**3)
        parj(124)=paru(101)*parj(123)*rva/(48.*paru(102)*(1.-paru(102)))
      endif

c...calculate propagator and related constants for qfd case.
      poll=1.-parj(131)*parj(132)
      if(mstj(102).ge.2) then
        sff=1./(16.*paru(102)*(1.-paru(102)))
        sfw=ecm**4/((ecm**2-parj(123)**2)**2+(parj(123)*parj(124))**2)
        sfi=sfw*(1.-(parj(123)/ecm)**2)
        ve=4.*paru(102)-1.
        sf1i=sff*(ve*poll+parj(132)-parj(131))
        sf1w=sff**2*((ve**2+1.)*poll+2.*ve*(parj(132)-parj(131)))
        hf1i=sfi*sf1i
        hf1w=sfw*sf1w
      endif

c...loop over different flavours: charge, velocity.
      rtot=0.
      rqq=0.
      rqv=0.
      rva=0.
      do 110 kflc=1,max(mstj(104),kfl)
      if(kfl.gt.0.and.kflc.ne.kfl) goto 110
      mstj(93)=1
      pmq=ulmass(kflc)
      if(ecm.lt.2.*pmq+parj(127)) goto 110
      qf=kchg(kflc,1)/3.
      vq=1.
      if(mod(mstj(103),2).eq.1) vq=sqrt(1.-(2.*pmq/ecm)**2)

c...calculate r and sum of charges for qed or qfd case.
      rqq=rqq+3.*qf**2*poll
      if(mstj(102).le.1) then
        rtot=rtot+3.*0.5*vq*(3.-vq**2)*qf**2*poll
      else
        vf=sign(1.,qf)-4.*qf*paru(102)
        rqv=rqv-6.*qf*vf*sf1i
        rva=rva+3.*(vf**2+1.)*sf1w
        rtot=rtot+3.*(0.5*vq*(3.-vq**2)*(qf**2*poll-2.*qf*vf*hf1i+
     &  vf**2*hf1w)+vq**3*hf1w)
      endif
  110 continue
      rsum=rqq
      if(mstj(102).ge.2) rsum=rqq+sfi*rqv+sfw*rva

c...calculate cross-section, including qcd corrections.
      parj(141)=rqq
      parj(142)=rtot
      parj(143)=rtot*rqcd
      parj(144)=parj(143)
      parj(145)=parj(141)*86.8/ecm**2
      parj(146)=parj(142)*86.8/ecm**2
      parj(147)=parj(143)*86.8/ecm**2
      parj(148)=parj(147)
      parj(157)=rsum*rqcd
      parj(158)=0.
      parj(159)=0.
      xtot=parj(147)
      if(mstj(107).le.0) return

c...virtual cross-section.
      xkl=parj(135)
      xku=min(parj(136),1.-(2.*parj(127)/ecm)**2)
      ale=2.*log(ecm/ulmass(11))-1.
      sigv=ale/3.+2.*log(ecm**2/(ulmass(13)*ulmass(15)))/3.-4./3.+
     &1.526*log(ecm**2/0.932)

c...soft and hard radiative cross-section in qed case.
      if(mstj(102).le.1) then
        sigv=1.5*ale-0.5+paru(1)**2/3.+2.*sigv
        sigs=ale*(2.*log(xkl)-log(1.-xkl)-xkl)
        sigh=ale*(2.*log(xku/xkl)-log((1.-xku)/(1.-xkl))-(xku-xkl))

c...soft and hard radiative cross-section in qfd case.
      else
        szm=1.-(parj(123)/ecm)**2
        szw=parj(123)*parj(124)/ecm**2
        parj(161)=-rqq/rsum
        parj(162)=-(rqq+rqv+rva)/rsum
        parj(163)=(rqv*(1.-0.5*szm-sfi)+rva*(1.5-szm-sfw))/rsum
        parj(164)=(rqv*szw**2*(1.-2.*sfw)+rva*(2.*sfi+szw**2-4.+3.*szm-
     &  szm**2))/(szw*rsum)
        sigv=1.5*ale-0.5+paru(1)**2/3.+((2.*rqq+sfi*rqv)/rsum)*sigv+
     &  (szw*sfw*rqv/rsum)*paru(1)*20./9.
        sigs=ale*(2.*log(xkl)+parj(161)*log(1.-xkl)+parj(162)*xkl+
     &  parj(163)*log(((xkl-szm)**2+szw**2)/(szm**2+szw**2))+
     &  parj(164)*(atan((xkl-szm)/szw)-atan(-szm/szw)))
        sigh=ale*(2.*log(xku/xkl)+parj(161)*log((1.-xku)/(1.-xkl))+
     &  parj(162)*(xku-xkl)+parj(163)*log(((xku-szm)**2+szw**2)/
     &  ((xkl-szm)**2+szw**2))+parj(164)*(atan((xku-szm)/szw)-
     &  atan((xkl-szm)/szw)))
      endif

c...total cross-section and fraction of hard photon events.
      parj(160)=sigh/(paru(1)/paru(101)+sigv+sigs+sigh)
      parj(157)=rsum*(1.+(paru(101)/paru(1))*(sigv+sigs+sigh))*rqcd
      parj(144)=parj(157)
      parj(148)=parj(144)*86.8/ecm**2
      xtot=parj(148)

      return
      end

c*********************************************************************

      subroutine luradk(ecm,mk,pak,thek,phik,alpk)

c...purpose: to generate initial state photon radiation.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/

c...function: cumulative hard photon spectrum in qfd case.
      fxk(xx)=2.*log(xx)+parj(161)*log(1.-xx)+parj(162)*xx+
     &parj(163)*log((xx-szm)**2+szw**2)+parj(164)*atan((xx-szm)/szw)

c...determine whether radiative photon or not.
      mk=0
      pak=0.
      if(parj(160).lt.rlu(0)) return
      mk=1

c...photon energy range. find photon momentum in qed case.
      xkl=parj(135)
      xku=min(parj(136),1.-(2.*parj(127)/ecm)**2)
      if(mstj(102).le.1) then
  100   xk=1./(1.+(1./xkl-1.)*((1./xku-1.)/(1./xkl-1.))**rlu(0))
        if(1.+(1.-xk)**2.lt.2.*rlu(0)) goto 100

c...ditto in qfd case, by numerical inversion of integrated spectrum.
      else
        szm=1.-(parj(123)/ecm)**2
        szw=parj(123)*parj(124)/ecm**2
        fxkl=fxk(xkl)
        fxku=fxk(xku)
        fxkd=1e-4*(fxku-fxkl)
        fxkr=fxkl+rlu(0)*(fxku-fxkl)
        nxk=0
  110   nxk=nxk+1
        xk=0.5*(xkl+xku)
        fxkv=fxk(xk)
        if(fxkv.gt.fxkr) then
          xku=xk
          fxku=fxkv
        else
          xkl=xk
          fxkl=fxkv
        endif
        if(nxk.lt.15.and.fxku-fxkl.gt.fxkd) goto 110
        xk=xkl+(xku-xkl)*(fxkr-fxkl)/(fxku-fxkl)
      endif
      pak=0.5*ecm*xk

c...photon polar and azimuthal angle.
      pme=2.*(ulmass(11)/ecm)**2
  120 cthm=pme*(2./pme)**rlu(0)
      if(1.-(xk**2*cthm*(1.-0.5*cthm)+2.*(1.-xk)*pme/max(pme,
     &cthm*(1.-0.5*cthm)))/(1.+(1.-xk)**2).lt.rlu(0)) goto 120
      cthe=1.-cthm
      if(rlu(0).gt.0.5) cthe=-cthe
      sthe=sqrt(max(0.,(cthm-pme)*(2.-cthm)))
      thek=ulangl(cthe,sthe)
      phik=paru(2)*rlu(0)

c...rotation angle for hadronic system.
      sgn=1.
      if(0.5*(2.-xk*(1.-cthe))**2/((2.-xk)**2+(xk*cthe)**2).gt.
     &rlu(0)) sgn=-1.
      alpk=asin(sgn*sthe*(xk-sgn*(2.*sqrt(1.-xk)-2.+xk)*cthe)/
     &(2.-xk*(1.-sgn*cthe)))

      return
      end

c*********************************************************************

      subroutine luxkfl(kfl,ecm,ecmc,kflc)

c...purpose: to select flavour for produced qqbar pair.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /ludat1/,/ludat2/

c...calculate maximum weight in qed or qfd case.
      if(mstj(102).le.1) then
        rfmax=4./9.
      else
        poll=1.-parj(131)*parj(132)
        sff=1./(16.*paru(102)*(1.-paru(102)))
        sfw=ecmc**4/((ecmc**2-parj(123)**2)**2+(parj(123)*parj(124))**2)
        sfi=sfw*(1.-(parj(123)/ecmc)**2)
        ve=4.*paru(102)-1.
        hf1i=sfi*sff*(ve*poll+parj(132)-parj(131))
        hf1w=sfw*sff**2*((ve**2+1.)*poll+2.*ve*(parj(132)-parj(131)))
        rfmax=max(4./9.*poll-4./3.*(1.-8.*paru(102)/3.)*hf1i+
     &  ((1.-8.*paru(102)/3.)**2+1.)*hf1w,1./9.*poll+2./3.*
     &  (-1.+4.*paru(102)/3.)*hf1i+((-1.+4.*paru(102)/3.)**2+1.)*hf1w)
      endif

c...choose flavour. gives charge and velocity.
      ntry=0
  100 ntry=ntry+1
      if(ntry.gt.100) then
        call luerrm(14,'(luxkfl:) caught in an infinite loop')
        kflc=0
        return
      endif
      kflc=kfl
      if(kfl.le.0) kflc=1+int(mstj(104)*rlu(0))
      mstj(93)=1
      pmq=ulmass(kflc)
      if(ecm.lt.2.*pmq+parj(127)) goto 100
      qf=kchg(kflc,1)/3.
      vq=1.
      if(mod(mstj(103),2).eq.1) vq=sqrt(max(0.,1.-(2.*pmq/ecmc)**2))

c...calculate weight in qed or qfd case.
      if(mstj(102).le.1) then
        rf=qf**2
        rfv=0.5*vq*(3.-vq**2)*qf**2
      else
        vf=sign(1.,qf)-4.*qf*paru(102)
        rf=qf**2*poll-2.*qf*vf*hf1i+(vf**2+1.)*hf1w
        rfv=0.5*vq*(3.-vq**2)*(qf**2*poll-2.*qf*vf*hf1i+vf**2*hf1w)+
     &  vq**3*hf1w
        if(rfv.gt.0.) parj(171)=min(1.,vq**3*hf1w/rfv)
      endif

c...weighting or new event (radiative photon). cross-section update.
      if(kfl.le.0.and.rf.lt.rlu(0)*rfmax) goto 100
      parj(158)=parj(158)+1.
      if(ecmc.lt.2.*pmq+parj(127).or.rfv.lt.rlu(0)*rf) kflc=0
      if(mstj(107).le.0.and.kflc.eq.0) goto 100
      if(kflc.ne.0) parj(159)=parj(159)+1.
      parj(144)=parj(157)*parj(159)/parj(158)
      parj(148)=parj(144)*86.8/ecm**2

      return
      end

c*********************************************************************

      subroutine luxjet(ecm,njet,cut)

c...purpose: to select number of jets in matrix element approach.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/
      dimension zhut(5)

c...relative three-jet rate in zhu second order parametrization.
      data zhut/3.0922, 6.2291, 7.4782, 7.8440, 8.2560/

c...trivial result for two-jets only, including parton shower.
      if(mstj(101).eq.0.or.mstj(101).eq.5) then
        cut=0.

c...qcd and abelian vector gluon theory: q^2 for jet rate and r.
      elseif(mstj(109).eq.0.or.mstj(109).eq.2) then
        cf=4./3.
        if(mstj(109).eq.2) cf=1.
        if(mstj(111).eq.0) then
          q2=ecm**2
          q2r=ecm**2
        elseif(mstu(111).eq.0) then
          parj(169)=min(1.,parj(129))
          q2=parj(169)*ecm**2
          parj(168)=min(1.,max(parj(128),exp(-12.*paru(1)/
     &    ((33.-2.*mstu(112))*paru(111)))))
          q2r=parj(168)*ecm**2
        else
          parj(169)=min(1.,max(parj(129),(2.*paru(112)/ecm)**2))
          q2=parj(169)*ecm**2
          parj(168)=min(1.,max(parj(128),paru(112)/ecm,
     &    (2.*paru(112)/ecm)**2))
          q2r=parj(168)*ecm**2
        endif

c...alpha_strong for r and r itself.
        alspi=(3./4.)*cf*ulalps(q2r)/paru(1)
        if(iabs(mstj(101)).eq.1) then
          rqcd=1.+alspi
        elseif(mstj(109).eq.0) then
          rqcd=1.+alspi+(1.986-0.115*mstu(118))*alspi**2
          if(mstj(111).eq.1) rqcd=max(1.,rqcd+(33.-2.*mstu(112))/12.*
     &    log(parj(168))*alspi**2)
        else
          rqcd=1.+alspi-(3./32.+0.519*mstu(118))*(4.*alspi/3.)**2
        endif

c...alpha_strong for jet rate. initial value for y cut.
        alspi=(3./4.)*cf*ulalps(q2)/paru(1)
        cut=max(0.001,parj(125),(parj(126)/ecm)**2)
        if(iabs(mstj(101)).le.1.or.(mstj(109).eq.0.and.mstj(111).eq.0))
     &  cut=max(cut,exp(-sqrt(0.75/alspi))/2.)
        if(mstj(110).eq.2) cut=max(0.01,min(0.05,cut))

c...parametrization of first order three-jet cross-section.
  100   if(mstj(101).eq.0.or.cut.ge.0.25) then
          parj(152)=0.
        else
          parj(152)=(2.*alspi/3.)*((3.-6.*cut+2.*log(cut))*
     &    log(cut/(1.-2.*cut))+(2.5+1.5*cut-6.571)*(1.-3.*cut)+
     &    5.833*(1.-3.*cut)**2-3.894*(1.-3.*cut)**3+
     &    1.342*(1.-3.*cut)**4)/rqcd
          if(mstj(109).eq.2.and.(mstj(101).eq.2.or.mstj(101).le.-2))
     &    parj(152)=0.
        endif

c...parametrization of second order three-jet cross-section.
        if(iabs(mstj(101)).le.1.or.mstj(101).eq.3.or.mstj(109).eq.2.or.
     &  cut.ge.0.25) then
          parj(153)=0.
        elseif(mstj(110).le.1) then
          ct=log(1./cut-2.)
          parj(153)=alspi**2*ct**2*(2.419+0.5989*ct+0.6782*ct**2-
     &    0.2661*ct**3+0.01159*ct**4)/rqcd

c...interpolation in second/first order ratio for zhu parametrization.
        elseif(mstj(110).eq.2) then
          iza=0
          do 110 iy=1,5
  110     if(abs(cut-0.01*iy).lt.0.0001) iza=iy
          if(iza.ne.0) then
            zhurat=zhut(iza)
          else
            iz=100.*cut
            zhurat=zhut(iz)+(100.*cut-iz)*(zhut(iz+1)-zhut(iz))
          endif
          parj(153)=alspi*parj(152)*zhurat
        endif

c...shift in second order three-jet cross-section with optimized q^2.
        if(mstj(111).eq.1.and.iabs(mstj(101)).ge.2.and.mstj(101).ne.3.
     &  and.cut.lt.0.25) parj(153)=parj(153)+(33.-2.*mstu(112))/12.*
     &  log(parj(169))*alspi*parj(152)

c...parametrization of second order four-jet cross-section.
        if(iabs(mstj(101)).le.1.or.cut.ge.0.125) then
          parj(154)=0.
        else
          ct=log(1./cut-5.)
          if(cut.le.0.018) then
            xqqgg=6.349-4.330*ct+0.8304*ct**2
            if(mstj(109).eq.2) xqqgg=(4./3.)**2*(3.035-2.091*ct+
     &      0.4059*ct**2)
            xqqqq=1.25*(-0.1080+0.01486*ct+0.009364*ct**2)
            if(mstj(109).eq.2) xqqqq=8.*xqqqq
          else
            xqqgg=-0.09773+0.2959*ct-0.2764*ct**2+0.08832*ct**3
            if(mstj(109).eq.2) xqqgg=(4./3.)**2*(-0.04079+0.1340*ct-
     &      0.1326*ct**2+0.04365*ct**3)
            xqqqq=1.25*(0.003661-0.004888*ct-0.001081*ct**2+0.002093*
     &      ct**3)
            if(mstj(109).eq.2) xqqqq=8.*xqqqq
          endif
          parj(154)=alspi**2*ct**2*(xqqgg+xqqqq)/rqcd
          parj(155)=xqqqq/(xqqgg+xqqqq)
        endif

c...if negative three-jet rate, change y' optimization parameter.
        if(mstj(111).eq.1.and.parj(152)+parj(153).lt.0..and.
     &  parj(169).lt.0.99) then
          parj(169)=min(1.,1.2*parj(169))
          q2=parj(169)*ecm**2
          alspi=(3./4.)*cf*ulalps(q2)/paru(1)
          goto 100
        endif

c...if too high cross-section, use harder cuts, or fail.
        if(parj(152)+parj(153)+parj(154).ge.1) then
          if(mstj(110).eq.2.and.cut.gt.0.0499.and.mstj(111).eq.1.and.
     &    parj(169).lt.0.99) then
            parj(169)=min(1.,1.2*parj(169))
            q2=parj(169)*ecm**2
            alspi=(3./4.)*cf*ulalps(q2)/paru(1)
            goto 100
          elseif(mstj(110).eq.2.and.cut.gt.0.0499) then
            call luerrm(26,
     &      '(luxjet:) no allowed y cut value for zhu parametrization')
          endif
          cut=0.26*(4.*cut)**(parj(152)+parj(153)+parj(154))**(-1./3.)
          if(mstj(110).eq.2) cut=max(0.01,min(0.05,cut))
          goto 100
        endif

c...scalar gluon (first order only).
      else
        alspi=ulalps(ecm**2)/paru(1)
        cut=max(0.001,parj(125),(parj(126)/ecm)**2,exp(-3./alspi))
        parj(152)=0.
        if(cut.lt.0.25) parj(152)=(alspi/3.)*((1.-2.*cut)*
     &  log((1.-2.*cut)/cut)+0.5*(9.*cut**2-1.))
        parj(153)=0.
        parj(154)=0.
      endif

c...select number of jets.
      parj(150)=cut
      if(mstj(101).eq.0.or.mstj(101).eq.5) then
        njet=2
      elseif(mstj(101).le.0) then
        njet=min(4,2-mstj(101))
      else
        rnj=rlu(0)
        njet=2
        if(parj(152)+parj(153)+parj(154).gt.rnj) njet=3
        if(parj(154).gt.rnj) njet=4
      endif

      return
      end

c*********************************************************************

      subroutine lux3jt(njet,cut,kfl,ecm,x1,x2)

c...purpose: to select the kinematical variables of three-jet events.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/
      dimension zhup(5,12)

c...coefficients of zhu second order parametrization.
      data ((zhup(ic1,ic2),ic2=1,12),ic1=1,5)/
     &    18.29,    89.56,    4.541,   -52.09,   -109.8,    24.90,
     &    11.63,    3.683,    17.50, 0.002440,   -1.362,  -0.3537,
     &    11.42,    6.299,   -22.55,   -8.915,    59.25,   -5.855,
     &   -32.85,   -1.054,   -16.90, 0.006489,  -0.8156,  0.01095,
     &    7.847,   -3.964,   -35.83,    1.178,    29.39,   0.2806,
     &    47.82,   -12.36,   -56.72,  0.04054,  -0.4365,   0.6062,
     &    5.441,   -56.89,   -50.27,    15.13,    114.3,   -18.19,
     &    97.05,   -1.890,   -139.9,  0.08153,  -0.4984,   0.9439,
     &   -17.65,    51.44,   -58.32,    70.95,   -255.7,   -78.99,
     &    476.9,    29.65,   -239.3,   0.4745,   -1.174,    6.081/

c...dilogarithm of x for x<0.5 (x>0.5 obtained by analytic trick).
      dilog(x)=x+x**2/4.+x**3/9.+x**4/16.+x**5/25.+x**6/36.+x**7/49.

c...event type. mass effect factors and other common constants.
      mstj(120)=2
      mstj(121)=0
      pmq=ulmass(kfl)
      qme=(2.*pmq/ecm)**2
      if(mstj(109).ne.1) then
        cutl=log(cut)
        cutd=log(1./cut-2.)
        if(mstj(109).eq.0) then
          cf=4./3.
          cn=3.
          tr=2.
          wtmx=min(20.,37.-6.*cutd)
          if(mstj(110).eq.2) wtmx=2.*(7.5+80.*cut)
        else
          cf=1.
          cn=0.
          tr=12.
          wtmx=0.
        endif

c...alpha_strong and effects of optimized q^2 scale. maximum weight.
        als2pi=paru(118)/paru(2)
        wtopt=0.
        if(mstj(111).eq.1) wtopt=(33.-2.*mstu(112))/6.*log(parj(169))*
     &  als2pi
        wtmax=max(0.,1.+wtopt+als2pi*wtmx)

c...choose three-jet events in allowed region.
  100   njet=3
  110   y13l=cutl+cutd*rlu(0)
        y23l=cutl+cutd*rlu(0)
        y13=exp(y13l)
        y23=exp(y23l)
        y12=1.-y13-y23
        if(y12.le.cut) goto 110
        if(y13**2+y23**2+2.*y12.le.2.*rlu(0)) goto 110

c...second order corrections.
        if(mstj(101).eq.2.and.mstj(110).le.1) then
          y12l=log(y12)
          y13m=log(1.-y13)
          y23m=log(1.-y23)
          y12m=log(1.-y12)
          if(y13.le.0.5) y13i=dilog(y13)
          if(y13.ge.0.5) y13i=1.644934-y13l*y13m-dilog(1.-y13)
          if(y23.le.0.5) y23i=dilog(y23)
          if(y23.ge.0.5) y23i=1.644934-y23l*y23m-dilog(1.-y23)
          if(y12.le.0.5) y12i=dilog(y12)
          if(y12.ge.0.5) y12i=1.644934-y12l*y12m-dilog(1.-y12)
          wt1=(y13**2+y23**2+2.*y12)/(y13*y23)
          wt2=cf*(-2.*(cutl-y12l)**2-3.*cutl-1.+3.289868+
     &    2.*(2.*cutl-y12l)*cut/y12)+
     &    cn*((cutl-y12l)**2-(cutl-y13l)**2-(cutl-y23l)**2-11.*cutl/6.+
     &    67./18.+1.644934-(2.*cutl-y12l)*cut/y12+(2.*cutl-y13l)*
     &    cut/y13+(2.*cutl-y23l)*cut/y23)+
     &    tr*(2.*cutl/3.-10./9.)+
     &    cf*(y12/(y12+y13)+y12/(y12+y23)+(y12+y23)/y13+(y12+y13)/y23+
     &    y13l*(4.*y12**2+2.*y12*y13+4.*y12*y23+y13*y23)/(y12+y23)**2+
     &    y23l*(4.*y12**2+2.*y12*y23+4.*y12*y13+y13*y23)/(y12+y13)**2)/
     &    wt1+
     &    cn*(y13l*y13/(y12+y23)+y23l*y23/(y12+y13))/wt1+
     &    (cn-2.*cf)*((y12**2+(y12+y13)**2)*(y12l*y23l-y12l*y12m-y23l*
     &    y23m+1.644934-y12i-y23i)/(y13*y23)+(y12**2+(y12+y23)**2)*
     &    (y12l*y13l-y12l*y12m-y13l*y13m+1.644934-y12i-y13i)/
     &    (y13*y23)+(y13**2+y23**2)/(y13*y23*(y13+y23))-
     &    2.*y12l*y12**2/(y13+y23)**2-4.*y12l*y12/(y13+y23))/wt1-
     &    cn*(y13l*y23l-y13l*y13m-y23l*y23m+1.644934-y13i-y23i)
          if(1.+wtopt+als2pi*wt2.le.0.) mstj(121)=1
          if(1.+wtopt+als2pi*wt2.le.wtmax*rlu(0)) goto 110
          parj(156)=(wtopt+als2pi*wt2)/(1.+wtopt+als2pi*wt2)

        elseif(mstj(101).eq.2.and.mstj(110).eq.2) then
c...second order corrections; zhu parametrization of ert.
          zx=(y23-y13)**2
          zy=1.-y12
          iza=0
          do 120 iy=1,5
  120     if(abs(cut-0.01*iy).lt.0.0001) iza=iy
          if(iza.ne.0) then
            iz=iza
            wt2=zhup(iz,1)+zhup(iz,2)*zx+zhup(iz,3)*zx**2+(zhup(iz,4)+
     &      zhup(iz,5)*zx)*zy+(zhup(iz,6)+zhup(iz,7)*zx)*zy**2+
     &      (zhup(iz,8)+zhup(iz,9)*zx)*zy**3+zhup(iz,10)/(zx-zy**2)+
     &      zhup(iz,11)/(1.-zy)+zhup(iz,12)/zy
          else
            iz=100.*cut
            wtl=zhup(iz,1)+zhup(iz,2)*zx+zhup(iz,3)*zx**2+(zhup(iz,4)+
     &      zhup(iz,5)*zx)*zy+(zhup(iz,6)+zhup(iz,7)*zx)*zy**2+
     &      (zhup(iz,8)+zhup(iz,9)*zx)*zy**3+zhup(iz,10)/(zx-zy**2)+
     &      zhup(iz,11)/(1.-zy)+zhup(iz,12)/zy
            iz=iz+1
            wtu=zhup(iz,1)+zhup(iz,2)*zx+zhup(iz,3)*zx**2+(zhup(iz,4)+
     &      zhup(iz,5)*zx)*zy+(zhup(iz,6)+zhup(iz,7)*zx)*zy**2+
     &      (zhup(iz,8)+zhup(iz,9)*zx)*zy**3+zhup(iz,10)/(zx-zy**2)+
     &      zhup(iz,11)/(1.-zy)+zhup(iz,12)/zy
            wt2=wtl+(wtu-wtl)*(100.*cut+1.-iz)
          endif
          if(1.+wtopt+2.*als2pi*wt2.le.0.) mstj(121)=1
          if(1.+wtopt+2.*als2pi*wt2.le.wtmax*rlu(0)) goto 110
          parj(156)=(wtopt+2.*als2pi*wt2)/(1.+wtopt+2.*als2pi*wt2)
        endif

c...impose mass cuts (gives two jets). for fixed jet number new try.
        x1=1.-y23
        x2=1.-y13
        x3=1.-y12
        if(4.*y23*y13*y12/x3**2.le.qme) njet=2
        if(mod(mstj(103),4).ge.2.and.iabs(mstj(101)).le.1.and.qme*x3+
     &  0.5*qme**2+(0.5*qme+0.25*qme**2)*((1.-x2)/(1.-x1)+
     &  (1.-x1)/(1.-x2)).gt.(x1**2+x2**2)*rlu(0)) njet=2
        if(mstj(101).eq.-1.and.njet.eq.2) goto 100

c...scalar gluon model (first order only, no mass effects).
      else
  130   njet=3
  140   x3=sqrt(4.*cut**2+rlu(0)*((1.-cut)**2-4.*cut**2))
        if(log((x3-cut)/cut).le.rlu(0)*log((1.-2.*cut)/cut)) goto 140
        yd=sign(2.*cut*((x3-cut)/cut)**rlu(0)-x3,rlu(0)-0.5)
        x1=1.-0.5*(x3+yd)
        x2=1.-0.5*(x3-yd)
        if(4.*(1.-x1)*(1.-x2)*(1.-x3)/x3**2.le.qme) njet=2
        if(mstj(102).ge.2) then
          if(x3**2-2.*(1.+x3)*(1.-x1)*(1.-x2)*parj(171).lt.
     &    x3**2*rlu(0)) njet=2
        endif
        if(mstj(101).eq.-1.and.njet.eq.2) goto 130
      endif

      return
      end

c*********************************************************************

      subroutine lux4jt(njet,cut,kfl,ecm,kfln,x1,x2,x4,x12,x14)

c...purpose: to select the kinematical variables of four-jet events.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /ludat1/
      dimension wta(4),wtb(4),wtc(4),wtd(4),wte(4)

c...common constants. colour factors for qcd and abelian gluon theory.
      pmq=ulmass(kfl)
      qme=(2.*pmq/ecm)**2
      ct=log(1./cut-5.)
      if(mstj(109).eq.0) then
        cf=4./3.
        cn=3.
        tr=2.5
      else
        cf=1.
        cn=0.
        tr=15.
      endif

c...choice of process (qqbargg or qqbarqqbar).
  100 njet=4
      it=1
      if(parj(155).gt.rlu(0)) it=2
      if(mstj(101).le.-3) it=-mstj(101)-2
      if(it.eq.1) wtmx=0.7/cut**2
      if(it.eq.1.and.mstj(109).eq.2) wtmx=0.6/cut**2
      if(it.eq.2) wtmx=0.1125*cf*tr/cut**2
      id=1

c...sample the five kinematical variables (for qqgg preweighted in y34).
  110 y134=3.*cut+(1.-6.*cut)*rlu(0)
      y234=3.*cut+(1.-6.*cut)*rlu(0)
      if(it.eq.1) y34=(1.-5.*cut)*exp(-ct*rlu(0))
      if(it.eq.2) y34=cut+(1.-6.*cut)*rlu(0)
      if(y34.le.y134+y234-1..or.y34.ge.y134*y234) goto 110
      vt=rlu(0)
      cp=cos(paru(1)*rlu(0))
      y14=(y134-y34)*vt
      y13=y134-y14-y34
      vb=y34*(1.-y134-y234+y34)/((y134-y34)*(y234-y34))
      y24=0.5*(y234-y34)*(1.-4.*sqrt(max(0.,vt*(1.-vt)*vb*(1.-vb)))*
     &cp-(1.-2.*vt)*(1.-2.*vb))
      y23=y234-y34-y24
      y12=1.-y134-y23-y24
      if(min(y12,y13,y14,y23,y24).le.cut) goto 110
      y123=y12+y13+y23
      y124=y12+y14+y24

c...calculate matrix elements for qqgg or qqqq process.
      ic=0
      wttot=0.
  120 ic=ic+1
      if(it.eq.1) then
        wta(ic)=(y12*y34**2-y13*y24*y34+y14*y23*y34+3.*y12*y23*y34+
     &  3.*y12*y14*y34+4.*y12**2*y34-y13*y23*y24+2.*y12*y23*y24-
     &  y13*y14*y24-2.*y12*y13*y24+2.*y12**2*y24+y14*y23**2+2.*y12*
     &  y23**2+y14**2*y23+4.*y12*y14*y23+4.*y12**2*y23+2.*y12*y14**2+
     &  2.*y12*y13*y14+4.*y12**2*y14+2.*y12**2*y13+2.*y12**3)/(2.*y13*
     &  y134*y234*y24)+(y24*y34+y12*y34+y13*y24-y14*y23+y12*y13)/(y13*
     &  y134**2)+2.*y23*(1.-y13)/(y13*y134*y24)+y34/(2.*y13*y24)
        wtb(ic)=(y12*y24*y34+y12*y14*y34-y13*y24**2+y13*y14*y24+2.*y12*
     &  y14*y24)/(y13*y134*y23*y14)+y12*(1.+y34)*y124/(y134*y234*y14*
     &  y24)-(2.*y13*y24+y14**2+y13*y23+2.*y12*y13)/(y13*y134*y14)+
     &  y12*y123*y124/(2.*y13*y14*y23*y24)
        wtc(ic)=-(5.*y12*y34**2+2.*y12*y24*y34+2.*y12*y23*y34+2.*y12*
     &  y14*y34+2.*y12*y13*y34+4.*y12**2*y34-y13*y24**2+y14*y23*y24+
     &  y13*y23*y24+y13*y14*y24-y12*y14*y24-y13**2*y24-3.*y12*y13*y24-
     &  y14*y23**2-y14**2*y23+y13*y14*y23-3.*y12*y14*y23-y12*y13*y23)/
     &  (4.*y134*y234*y34**2)+(3.*y12*y34**2-3.*y13*y24*y34+3.*y12*y24*
     &  y34+3.*y14*y23*y34-y13*y24**2-y12*y23*y34+6.*y12*y14*y34+2.*y12*
     &  y13*y34-2.*y12**2*y34+y14*y23*y24-3.*y13*y23*y24-2.*y13*y14*
     &  y24+4.*y12*y14*y24+2.*y12*y13*y24+3.*y14*y23**2+2.*y14**2*y23+
     &  2.*y14**2*y12+2.*y12**2*y14+6.*y12*y14*y23-2.*y12*y13**2-
     &  2.*y12**2*y13)/(4.*y13*y134*y234*y34)
        wtc(ic)=wtc(ic)+(2.*y12*y34**2-2.*y13*y24*y34+y12*y24*y34+
     &  4.*y13*y23*y34+4.*y12*y14*y34+2.*y12*y13*y34+2.*y12**2*y34-
     &  y13*y24**2+3.*y14*y23*y24+4.*y13*y23*y24-2.*y13*y14*y24+
     &  4.*y12*y14*y24+2.*y12*y13*y24+2.*y14*y23**2+4.*y13*y23**2+
     &  2.*y13*y14*y23+2.*y12*y14*y23+4.*y12*y13*y23+2.*y12*y14**2+4.*
     &  y12**2*y13+4.*y12*y13*y14+2.*y12**2*y14)/(4.*y13*y134*y24*y34)-
     &  (y12*y34**2-2.*y14*y24*y34-2.*y13*y24*y34-y14*y23*y34+y13*y23*
     &  y34+y12*y14*y34+2.*y12*y13*y34-2.*y14**2*y24-4.*y13*y14*y24-
     &  4.*y13**2*y24-y14**2*y23-y13**2*y23+y12*y13*y14-y12*y13**2)/
     &  (2.*y13*y34*y134**2)+(y12*y34**2-4.*y14*y24*y34-2.*y13*y24*y34-
     &  2.*y14*y23*y34-4.*y13*y23*y34-4.*y12*y14*y34-4.*y12*y13*y34-
     &  2.*y13*y14*y24+2.*y13**2*y24+2.*y14**2*y23-2.*y13*y14*y23-
     &  y12*y14**2-6.*y12*y13*y14-y12*y13**2)/(4.*y34**2*y134**2)
        wttot=wttot+y34*cf*(cf*wta(ic)+(cf-0.5*cn)*wtb(ic)+cn*wtc(ic))/
     &  8.
      else
        wtd(ic)=(y13*y23*y34+y12*y23*y34-y12**2*y34+y13*y23*y24+2.*y12*
     &  y23*y24-y14*y23**2+y12*y13*y24+y12*y14*y23+y12*y13*y14)/(y13**2*
     &  y123**2)-(y12*y34**2-y13*y24*y34+y12*y24*y34-y14*y23*y34-y12*
     &  y23*y34-y13*y24**2+y14*y23*y24-y13*y23*y24-y13**2*y24+y14*
     &  y23**2)/(y13**2*y123*y134)+(y13*y14*y12+y34*y14*y12-y34**2*y12+
     &  y13*y14*y24+2.*y34*y14*y24-y23*y14**2+y34*y13*y24+y34*y23*y14+
     &  y34*y13*y23)/(y13**2*y134**2)-(y34*y12**2-y13*y24*y12+y34*y24*
     &  y12-y23*y14*y12-y34*y14*y12-y13*y24**2+y23*y14*y24-y13*y14*y24-
     &  y13**2*y24+y23*y14**2)/(y13**2*y134*y123)
        wte(ic)=(y12*y34*(y23-y24+y14+y13)+y13*y24**2-y14*y23*y24+y13*
     &  y23*y24+y13*y14*y24+y13**2*y24-y14*y23*(y14+y23+y13))/(y13*y23*
     &  y123*y134)-y12*(y12*y34-y23*y24-y13*y24-y14*y23-y14*y13)/(y13*
     &  y23*y123**2)-(y14+y13)*(y24+y23)*y34/(y13*y23*y134*y234)+
     &  (y12*y34*(y14-y24+y23+y13)+y13*y24**2-y23*y14*y24+y13*y14*y24+
     &  y13*y23*y24+y13**2*y24-y23*y14*(y14+y23+y13))/(y13*y14*y134*
     &  y123)-y34*(y34*y12-y14*y24-y13*y24-y23*y14-y23*y13)/(y13*y14*
     &  y134**2)-(y23+y13)*(y24+y14)*y12/(y13*y14*y123*y124)
        wttot=wttot+cf*(tr*wtd(ic)+(cf-0.5*cn)*wte(ic))/16.
      endif

c...permutations of momenta in matrix element. weighting.
  130 if(ic.eq.1.or.ic.eq.3.or.id.eq.2.or.id.eq.3) then
        ysav=y13
        y13=y14
        y14=ysav
        ysav=y23
        y23=y24
        y24=ysav
        ysav=y123
        y123=y124
        y124=ysav
      endif
      if(ic.eq.2.or.ic.eq.4.or.id.eq.3.or.id.eq.4) then
        ysav=y13
        y13=y23
        y23=ysav
        ysav=y14
        y14=y24
        y24=ysav
        ysav=y134
        y134=y234
        y234=ysav
      endif
      if(ic.le.3) goto 120
      if(id.eq.1.and.wttot.lt.rlu(0)*wtmx) goto 110
      ic=5

c...qqgg events: string configuration and event type.
      if(it.eq.1) then
        if(mstj(109).eq.0.and.id.eq.1) then
          parj(156)=y34*(2.*(wta(1)+wta(2)+wta(3)+wta(4))+4.*(wtc(1)+
     &    wtc(2)+wtc(3)+wtc(4)))/(9.*wttot)
          if(wta(2)+wta(4)+2.*(wtc(2)+wtc(4)).gt.rlu(0)*(wta(1)+wta(2)+
     &    wta(3)+wta(4)+2.*(wtc(1)+wtc(2)+wtc(3)+wtc(4)))) id=2
          if(id.eq.2) goto 130
        elseif(mstj(109).eq.2.and.id.eq.1) then
          parj(156)=y34*(wta(1)+wta(2)+wta(3)+wta(4))/(8.*wttot)
          if(wta(2)+wta(4).gt.rlu(0)*(wta(1)+wta(2)+wta(3)+wta(4))) id=2
          if(id.eq.2) goto 130
        endif
        mstj(120)=3
        if(mstj(109).eq.0.and.0.5*y34*(wtc(1)+wtc(2)+wtc(3)+wtc(4)).gt.
     &  rlu(0)*wttot) mstj(120)=4
        kfln=21

c...mass cuts. kinematical variables out.
        if(y12.le.cut+qme) njet=2
        if(njet.eq.2) goto 150
        q12=0.5*(1.-sqrt(1.-qme/y12))
        x1=1.-(1.-q12)*y234-q12*y134
        x4=1.-(1.-q12)*y134-q12*y234
        x2=1.-y124
        x12=(1.-q12)*y13+q12*y23
        x14=y12-0.5*qme
        if(y134*y234/((1.-x1)*(1.-x4)).le.rlu(0)) njet=2

c...qqbarqqbar events: string configuration, choose new flavour.
      else
        if(id.eq.1) then
          wtr=rlu(0)*(wtd(1)+wtd(2)+wtd(3)+wtd(4))
          if(wtr.lt.wtd(2)+wtd(3)+wtd(4)) id=2
          if(wtr.lt.wtd(3)+wtd(4)) id=3
          if(wtr.lt.wtd(4)) id=4
          if(id.ge.2) goto 130
        endif
        mstj(120)=5
        parj(156)=cf*tr*(wtd(1)+wtd(2)+wtd(3)+wtd(4))/(16.*wttot)
  140   kfln=1+int(5.*rlu(0))
        if(kfln.ne.kfl.and.0.2*parj(156).le.rlu(0)) goto 140
        if(kfln.eq.kfl.and.1.-0.8*parj(156).le.rlu(0)) goto 140
        if(kfln.gt.mstj(104)) njet=2
        pmqn=ulmass(kfln)
        qmen=(2.*pmqn/ecm)**2

c...mass cuts. kinematical variables out.
        if(y24.le.cut+qme.or.y13.le.1.1*qmen) njet=2
        if(njet.eq.2) goto 150
        q24=0.5*(1.-sqrt(1.-qme/y24))
        q13=0.5*(1.-sqrt(1.-qmen/y13))
        x1=1.-(1.-q24)*y123-q24*y134
        x4=1.-(1.-q24)*y134-q24*y123
        x2=1.-(1.-q13)*y234-q13*y124
        x12=(1.-q24)*((1.-q13)*y14+q13*y34)+q24*((1.-q13)*y12+q13*y23)
        x14=y24-0.5*qme
        x34=(1.-q24)*((1.-q13)*y23+q13*y12)+q24*((1.-q13)*y34+q13*y14)
        if(pmq**2+pmqn**2+min(x12,x34)*ecm**2.le.
     &  (parj(127)+pmq+pmqn)**2) njet=2
        if(y123*y134/((1.-x1)*(1.-x4)).le.rlu(0)) njet=2
      endif
  150 if(mstj(101).le.-2.and.njet.eq.2) goto 100

      return
      end

c*********************************************************************

      subroutine luxdif(nc,njet,kfl,ecm,chi,the,phi)

c...purpose: to give the angular orientation of events.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...charge. factors depending on polarization for qed case.
      qf=kchg(kfl,1)/3.
      poll=1.-parj(131)*parj(132)
      pold=parj(132)-parj(131)
      if(mstj(102).le.1.or.mstj(109).eq.1) then
        hf1=poll
        hf2=0.
        hf3=parj(133)**2
        hf4=0.

c...factors depending on flavour, energy and polarization for qfd case.
      else
        sff=1./(16.*paru(102)*(1.-paru(102)))
        sfw=ecm**4/((ecm**2-parj(123)**2)**2+(parj(123)*parj(124))**2)
        sfi=sfw*(1.-(parj(123)/ecm)**2)
        ae=-1.
        ve=4.*paru(102)-1.
        af=sign(1.,qf)
        vf=af-4.*qf*paru(102)
        hf1=qf**2*poll-2.*qf*vf*sfi*sff*(ve*poll-ae*pold)+
     &  (vf**2+af**2)*sfw*sff**2*((ve**2+ae**2)*poll-2.*ve*ae*pold)
        hf2=-2.*qf*af*sfi*sff*(ae*poll-ve*pold)+2.*vf*af*sfw*sff**2*
     &  (2.*ve*ae*poll-(ve**2+ae**2)*pold)
        hf3=parj(133)**2*(qf**2-2.*qf*vf*sfi*sff*ve+(vf**2+af**2)*
     &  sfw*sff**2*(ve**2-ae**2))
        hf4=-parj(133)**2*2.*qf*vf*sfw*(parj(123)*parj(124)/ecm**2)*
     &  sff*ae
      endif

c...mass factor. differential cross-sections for two-jet events.
      sq2=sqrt(2.)
      qme=0.
      if(mstj(103).ge.4.and.iabs(mstj(101)).le.1.and.mstj(102).le.1.and.
     &mstj(109).ne.1) qme=(2.*ulmass(kfl)/ecm)**2
      if(njet.eq.2) then
        sigu=4.*sqrt(1.-qme)
        sigl=2.*qme*sqrt(1.-qme)
        sigt=0.
        sigi=0.
        siga=0.
        sigp=4.

c...kinematical variables. reduce four-jet event to three-jet one.
      else
        if(njet.eq.3) then
          x1=2.*p(nc+1,4)/ecm
          x2=2.*p(nc+3,4)/ecm
        else
          ecmr=p(nc+1,4)+p(nc+4,4)+sqrt((p(nc+2,1)+p(nc+3,1))**2+
     &    (p(nc+2,2)+p(nc+3,2))**2+(p(nc+2,3)+p(nc+3,3))**2)
          x1=2.*p(nc+1,4)/ecmr
          x2=2.*p(nc+4,4)/ecmr
        endif

c...differential cross-sections for three-jet (or reduced four-jet).
        xq=(1.-x1)/(1.-x2)
        ct12=(x1*x2-2.*x1-2.*x2+2.+qme)/sqrt((x1**2-qme)*(x2**2-qme))
        st12=sqrt(1.-ct12**2)
        if(mstj(109).ne.1) then
          sigu=2.*x1**2+x2**2*(1.+ct12**2)-qme*(3.+ct12**2-x1-x2)-
     &    qme*x1/xq+0.5*qme*((x2**2-qme)*st12**2-2.*x2)*xq
          sigl=(x2*st12)**2-qme*(3.-ct12**2-2.5*(x1+x2)+x1*x2+qme)+
     &    0.5*qme*(x1**2-x1-qme)/xq+0.5*qme*((x2**2-qme)*ct12**2-x2)*xq
          sigt=0.5*(x2**2-qme-0.5*qme*(x2**2-qme)/xq)*st12**2
          sigi=((1.-0.5*qme*xq)*(x2**2-qme)*st12*ct12+qme*(1.-x1-x2+
     &    0.5*x1*x2+0.5*qme)*st12/ct12)/sq2
          siga=x2**2*st12/sq2
          sigp=2.*(x1**2-x2**2*ct12)

c...differential cross-sect for scalar gluons (no mass or qfd effects).
        else
          sigu=2.*(2.-x1-x2)**2-(x2*st12)**2
          sigl=(x2*st12)**2
          sigt=0.5*sigl
          sigi=-(2.-x1-x2)*x2*st12/sq2
          siga=0.
          sigp=0.
        endif
      endif

c...upper bounds for differential cross-section.
      hf1a=abs(hf1)
      hf2a=abs(hf2)
      hf3a=abs(hf3)
      hf4a=abs(hf4)
      sigmax=(2.*hf1a+hf3a+hf4a)*abs(sigu)+2.*(hf1a+hf3a+hf4a)*
     &abs(sigl)+2.*(hf1a+2.*hf3a+2.*hf4a)*abs(sigt)+2.*sq2*
     &(hf1a+2.*hf3a+2.*hf4a)*abs(sigi)+4.*sq2*hf2a*abs(siga)+
     &2.*hf2a*abs(sigp)

c...generate angular orientation according to differential cross-sect.
  100 chi=paru(2)*rlu(0)
      cthe=2.*rlu(0)-1.
      phi=paru(2)*rlu(0)
      cchi=cos(chi)
      schi=sin(chi)
      c2chi=cos(2.*chi)
      s2chi=sin(2.*chi)
      the=acos(cthe)
      sthe=sin(the)
      c2phi=cos(2.*(phi-parj(134)))
      s2phi=sin(2.*(phi-parj(134)))
      sig=((1.+cthe**2)*hf1+sthe**2*(c2phi*hf3-s2phi*hf4))*sigu+
     &2.*(sthe**2*hf1-sthe**2*(c2phi*hf3-s2phi*hf4))*sigl+
     &2.*(sthe**2*c2chi*hf1+((1.+cthe**2)*c2chi*c2phi-2.*cthe*s2chi*
     &s2phi)*hf3-((1.+cthe**2)*c2chi*s2phi+2.*cthe*s2chi*c2phi)*hf4)*
     &sigt-2.*sq2*(2.*sthe*cthe*cchi*hf1-2.*sthe*(cthe*cchi*c2phi-
     &schi*s2phi)*hf3+2.*sthe*(cthe*cchi*s2phi+schi*c2phi)*hf4)*sigi+
     &4.*sq2*sthe*cchi*hf2*siga+2.*cthe*hf2*sigp
      if(sig.lt.sigmax*rlu(0)) goto 100

      return
      end

c*********************************************************************

      subroutine luonia(kfl,ecm)

c...purpose: to generate upsilon and toponium decays into three
c...gluons or two gluons and a photon.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /lujets/,/ludat1/,/ludat2/

c...printout. check input parameters.
      if(mstu(12).ge.1) call lulist(0)
      if(kfl.lt.0.or.kfl.gt.8) then
        call luerrm(16,'(luonia:) called with unknown flavour code')
        if(mstu(21).ge.1) return
      endif
      if(ecm.lt.parj(127)+2.02*parf(101)) then
        call luerrm(16,'(luonia:) called with too small cm energy')
        if(mstu(21).ge.1) return
      endif

c...initial e+e- and onium state (optional).
      nc=0
      if(mstj(115).ge.2) then
        nc=nc+2
        call lu1ent(nc-1,11,0.5*ecm,0.,0.)
        k(nc-1,1)=21
        call lu1ent(nc,-11,0.5*ecm,paru(1),0.)
        k(nc,1)=21
      endif
      kflc=iabs(kfl)
      if(mstj(115).ge.3.and.kflc.ge.5) then
        nc=nc+1
        kf=110*kflc+3
        mstu10=mstu(10)
        mstu(10)=1
        p(nc,5)=ecm
        call lu1ent(nc,kf,ecm,0.,0.)
        k(nc,1)=21
        k(nc,3)=1
        mstu(10)=mstu10
      endif

c...choose x1 and x2 according to matrix element.
      ntry=0
  100 x1=rlu(0)
      x2=rlu(0)
      x3=2.-x1-x2
      if(x3.ge.1..or.((1.-x1)/(x2*x3))**2+((1.-x2)/(x1*x3))**2+
     &((1.-x3)/(x1*x2))**2.le.2.*rlu(0)) goto 100
      ntry=ntry+1
      njet=3
      if(mstj(101).le.4) call lu3ent(nc+1,21,21,21,ecm,x1,x3)
      if(mstj(101).ge.5) call lu3ent(-(nc+1),21,21,21,ecm,x1,x3)

c...photon-gluon-gluon events. small system modifications. jet origin.
      mstu(111)=mstj(108)
      if(mstj(108).eq.2.and.(mstj(101).eq.0.or.mstj(101).eq.1))
     &mstu(111)=1
      paru(112)=parj(121)
      if(mstu(111).eq.2) paru(112)=parj(122)
      qf=0.
      if(kflc.ne.0) qf=kchg(kflc,1)/3.
      rgam=7.2*qf**2*paru(101)/ulalps(ecm**2)
      mk=0
      ecmc=ecm
      if(rlu(0).gt.rgam/(1.+rgam)) then
        if(1.-max(x1,x2,x3).le.max((parj(126)/ecm)**2,parj(125)))
     &  njet=2
        if(njet.eq.2.and.mstj(101).le.4) call lu2ent(nc+1,21,21,ecm)
        if(njet.eq.2.and.mstj(101).ge.5) call lu2ent(-(nc+1),21,21,ecm)
      else
        mk=1
        ecmc=sqrt(1.-x1)*ecm
        if(ecmc.lt.2.*parj(127)) goto 100
        k(nc+1,1)=1
        k(nc+1,2)=22
        k(nc+1,4)=0
        k(nc+1,5)=0
        if(mstj(101).ge.5) k(nc+2,4)=mstu(5)*(nc+3)
        if(mstj(101).ge.5) k(nc+2,5)=mstu(5)*(nc+3)
        if(mstj(101).ge.5) k(nc+3,4)=mstu(5)*(nc+2)
        if(mstj(101).ge.5) k(nc+3,5)=mstu(5)*(nc+2)
        njet=2
        if(ecmc.lt.4.*parj(127)) then
          mstu10=mstu(10)
          mstu(10)=1
          p(nc+2,5)=ecmc
          call lu1ent(nc+2,83,0.5*(x2+x3)*ecm,paru(1),0.)
          mstu(10)=mstu10
          njet=0
        endif
      endif
      do 110 ip=nc+1,n
  110 k(ip,3)=k(ip,3)+(mstj(115)/2)+(kflc/5)*(mstj(115)/3)*(nc-1)

c...differential cross-sections. upper limit for cross-section.
      if(mstj(106).eq.1) then
        sq2=sqrt(2.)
        hf1=1.-parj(131)*parj(132)
        hf3=parj(133)**2
        ct13=(x1*x3-2.*x1-2.*x3+2.)/(x1*x3)
        st13=sqrt(1.-ct13**2)
        sigl=0.5*x3**2*((1.-x2)**2+(1.-x3)**2)*st13**2
        sigu=(x1*(1.-x1))**2+(x2*(1.-x2))**2+(x3*(1.-x3))**2-sigl
        sigt=0.5*sigl
        sigi=(sigl*ct13/st13+0.5*x1*x3*(1.-x2)**2*st13)/sq2
        sigmax=(2.*hf1+hf3)*abs(sigu)+2.*(hf1+hf3)*abs(sigl)+2.*(hf1+
     &  2.*hf3)*abs(sigt)+2.*sq2*(hf1+2.*hf3)*abs(sigi)

c...angular orientation of event.
  120   chi=paru(2)*rlu(0)
        cthe=2.*rlu(0)-1.
        phi=paru(2)*rlu(0)
        cchi=cos(chi)
        schi=sin(chi)
        c2chi=cos(2.*chi)
        s2chi=sin(2.*chi)
        the=acos(cthe)
        sthe=sin(the)
        c2phi=cos(2.*(phi-parj(134)))
        s2phi=sin(2.*(phi-parj(134)))
        sig=((1.+cthe**2)*hf1+sthe**2*c2phi*hf3)*sigu+2.*(sthe**2*hf1-
     &  sthe**2*c2phi*hf3)*sigl+2.*(sthe**2*c2chi*hf1+((1.+cthe**2)*
     &  c2chi*c2phi-2.*cthe*s2chi*s2phi)*hf3)*sigt-2.*sq2*(2.*sthe*cthe*
     &  cchi*hf1-2.*sthe*(cthe*cchi*c2phi-schi*s2phi)*hf3)*sigi
        if(sig.lt.sigmax*rlu(0)) goto 120
        call ludbrb(nc+1,n,0.,chi,0d0,0d0,0d0)
        call ludbrb(nc+1,n,the,phi,0d0,0d0,0d0)
      endif

c...generate parton shower. rearrange along strings and check.
      if(mstj(101).ge.5.and.njet.ge.2) then
        call lushow(nc+mk+1,-njet,ecmc)
        mstj14=mstj(14)
        if(mstj(105).eq.-1) mstj(14)=-1
        if(mstj(105).ge.0) mstu(28)=0
        call luprep(0)
        mstj(14)=mstj14
        if(mstj(105).ge.0.and.mstu(28).ne.0) goto 100
      endif

c...generate fragmentation. information for lutabu:
      if(mstj(105).eq.1) call luexec
      mstu(161)=110*kflc+3
      mstu(162)=0

      return
      end

c*********************************************************************

      subroutine luhepc(mconv)

c...purpose: to convert jetset event record contents to or from
c...the standard event record commonblock.
      parameter (nmxhep=2000)
      common/hepevt/nevhep,nhep,isthep(nmxhep),idhep(nmxhep),
     &jmohep(2,nmxhep),jdahep(2,nmxhep),phep(5,nmxhep),vhep(4,nmxhep)
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      save /hepevt/
      save /lujets/,/ludat1/,/ludat2/

c...conversion from jetset to standard, the easy part.
      if(mconv.eq.1) then
        nevhep=0
        if(n.gt.nmxhep) call luerrm(8,
     &  '(luhepc:) no more space in /hepevt/')
        nhep=min(n,nmxhep)
        do 140 i=1,nhep
        isthep(i)=0
        if(k(i,1).ge.1.and.k(i,1).le.10) isthep(i)=1
        if(k(i,1).ge.11.and.k(i,1).le.20) isthep(i)=2
        if(k(i,1).ge.21.and.k(i,1).le.30) isthep(i)=3
        if(k(i,1).ge.31.and.k(i,1).le.100) isthep(i)=k(i,1)
        idhep(i)=k(i,2)
        jmohep(1,i)=k(i,3)
        jmohep(2,i)=0
        if(k(i,1).ne.3.and.k(i,1).ne.13.and.k(i,1).ne.14) then
          jdahep(1,i)=k(i,4)
          jdahep(2,i)=k(i,5)
        else
          jdahep(1,i)=0
          jdahep(2,i)=0
        endif
        do 100 j=1,5
  100   phep(j,i)=p(i,j)
        do 110 j=1,4
  110   vhep(j,i)=v(i,j)

c...fill in missing mother information.
        if(i.ge.3.and.k(i,1).eq.21.and.k(i,3).eq.0) then
          imo1=i-2
          if(i.ge.4.and.k(i-1,1).eq.21.and.k(i-1,3).eq.0) imo1=imo1-1
          jmohep(1,i)=imo1
          jmohep(2,i)=imo1+1
        elseif(k(i,2).ge.91.and.k(i,2).le.93) then
          i1=k(i,3)-1
  120     i1=i1+1
          if(i1.ge.i) call luerrm(8,
     &    '(luhepc:) translation of inconsistent event history')
          if(i1.lt.i.and.k(i1,1).ne.1.and.k(i1,1).ne.11) goto 120
          kc=lucomp(k(i1,2))
          if(i1.lt.i.and.kc.eq.0) goto 120
          if(i1.lt.i.and.kchg(kc,2).eq.0) goto 120
          jmohep(2,i)=i1
        elseif(k(i,2).eq.94) then
          njet=2
          if(nhep.ge.i+3.and.k(i+3,3).le.i) njet=3
          if(nhep.ge.i+4.and.k(i+4,3).le.i) njet=4
          jmohep(2,i)=mod(k(i+njet,4)/mstu(5),mstu(5))
          if(jmohep(2,i).eq.jmohep(1,i)) jmohep(2,i)=
     &    mod(k(i+1,4)/mstu(5),mstu(5))
        endif

c...fill in missing daughter information.
        if(k(i,2).eq.94.and.mstu(16).ne.2) then
          do 130 i1=jdahep(1,i),jdahep(2,i)
          i2=mod(k(i1,4)/mstu(5),mstu(5))
  130     jdahep(1,i2)=i
        endif
        if(k(i,2).ge.91.and.k(i,2).le.94) goto 140
        i1=jmohep(1,i)
        if(i1.le.0.or.i1.gt.nhep) goto 140
        if(k(i1,1).ne.13.and.k(i1,1).ne.14) goto 140
        if(jdahep(1,i1).eq.0) then
          jdahep(1,i1)=i
        else
          jdahep(2,i1)=i
        endif
  140   continue
        do 150 i=1,nhep
        if(k(i,1).ne.13.and.k(i,1).ne.14) goto 150
        if(jdahep(2,i).eq.0) jdahep(2,i)=jdahep(1,i)
  150   continue

c...conversion from standard to jetset, the easy part.
      else
        if(nhep.gt.mstu(4)) call luerrm(8,
     &  '(luhepc:) no more space in /lujets/')
        n=min(nhep,mstu(4))
        nkq=0
        kqsum=0
        do 180 i=1,n
        k(i,1)=0
        if(isthep(i).eq.1) k(i,1)=1
        if(isthep(i).eq.2) k(i,1)=11
        if(isthep(i).eq.3) k(i,1)=21
        k(i,2)=idhep(i)
        k(i,3)=jmohep(1,i)
        k(i,4)=jdahep(1,i)
        k(i,5)=jdahep(2,i)
        do 160 j=1,5
  160   p(i,j)=phep(j,i)
        do 170 j=1,4
  170   v(i,j)=vhep(j,i)
        v(i,5)=0.
        if(isthep(i).eq.2.and.phep(4,i).gt.phep(5,i)) then
          i1=jdahep(1,i)
          if(i1.gt.0.and.i1.le.nhep) v(i,5)=(vhep(4,i1)-vhep(4,i))*
     &    phep(5,i)/phep(4,i)
        endif

c...fill in missing information on colour connection in jet systems.
        if(isthep(i).eq.1) then
          kc=lucomp(k(i,2))
          kq=0
          if(kc.ne.0) kq=kchg(kc,2)*isign(1,k(i,2))
          if(kq.ne.0) nkq=nkq+1
          if(kq.ne.2) kqsum=kqsum+kq
          if(kq.ne.0.and.kqsum.ne.0) then
            k(i,1)=2
          elseif(kq.eq.2.and.i.lt.n) then
            if(k(i+1,2).eq.21) k(i,1)=2
          endif
        endif
  180   continue
        if(nkq.eq.1.or.kqsum.ne.0) call luerrm(8,
     &  '(luhepc:) input parton configuration not colour singlet')
      endif

      end

c*********************************************************************

      subroutine lutest(mtest)

c...purpose: to provide a simple program (disguised as subroutine) to
c...run at installation as a check that the program works as intended.
      common/lujets/n,k(4000,5),p(4000,5),v(4000,5)
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      save /lujets/,/ludat1/
      dimension psum(5),pini(6),pfin(6)

c...loop over events to be generated.
      if(mtest.ge.1) call lutabu(20)
      nerr=0
      do 170 iev=1,600

c...reset parameter values. switch on some nonstandard features.
      mstj(1)=1
      mstj(3)=0
      mstj(11)=1
      mstj(42)=2
      mstj(43)=4
      mstj(44)=2
      parj(17)=0.1
      parj(22)=1.5
      parj(43)=1.
      parj(54)=-0.05
      mstj(101)=5
      mstj(104)=5
      mstj(105)=0
      mstj(107)=1
      if(iev.eq.301.or.iev.eq.351.or.iev.eq.401) mstj(116)=3

c...ten events each for some single jets configurations.
      if(iev.le.50) then
        ity=(iev+9)/10
        mstj(3)=-1
        if(ity.eq.3.or.ity.eq.4) mstj(11)=2
        if(ity.eq.1) call lu1ent(1,1,15.,0.,0.)
        if(ity.eq.2) call lu1ent(1,3101,15.,0.,0.)
        if(ity.eq.3) call lu1ent(1,-2203,15.,0.,0.)
        if(ity.eq.4) call lu1ent(1,-4,30.,0.,0.)
        if(ity.eq.5) call lu1ent(1,21,15.,0.,0.)

c...ten events each for some simple jet systems; string fragmentation.
      elseif(iev.le.130) then
        ity=(iev-41)/10
        if(ity.eq.1) call lu2ent(1,1,-1,40.)
        if(ity.eq.2) call lu2ent(1,4,-4,30.)
        if(ity.eq.3) call lu2ent(1,2,2103,100.)
        if(ity.eq.4) call lu2ent(1,21,21,40.)
        if(ity.eq.5) call lu3ent(1,2101,21,-3203,30.,0.6,0.8)
        if(ity.eq.6) call lu3ent(1,5,21,-5,40.,0.9,0.8)
        if(ity.eq.7) call lu3ent(1,21,21,21,60.,0.7,0.5)
        if(ity.eq.8) call lu4ent(1,2,21,21,-2,40.,0.4,0.64,0.6,0.12,0.2)

c...seventy events with independent fragmentation and momentum cons.
      elseif(iev.le.200) then
        ity=1+(iev-131)/16
        mstj(2)=1+mod(iev-131,4)
        mstj(3)=1+mod((iev-131)/4,4)
        if(ity.eq.1) call lu2ent(1,4,-5,40.)
        if(ity.eq.2) call lu3ent(1,3,21,-3,40.,0.9,0.4)
        if(ity.eq.3) call lu4ent(1,2,21,21,-2,40.,0.4,0.64,0.6,0.12,0.2)
        if(ity.ge.4) call lu4ent(1,2,-3,3,-2,40.,0.4,0.64,0.6,0.12,0.2)

c...a hundred events with random jets (check invariant mass).
      elseif(iev.le.300) then
  100   do 110 j=1,5
  110   psum(j)=0.
        njet=2.+6.*rlu(0)
        do 120 i=1,njet
        kfl=21
        if(i.eq.1) kfl=int(1.+4.*rlu(0))
        if(i.eq.njet) kfl=-int(1.+4.*rlu(0))
        ejet=5.+20.*rlu(0)
        theta=acos(2.*rlu(0)-1.)
        phi=6.2832*rlu(0)
        if(i.lt.njet) call lu1ent(-i,kfl,ejet,theta,phi)
        if(i.eq.njet) call lu1ent(i,kfl,ejet,theta,phi)
        if(i.eq.1.or.i.eq.njet) psum(5)=psum(5)+ulmass(kfl)
        do 120 j=1,4
  120   psum(j)=psum(j)+p(i,j)
        if(psum(4)**2-psum(1)**2-psum(2)**2-psum(3)**2.lt.
     &  (psum(5)+parj(32))**2) goto 100

c...fifty e+e- continuum events with matrix elements.
      elseif(iev.le.350) then
        mstj(101)=2
        call lueevt(0,40.)

c...fifty e+e- continuum event with varying shower options.
      elseif(iev.le.400) then
        mstj(42)=1+mod(iev,2)
        mstj(43)=1+mod(iev/2,4)
        mstj(44)=mod(iev/8,3)
        call lueevt(0,90.)

c...fifty e+e- continuum events with coherent shower, including top.
      elseif(iev.le.450) then
        mstj(104)=6
        call lueevt(0,500.)

c...fifty upsilon decays to ggg or gammagg with coherent shower.
      elseif(iev.le.500) then
        call luonia(5,9.46)

c...one decay each for some heavy mesons.
      elseif(iev.le.560) then
        ity=iev-501
        kfls=2*(ity/20)+1
        kflb=8-mod(ity/5,4)
        kflc=kflb-mod(ity,5)
        call lu1ent(1,100*kflb+10*kflc+kfls,0.,0.,0.)

c...one decay each for some heavy baryons.
      elseif(iev.le.600) then
        ity=iev-561
        kfls=2*(ity/20)+2
        kfla=8-mod(ity/5,4)
        kflb=kfla-mod(ity,5)
        kflc=max(1,kflb-1)
        call lu1ent(1,1000*kfla+100*kflb+10*kflc+kfls,0.,0.,0.)
      endif

c...generate event. find total momentum, energy and charge.
      do 130 j=1,4
  130 pini(j)=plu(0,j)
      pini(6)=plu(0,6)
      call luexec
      do 140 j=1,4
  140 pfin(j)=plu(0,j)
      pfin(6)=plu(0,6)

c...check conservation of energy, momentum and charge;
c...usually exact, but only approximate for single jets.
      merr=0
      if(iev.le.50) then
        if((pfin(1)-pini(1))**2+(pfin(2)-pini(2))**2.ge.4.) merr=merr+1
        epzrem=pini(4)+pini(3)-pfin(4)-pfin(3)
        if(epzrem.lt.0..or.epzrem.gt.2.*parj(31)) merr=merr+1
        if(abs(pfin(6)-pini(6)).gt.2.1) merr=merr+1
      else
        do 150 j=1,4
  150   if(abs(pfin(j)-pini(j)).gt.0001*pini(4)) merr=merr+1
        if(abs(pfin(6)-pini(6)).gt.0.1) merr=merr+1
      endif
      if(merr.ne.0) write(mstu(11),5000) (pini(j),j=1,4),pini(6),
     &(pfin(j),j=1,4),pfin(6)

c...check that all kf codes are known ones, and that partons/particles
c...satisfy energy-momentum-mass relation. store particle statistics.
      do 160 i=1,n
      if(k(i,1).gt.20) goto 160
      if(lucomp(k(i,2)).eq.0) then
        write(mstu(11),5100) i
        merr=merr+1
      endif
      pd=p(i,4)**2-p(i,1)**2-p(i,2)**2-p(i,3)**2-p(i,5)**2
      if(abs(pd).gt.max(0.1,0.001*p(i,4)**2).or.p(i,4).lt.0.) then
        write(mstu(11),5200) i
        merr=merr+1
      endif
  160 continue
      if(mtest.ge.1) call lutabu(21)

c...list all erroneous events and some normal ones.
      if(merr.ne.0.or.mstu(24).ne.0.or.mstu(28).ne.0) then
        call lulist(2)
      elseif(mtest.ge.1.and.mod(iev-5,100).eq.0) then
        call lulist(1)
      endif

c...stop execution if too many errors. endresult of run.
      if(merr.ne.0) nerr=nerr+1
      if(nerr.ge.10) then
        write(mstu(11),5300) iev
        stop
      endif
  170 continue
      if(mtest.ge.1) call lutabu(22)
      write(mstu(11),5400) nerr

c...reset commonblock variables changed during run.
      mstj(2)=3
      parj(17)=0.
      parj(22)=1.
      parj(43)=0.5
      parj(54)=0.
      mstj(105)=1
      mstj(107)=0

c...format statements for output.
 5000 format(/' momentum, energy and/or charge were not conserved ',
     &'in following event'/' sum of',9x,'px',11x,'py',11x,'pz',11x,
     &'e',8x,'charge'/' before',2x,4(1x,f12.5),1x,f8.2/' after',3x,
     &4(1x,f12.5),1x,f8.2)
 5100 format(/5x,'entry no.',i4,' in following event not known code')
 5200 format(/5x,'entry no.',i4,' in following event has faulty ',
     &'kinematics')
 5300 format(/5x,'ten errors experienced by event ',i3/
     &5x,'something is seriously wrong! execution stopped now!')
 5400 format(/5x,'number of erroneous or suspect events in run:',i3/
     &5x,'(0 fine, 1 acceptable if a single jet, ',
     &'>=2 something is wrong)')

      return
      end

c*********************************************************************

      block data ludata

c...purpose: to give default values to parameters and particle and
c...decay data.
      common/ludat1/mstu(200),paru(200),mstj(200),parj(200)
      common/ludat2/kchg(500,3),pmas(500,4),parf(2000),vckm(4,4)
      common/ludat3/mdcy(500,3),mdme(2000,2),brat(2000),kfdp(2000,5)
      common/ludat4/chaf(500)
      character chaf*8
      common/ludatr/mrlu(6),rrlu(100)
      save /ludat1/,/ludat2/,/ludat3/,/ludat4/,/ludatr/

c...ludat1, containing status codes and most parameters.
      data mstu/
     &    0,    0,    0, 4000,10000,  500, 2000,    0,    0,    2,
     1    6,    1,    1,    0,    1,    1,    0,    0,    0,    0,
     2    2,   10,    0,    0,    1,   10,    0,    0,    0,    0,
     3    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
     4    2,    2,    1,    4,    2,    1,    1,    0,    0,    0,
     5   25,   24,    0,    1,    0,    0,    0,    0,    0,    0,
     6    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
     7  30*0,
     &    1,    0,    0,    0,    0,    0,    0,    0,    0,    0,
     1    1,    5,    3,   23,    0,    0,    0,    0,    0,    0,
     2  60*0,
     8    7,    3, 1992,    2,   21,    0,    0,    0,    0,    0,
     9    0,    0,    0,    0,    0,    0,    0,    0,    0,    0/
      data paru/
     & 3.1415927, 6.2831854, 0.1973, 5.068, 0.3894, 2.568,   4*0.,
     1 0.001, 0.09, 0.01,  0.,   0.,   0.,   0.,   0.,   0.,   0.,
     2   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     3   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     4  2.0,  1.0, 0.25,  2.5, 0.05,   0.,   0., 0.0001, 0.,   0.,
     5  2.5,  1.5,  7.0,  1.0,  0.5,  2.0,  3.2,   0.,   0.,   0.,
     6  40*0.,
     & 0.00729735, 0.230, 0., 0., 0.,  0.,   0.,   0.,   0.,   0.,
     1 0.20, 0.25,  1.0,  4.0,  10.,   0.,   0.,   0.,   0.,   0.,
     2 -0.693, -1.0, 0.387, 1.0, -0.08, -1.0, 1.0, 1.0, 1.0,   0.,
     3  1.0, -1.0,  1.0, -1.0,  1.0,   0.,   0.,   0.,   0.,   0.,
     4  5.0,  1.0,  1.0,   0.,  1.0,  1.0,   0.,   0.,   0.,   0.,
     5  1.0,   0.,   0.,   0., 1000., 1.0,  1.0,  1.0,  1.0,   0.,
     6  1.0,  1.0,  1.0,  1.0,  1.0,   0.,   0.,   0.,   0.,   0.,
     7  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,   0.,   0.,   0.,
     8  1.0,  1.0,  1.0,  0.0,  0.0,  1.0,  1.0,  0.0,  0.0,   0.,
     9   0.,   0.,   0.,   0.,  1.0,   0.,   0.,   0.,   0.,   0./
      data mstj/
     &    1,    3,    0,    0,    0,    0,    0,    0,    0,    0,
     1    1,    2,    0,    1,    0,    0,    0,    0,    0,    0,
     2    2,    1,    1,    2,    1,    0,    0,    0,    0,    0,
     3    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
     4    1,    2,    4,    2,    5,    0,    1,    0,    0,    0,
     5    0,    3,    0,    0,    0,    0,    0,    0,    0,    0,
     6  40*0,
     &    5,    2,    7,    5,    1,    1,    0,    2,    0,    1,
     1    0,    0,    0,    0,    1,    1,    0,    0,    0,    0,
     2  80*0/
      data parj/
     & 0.10, 0.30, 0.40, 0.05, 0.50, 0.50, 0.50,   0.,   0.,   0.,
     1 0.50, 0.60, 0.75,   0.,   0.,   0.,   0.,  1.0,  1.0,   0.,
     2 0.35,  1.0,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     3 0.10,  1.0,  0.8,  1.5,   0.,  2.0,  0.2,  2.5,  0.6,   0.,
     4  0.5,  0.9,  0.5,  0.9,  0.5,  1.0,  1.0,  1.0,   0.,   0.,
     5 0.77, 0.77, 0.77,   0.,   0.,   0.,   0.,   0.,  1.0,   0.,
     6  4.5,  0.7,  0., 0.003,  0.5,  0.5,   0.,   0.,   0.,   0.,
     7  10., 1000., 100., 1000., 0.,  0.7,  10.,   0.,   0.,   0.,
     8  0.4,  1.0,  1.0,   0.,  10.,  10.,   0.,   0.,   0.,   0.,
     9 0.02,  1.0,  0.2,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     &   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     1   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     2  1.5,  0.5, 91.2, 2.40, 0.02,  2.0,  1.0, 0.25,0.002,   0.,
     3   0.,   0.,   0.,   0., 0.01, 0.99,   0.,   0.,  0.2,   0.,
     4  60*0./

c...ludat2, with particle data and flavour treatment parameters.
      data (kchg(i,1),i=   1, 500)/-1,2,-1,2,-1,2,-1,2,2*0,-3,0,-3,0,
     &-3,0,-3,6*0,3,9*0,3,2*0,3,0,-1,44*0,2,-1,2,-1,2,3,11*0,3,0,2*3,0,
     &3,0,3,12*0,3,0,2*3,0,3,0,3,12*0,3,0,2*3,0,3,0,3,12*0,3,0,2*3,0,3,
     &0,3,12*0,3,0,2*3,0,3,0,3,12*0,3,0,2*3,0,3,0,3,72*0,3,0,3,28*0,3,
     &2*0,3,8*0,-3,8*0,3,0,-3,0,3,-3,3*0,3,6,0,3,5*0,-3,0,3,-3,0,-3,
     &4*0,-3,0,3,6,-3,0,3,-3,0,-3,0,3,6,0,3,5*0,-3,0,3,-3,0,-3,114*0/
      data (kchg(i,2),i=   1, 500)/8*1,12*0,2,17*0,1,50*0,-1,410*0/
      data (kchg(i,3),i=   1, 500)/8*1,2*0,8*1,5*0,1,9*0,1,2*0,1,0,2*1,
     &41*0,1,0,7*1,10*0,9*1,11*0,9*1,11*0,9*1,11*0,9*1,11*0,9*1,11*0,
     &9*1,71*0,3*1,22*0,1,5*0,1,0,2*1,6*0,1,0,2*1,6*0,2*1,0,5*1,0,6*1,
     &4*0,6*1,4*0,16*1,4*0,6*1,114*0/
      data (pmas(i,1),i=   1, 500)/0.0099,0.0056,0.199,1.35,5.,2*120.,
     &200.,2*0.,0.00051,0.,0.1057,0.,1.7841,0.,100.,5*0.,91.2,80.,50.,
     &6*0.,500.,900.,500.,3*300.,0.,200.,5000.,60*0.,0.1396,0.4977,
     &0.4936,1.8693,1.8645,1.9693,5.2794,5.2776,5.47972,0.,0.135,
     &0.5488,0.9575,2.9796,9.4,2*238.,397.,2*0.,0.7669,0.8962,0.8921,
     &2.0101,2.0071,2.1127,2*5.3354,5.5068,0.,0.77,0.782,1.0194,3.0969,
     &9.4603,2*238.,397.,2*0.,1.233,2*1.3,2*2.322,2.51,2*5.73,5.97,0.,
     &1.233,1.17,1.41,3.46,9.875,2*238.42,397.41992,2*0.,0.983,2*1.429,
     &2*2.272,2.46,2*5.68,5.92,0.,0.983,1.,1.4,3.4151,9.8598,
     &2*238.39999,397.3999,2*0.,1.26,2*1.401,2*2.372,2.56,2*5.78,6.02,
     &0.,1.26,1.283,1.422,3.5106,9.8919,2*238.5,397.5,2*0.,1.318,
     &2*1.426,2*2.422,2.61,2*5.83,6.07,0.,1.318,1.274,1.525,3.5563,
     &9.9132,2*238.45,397.44995,2*0.,2*0.4977,83*0.,1.1156,5*0.,2.2849,
     &0.,2*2.46,6*0.,5.62,0.,2*5.84,6*0.,0.9396,0.9383,0.,1.1974,
     &1.1926,1.1894,1.3213,1.3149,0.,2.454,2.4529,2.4522,2*2.55,2.73,
     &4*0.,3*5.8,2*5.96,6.12,4*0.,1.234,1.233,1.232,1.231,1.3872,
     &1.3837,1.3828,1.535,1.5318,1.6724,3*2.5,2*2.63,2.8,4*0.,3*5.81,
     &2*5.97,6.13,114*0./
      data (pmas(i,2),i=   1, 500)/22*0.,2.5,2.1,88*0.,0.0002,0.001,
     &6*0.,0.149,0.0505,0.0513,7*0.,0.153,0.0085,0.0044,7*0.,0.15,
     &2*0.09,2*0.06,0.04,3*0.1,0.,0.15,0.335,0.08,2*0.01,5*0.,0.057,
     &2*0.287,2*0.06,0.04,3*0.1,0.,0.057,0.,0.25,0.0135,6*0.,0.4,
     &2*0.184,2*0.06,0.04,3*0.1,0.,0.4,0.025,0.055,0.00135,6*0.,0.11,
     &0.115,0.099,2*0.06,4*0.1,0.,0.11,0.185,0.076,0.0026,146*0.,
     &4*0.115,0.039,2*0.036,0.0099,0.0091,131*0./
      data (pmas(i,3),i=   1, 500)/22*0.,2*20.,88*0.,0.002,0.005,6*0.,
     &0.4,2*0.2,7*0.,0.4,0.1,0.015,7*0.,0.25,2*0.01,3*0.08,2*0.2,0.12,
     &0.,0.25,0.2,0.001,2*0.02,5*0.,0.05,2*0.4,3*0.08,2*0.2,0.12,0.,
     &0.05,0.,0.35,0.05,6*0.,3*0.3,2*0.08,0.06,2*0.2,0.12,0.,0.3,0.05,
     &0.025,0.001,6*0.,0.25,4*0.12,4*0.2,0.,0.25,0.17,0.2,0.01,146*0.,
     &4*0.14,0.04,2*0.035,2*0.05,131*0./
      data (pmas(i,4),i=   1, 500)/12*0.,658650.,0.,0.091,68*0.,0.1,
     &0.43,15*0.,7803.,0.,3709.,0.32,0.128,0.131,3*0.393,84*0.,0.,
     &26*0.,15540.,26.75,83*0.,78.88,5*0.,0.054,0.,2*0.13,6*0.,0.393,
     &0.,2*0.393,9*0.,44.3,0.,24.,49.10001,86.89999,6*0.,0.13,9*0.,
     &0.393,13*0.,24.60001,130*0./
      data parf/
     &  0.5, 0.25,  0.5, 0.25,   1.,  0.5,   0.,   0.,   0.,   0.,
     1  0.5,   0.,  0.5,   0.,   1.,   1.,   0.,   0.,   0.,   0.,
     2  0.5,   0.,  0.5,   0.,   1.,   1.,   0.,   0.,   0.,   0.,
     3  0.5,   0.,  0.5,   0.,   1.,   1.,   0.,   0.,   0.,   0.,
     4  0.5,   0.,  0.5,   0.,   1.,   1.,   0.,   0.,   0.,   0.,
     5  0.5,   0.,  0.5,   0.,   1.,   1.,   0.,   0.,   0.,   0.,
     6 0.75,  0.5,   0., 0.1667, 0.0833, 0.1667, 0., 0., 0.,   0.,
     7   0.,   0.,   1., 0.3333, 0.6667, 0.3333, 0., 0., 0.,   0.,
     8   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     9   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     & 0.325, 0.325, 0.5, 1.6,  5.0,   0.,   0.,   0.,   0.,   0.,
     1   0., 0.11, 0.16, 0.048, 0.50, 0.45, 0.55, 0.60,  0.,   0.,
     2  0.2,  0.1,   0.,   0.,   0.,   0.,   0.,   0.,   0.,   0.,
     3  1870*0./
      data ((vckm(i,j),j=1,4),i=1,4)/
     1  0.95150,  0.04847,  0.00003,  0.00000,
     2  0.04847,  0.94936,  0.00217,  0.00000,
     3  0.00003,  0.00217,  0.99780,  0.00000,
     4  0.00000,  0.00000,  0.00000,  1.00000/

c...ludat3, with particle decay parameters and data.
      data (mdcy(i,1),i=   1, 500)/14*0,1,0,1,5*0,3*1,6*0,1,0,1,2*0,1,
     &0,2*1,42*0,7*1,12*0,1,0,6*1,0,8*1,2*0,9*1,0,8*1,2*0,9*1,0,8*1,
     &2*0,9*1,0,8*1,2*0,9*1,0,8*1,2*0,9*1,0,8*1,3*0,1,83*0,1,5*0,1,0,
     &2*1,6*0,1,0,2*1,9*0,5*1,0,6*1,4*0,6*1,4*0,16*1,4*0,6*1,114*0/
      data (mdcy(i,2),i=   1, 500)/1,9,17,25,33,41,49,57,2*0,65,69,71,
     &76,78,118,120,125,2*0,127,136,148,164,184,6*0,201,0,223,246,266,
     &284,0,293,294,42*0,303,304,308,317,320,325,327,11*0,347,348,350,
     &356,477,645,677,678,679,0,680,682,688,694,695,696,697,698,2*0,
     &699,700,703,706,709,711,712,713,714,0,715,716,721,729,732,741,
     &756,757,2*0,758,759,764,769,771,773,774,776,778,0,780,781,784,
     &788,789,790,792,793,2*0,794,797,799,801,805,809,811,815,819,0,
     &823,826,830,834,836,838,840,841,2*0,842,844,846,848,850,852,855,
     &857,859,0,862,864,877,881,883,885,887,888,2*0,889,895,906,917,
     &925,933,938,946,954,0,959,966,974,976,978,980,982,983,2*0,984,
     &992,83*0,994,5*0,998,0,1072,1073,6*0,1074,0,1075,1076,9*0,1077,
     &1079,1080,1083,1084,0,1086,1087,1088,1089,1090,1091,4*0,1092,
     &1093,1094,1095,1096,1097,4*0,1098,1099,1102,1105,1106,1109,1112,
     &1115,1117,1119,1123,1124,1125,1126,1128,1130,4*0,1131,1132,1133,
     &1134,1135,1136,114*0/
      data (mdcy(i,3),i=   1, 500)/8*8,2*0,4,2,5,2,40,2,5,2,2*0,9,12,
     &16,20,17,6*0,22,0,23,20,18,9,0,1,9,42*0,1,4,9,3,5,2,20,11*0,1,2,
     &6,121,168,32,3*1,0,2,2*6,5*1,2*0,1,3*3,2,4*1,0,1,5,8,3,9,15,2*1,
     &2*0,1,2*5,2*2,1,3*2,0,1,3,4,2*1,2,2*1,2*0,3,2*2,2*4,2,3*4,0,3,
     &2*4,3*2,2*1,2*0,5*2,3,2*2,3,0,2,13,4,3*2,2*1,2*0,6,2*11,2*8,5,
     &2*8,5,0,7,8,4*2,2*1,2*0,8,2,83*0,4,5*0,74,0,2*1,6*0,1,0,2*1,9*0,
     &2,1,3,1,2,0,6*1,4*0,6*1,4*0,1,2*3,1,3*3,2*2,4,3*1,2*2,1,4*0,6*1,
     &114*0/
      data (mdme(i,1),i=   1,2000)/6*1,-1,7*1,-1,7*1,-1,7*1,-1,7*1,-1,
     &7*1,-1,85*1,2*-1,7*1,2*-1,3*1,-1,6*1,2*-1,6*1,2*-1,3*1,-1,3*1,-1,
     &3*1,5*-1,3*1,-1,6*1,2*-1,3*1,-1,11*1,2*-1,6*1,8*-1,3*1,-1,3*1,-1,
     &3*1,5*-1,3*1,4*-1,6*1,2*-1,3*1,-1,5*1,-1,8*1,2*-1,3*1,-1,9*1,-1,
     &3*1,-1,6*1,2*-1,2*1,-1,16*1,-1,2*1,3*-1,470*1,2*0,1204*1/
      data (mdme(i,2),i=   1,2000)/70*102,42,6*102,2*42,2*0,7*41,2*0,
     &23*41,6*102,45,27*102,8*32,8*0,16*32,4*0,8*32,4*0,32,4*0,8*32,
     &14*0,16*32,7*0,8*32,4*0,32,7*0,8*32,4*0,32,5*0,4*32,6*0,6*32,3*0,
     &12,2*42,2*11,9*42,2*45,31,2*45,2*33,31,2*45,20*46,7*0,34*42,86*0,
     &2*25,26,24*42,142*0,25,26,0,10*42,19*0,2*13,3*85,0,2,4*0,2,8*0,
     &2*32,87,88,3*3,0,2*3,0,2*3,0,3,5*0,3,1,0,3,2*0,2*3,3*0,1,4*0,12,
     &3*0,4*32,2*4,2*45,6*0,5*32,2*4,87,88,30*0,12,32,0,32,87,88,41*0,
     &12,0,32,0,32,87,88,40*0,12,0,32,0,32,87,88,88*0,12,0,32,0,32,87,
     &88,2*0,4*42,8*0,14*42,50*0,10*13,2*84,3*85,14*0,84,5*0,85,903*0/
      data (brat(i)  ,i=   1, 501)/70*0.,1.,6*0.,2*0.177,0.108,0.225,
     &0.003,0.06,0.02,0.025,0.013,2*0.004,0.007,0.014,2*0.002,2*0.001,
     &0.054,0.014,0.016,0.005,2*0.012,5*0.006,0.002,2*0.001,5*0.002,
     &6*0.,1.,27*0.,0.143,0.111,0.143,0.111,0.143,0.085,2*0.,0.03,
     &0.058,0.03,0.058,0.03,0.058,2*0.,0.25,0.01,2*0.,0.01,0.25,4*0.,
     &0.24,5*0.,3*0.08,3*0.,0.01,0.08,0.82,5*0.,0.09,6*0.,0.143,0.111,
     &0.143,0.111,0.143,0.085,2*0.,0.03,0.058,0.03,0.058,0.03,0.058,
     &8*0.,0.25,0.01,2*0.,0.01,0.25,4*0.,0.24,5*0.,3*0.08,6*0.,0.01,
     &0.08,0.82,5*0.,0.09,11*0.,0.01,0.08,0.82,5*0.,0.09,9*0.,1.,6*0.,
     &1.,4*0.215,2*0.,2*0.07,0.,1.,2*0.08,0.76,0.08,2*0.112,0.05,0.476,
     &0.08,0.14,0.01,0.015,0.005,1.,3*0.,1.,3*0.,1.,0.,0.25,0.01,2*0.,
     &0.01,0.25,4*0.,0.24,5*0.,3*0.08,0.,1.,2*0.5,0.635,0.212,0.056,
     &0.017,0.048,0.032,0.035,0.03,2*0.015,0.044,2*0.022,9*0.001,0.035,
     &0.03,2*0.015,0.044,2*0.022,9*0.001,0.028,0.017,0.066,0.02,0.008,
     &2*0.006,0.003,0.001,2*0.002,0.003,0.001,2*0.002,0.005,0.002,
     &0.005,0.006,0.004,0.012,2*0.005,0.008,2*0.005,0.037,0.004,0.067,
     &2*0.01,2*0.001,3*0.002,0.003,8*0.002,0.005,4*0.004,0.015,0.005,
     &0.027,2*0.005,0.007,0.014,0.007,0.01,0.008,0.012,0.015,11*0.002,
     &3*0.004,0.002,0.004,6*0.002,2*0.004,0.005,0.011,0.005,0.015,0.02,
     &2*0.01,3*0.004,5*0.002,0.015,0.02,2*0.01,3*0.004,5*0.002,0.038/
      data (brat(i)  ,i= 502, 841)/0.048,0.082,0.06,0.028,0.021,
     &2*0.005,2*0.002,0.005,0.018,0.005,0.01,0.008,0.005,3*0.004,0.001,
     &3*0.003,0.001,2*0.002,0.003,2*0.002,2*0.001,0.002,0.001,0.002,
     &0.001,0.005,4*0.003,0.001,2*0.002,0.003,2*0.001,0.013,0.03,0.058,
     &0.055,3*0.003,2*0.01,0.007,0.019,4*0.005,0.015,3*0.005,8*0.002,
     &3*0.001,0.002,2*0.001,0.003,16*0.001,0.019,2*0.003,0.002,0.005,
     &0.004,0.008,0.003,0.006,0.003,0.01,5*0.002,2*0.001,2*0.002,
     &11*0.001,0.002,14*0.001,0.018,0.005,0.01,2*0.015,0.017,4*0.015,
     &0.017,3*0.015,0.025,0.08,2*0.025,0.04,0.001,2*0.005,0.02,0.04,
     &2*0.06,0.04,0.01,4*0.005,0.25,0.115,3*1.,0.988,0.012,0.389,0.319,
     &0.237,0.049,0.005,0.001,0.441,0.205,0.301,0.03,0.022,0.001,6*1.,
     &0.665,0.333,0.002,0.666,0.333,0.001,0.49,0.34,0.17,0.52,0.48,
     &5*1.,0.893,0.08,0.017,2*0.005,0.495,0.343,3*0.043,0.019,0.013,
     &0.001,2*0.069,0.862,3*0.027,0.015,0.045,0.015,0.045,0.77,0.029,
     &1.,14*0.,3*1.,0.28,0.14,0.313,0.157,0.11,0.28,0.14,0.313,0.157,
     &0.11,0.667,0.333,0.667,0.333,1.,0.667,0.333,0.667,0.333,2*0.5,1.,
     &0.333,0.334,0.333,4*0.25,2*1.,0.3,0.7,2*1.,0.8,2*0.1,0.667,0.333,
     &0.667,0.333,0.6,0.3,0.067,0.033,0.6,0.3,0.067,0.033,2*0.5,0.6,
     &0.3,0.067,0.033,0.6,0.3,0.067,0.033,2*0.4,2*0.1,0.8,2*0.1,0.52,
     &0.26,2*0.11,0.62,0.31,2*0.035,0.007,0.993,0.02,0.98,0.3,0.7,2*1./
      data (brat(i)  ,i= 842,1136)/2*0.5,0.667,0.333,0.667,0.333,0.667,
     &0.333,0.667,0.333,2*0.35,0.3,0.667,0.333,0.667,0.333,2*0.35,0.3,
     &2*0.5,3*0.14,0.1,0.05,4*0.08,0.028,0.027,0.028,0.027,4*0.25,
     &0.273,0.727,0.35,0.65,0.3,0.7,2*1.,2*0.35,0.144,0.105,0.048,
     &0.003,0.332,0.166,0.168,0.084,0.086,0.043,0.059,2*0.029,2*0.002,
     &0.332,0.166,0.168,0.084,0.086,0.043,0.059,2*0.029,2*0.002,0.3,
     &0.15,0.16,0.08,0.13,0.06,0.08,0.04,0.3,0.15,0.16,0.08,0.13,0.06,
     &0.08,0.04,2*0.4,0.1,2*0.05,0.3,0.15,0.16,0.08,0.13,0.06,0.08,
     &0.04,0.3,0.15,0.16,0.08,0.13,0.06,0.08,0.04,2*0.4,0.1,2*0.05,
     &2*0.35,0.144,0.105,2*0.024,0.003,0.573,0.287,0.063,0.028,2*0.021,
     &0.004,0.003,2*0.5,0.15,0.85,0.22,0.78,0.3,0.7,2*1.,0.217,0.124,
     &2*0.193,2*0.135,0.002,0.001,0.686,0.314,0.641,0.357,2*0.001,
     &0.018,2*0.005,0.003,0.002,2*0.006,0.018,2*0.005,0.003,0.002,
     &2*0.006,0.005,0.025,0.015,0.006,2*0.005,0.004,0.005,5*0.004,
     &2*0.002,2*0.004,0.003,0.002,2*0.003,3*0.002,2*0.001,0.002,
     &2*0.001,2*0.002,5*0.001,4*0.003,2*0.005,2*0.002,2*0.001,2*0.002,
     &2*0.001,0.255,0.057,2*0.035,0.15,2*0.075,0.03,2*0.015,5*1.,0.999,
     &0.001,1.,0.516,0.483,0.001,1.,0.995,0.005,13*1.,0.331,0.663,
     &0.006,0.663,0.331,0.006,1.,0.88,2*0.06,0.88,2*0.06,0.88,2*0.06,
     &0.667,2*0.333,0.667,0.676,0.234,0.085,0.005,3*1.,4*0.5,7*1./
      data (brat(i)  ,i=1137,2000)/864*0./
      data (kfdp(i,1),i=   1, 530)/21,22,23,4*-24,25,21,22,23,4*24,25,
     &21,22,23,4*-24,25,21,22,23,4*24,25,21,22,23,4*-24,25,21,22,23,
     &4*24,25,21,22,23,4*-24,25,21,22,23,4*24,25,22,23,-24,25,23,24,
     &-12,22,23,-24,25,23,24,-12,-14,34*16,22,23,-24,25,23,24,-89,22,
     &23,-24,25,23,24,1,2,3,4,5,6,7,8,21,1,2,3,4,5,6,7,8,11,13,15,17,1,
     &2,3,4,5,6,7,8,11,12,13,14,15,16,17,18,4*-1,4*-3,4*-5,4*-7,-11,
     &-13,-15,-17,1,2,3,4,5,6,7,8,11,13,15,17,21,2*22,23,24,1,2,3,4,5,
     &6,7,8,11,12,13,14,15,16,17,18,24,37,2*23,25,35,4*-1,4*-3,4*-5,
     &4*-7,-11,-13,-15,-17,3*24,1,2,3,4,5,6,7,8,11,13,15,17,21,2*22,23,
     &24,23,25,36,1,2,3,4,5,6,7,8,11,13,15,17,21,2*22,23,24,23,-1,-3,
     &-5,-7,-11,-13,-15,-17,24,2,1,2,3,4,5,6,11,13,15,82,-11,-13,2*2,
     &-12,-14,-16,2*-2,2*-4,-2,-4,2*89,37,2*-89,2*5,-37,2*89,4*-1,4*-3,
     &4*-5,4*-7,-11,-13,-15,-17,-13,130,310,-13,3*211,12,14,16*-11,
     &16*-13,-311,-313,-311,-313,-311,-313,-311,-313,2*111,2*221,2*331,
     &2*113,2*223,2*333,-311,-313,2*-311,-313,3*-311,-321,-323,-321,
     &2*211,2*213,-213,113,3*213,3*211,2*213,2*-311,-313,-321,2*-311,
     &-313,-311,-313,4*-311,-321,-323,2*-321,3*211,213,2*211,213,5*211,
     &213,4*211,3*213,211,213,321,311,3,2*2,12*-11,12*-13,-321,-323,
     &-321,-323,-311,-313,-311,-313,-311,-313,-311,-313,-311,-313,-311,
     &-321,-323,-321,-323,211,213,211,213,111,221,331,113,223,333,221/
      data (kfdp(i,1),i= 531, 906)/331,113,223,113,223,113,223,333,223,
     &333,321,323,321,323,311,313,-321,-323,3*-321,-323,2*-321,-323,
     &-321,-311,-313,3*-311,-313,2*-311,-313,-321,-323,3*-321,-323,
     &2*-321,-311,2*333,211,213,2*211,2*213,4*211,10*111,-321,-323,
     &5*-321,-323,2*-321,-311,-313,4*-311,-313,4*-311,-321,-323,2*-321,
     &-323,-321,-313,-311,-313,-311,211,213,2*211,213,4*211,111,221,
     &113,223,113,223,2*3,-15,5*-11,5*-13,221,331,333,221,331,333,211,
     &213,211,213,321,323,321,323,2212,221,331,333,221,2*2,3*0,3*22,
     &111,211,2*22,2*211,111,3*22,111,3*21,2*0,211,321,3*311,2*321,421,
     &2*411,2*421,431,511,521,531,2*211,22,211,2*111,321,130,-213,113,
     &213,211,22,111,11,13,82,11,13,15,1,2,3,4,21,22,2*89,11,12,13,14,
     &15,16,1,2,3,4,5,21,22,2*0,223,321,311,323,313,2*311,321,313,323,
     &321,421,2*411,421,433,521,2*511,521,523,513,223,213,113,-213,313,
     &-313,323,-323,82,21,663,21,2*0,221,213,113,321,2*311,321,421,411,
     &423,413,411,421,413,423,431,433,521,511,523,513,511,521,513,523,
     &521,511,531,533,221,213,-213,211,111,321,130,211,111,321,130,443,
     &82,553,21,663,21,2*0,113,213,323,2*313,323,423,2*413,423,421,411,
     &433,523,2*513,523,521,511,533,213,-213,10211,10111,-10211,2*221,
     &213,2*113,-213,2*321,2*311,313,-313,323,-323,443,82,553,21,663,
     &21,2*0,213,113,221,223,321,211,321,311,323,313,323,313,321,5*311/
      data (kfdp(i,1),i= 907,2000)/321,313,323,313,323,311,4*321,421,
     &411,423,413,423,413,421,2*411,421,413,423,413,423,411,2*421,411,
     &433,2*431,521,511,523,513,523,513,521,2*511,521,513,523,513,523,
     &511,2*521,511,533,2*531,213,-213,221,223,321,130,111,211,111,
     &2*211,321,130,221,111,321,130,443,82,553,21,663,21,2*0,111,211,
     &-12,12,-14,14,211,111,211,111,2212,2*2112,-12,7*-11,7*-13,2*2224,
     &2*2212,2*2214,2*3122,2*3212,2*3214,5*3222,4*3224,2*3322,3324,
     &2*2224,5*2212,5*2214,2*2112,2*2114,2*3122,2*3212,2*3214,2*3222,
     &2*3224,4*2,3,2*2,1,2*2,5*0,2112,-12,3122,2212,2112,2212,3*3122,
     &3*4122,4132,4232,0,3*5122,5132,5232,0,2112,2212,2*2112,2212,2112,
     &2*2212,3122,3212,3112,3122,3222,3112,3122,3222,3212,3322,3312,
     &3322,3312,3122,3322,3312,-12,3*4122,2*4132,2*4232,4332,3*5122,
     &5132,5232,5332,864*0/
      data (kfdp(i,2),i=   1, 467)/3*1,2,4,6,8,1,3*2,1,3,5,7,2,3*3,2,4,
     &6,8,3,3*4,1,3,5,7,4,3*5,2,4,6,8,5,3*6,1,3,5,7,6,3*7,2,4,6,8,7,
     &3*8,1,3,5,7,8,2*11,12,11,12,2*11,2*13,14,13,14,13,11,13,-211,
     &-213,-211,-213,-211,-213,3*-211,-321,-323,-321,-323,2*-321,
     &4*-211,-213,-211,-213,-211,-213,-211,-213,-211,-213,6*-211,2*15,
     &16,15,16,15,18,2*17,18,17,18,17,-1,-2,-3,-4,-5,-6,-7,-8,21,-1,-2,
     &-3,-4,-5,-6,-7,-8,-11,-13,-15,-17,-1,-2,-3,-4,-5,-6,-7,-8,-11,
     &-12,-13,-14,-15,-16,-17,-18,2,4,6,8,2,4,6,8,2,4,6,8,2,4,6,8,12,
     &14,16,18,-1,-2,-3,-4,-5,-6,-7,-8,-11,-13,-15,-17,21,22,2*23,-24,
     &-1,-2,-3,-4,-5,-6,-7,-8,-11,-12,-13,-14,-15,-16,-17,-18,-24,-37,
     &22,25,2*36,2,4,6,8,2,4,6,8,2,4,6,8,2,4,6,8,12,14,16,18,23,22,25,
     &-1,-2,-3,-4,-5,-6,-7,-8,-11,-13,-15,-17,21,22,2*23,-24,2*25,36,
     &-1,-2,-3,-4,-5,-6,-7,-8,-11,-13,-15,-17,21,22,2*23,-24,25,2,4,6,
     &8,12,14,16,18,25,11,-3,-4,-5,-6,-7,-8,-13,-15,-17,-82,12,14,-1,
     &-3,11,13,15,1,4,3,4,1,3,5,3,5,6,4,21,22,4,7,5,2,4,6,8,2,4,6,8,2,
     &4,6,8,2,4,6,8,12,14,16,18,14,2*0,14,111,211,111,-11,-13,16*12,
     &16*14,2*211,2*213,2*321,2*323,211,213,211,213,211,213,211,213,
     &211,213,211,213,2*211,213,7*211,213,211,111,211,111,2*211,-213,
     &213,2*113,223,113,223,221,321,2*311,321,313,4*211,213,113,213,
     &-213,2*211,213,113,111,221,331,111,113,223,4*113,223,6*211,213/
      data (kfdp(i,2),i= 468, 873)/4*211,-321,-311,3*-1,12*12,12*14,
     &2*211,2*213,2*111,2*221,2*331,2*113,2*223,333,2*321,2*323,2*-211,
     &2*-213,6*111,4*221,2*331,3*113,2*223,2*-211,2*-213,113,111,2*211,
     &213,6*211,321,2*211,213,211,2*111,113,2*223,2*321,323,321,2*311,
     &313,2*311,111,211,2*-211,-213,-211,-213,-211,-213,3*-211,5*111,
     &2*113,223,113,223,2*211,213,5*211,213,3*211,213,2*211,2*111,221,
     &113,223,3*321,323,2*321,323,311,313,311,313,3*211,2*-211,-213,
     &3*-211,4*111,2*113,2*-1,16,5*12,5*14,3*211,3*213,2*111,2*113,
     &2*-311,2*-313,-2112,3*321,323,2*-1,3*0,22,11,22,111,-211,211,11,
     &2*-211,111,113,223,22,111,3*21,2*0,111,-211,111,22,211,111,22,
     &211,111,22,111,5*22,2*-211,111,-211,2*111,-321,310,211,111,
     &2*-211,221,22,-11,-13,-82,-11,-13,-15,-1,-2,-3,-4,2*21,5,3,-11,
     &-12,-13,-14,-15,-16,-1,-2,-3,-4,-5,2*21,2*0,211,-213,113,-211,
     &111,223,211,111,211,111,223,211,111,-211,2*111,-211,111,211,111,
     &-321,-311,111,-211,111,211,-311,311,-321,321,-82,21,22,21,2*0,
     &211,111,211,-211,111,211,111,211,111,211,111,-211,111,-211,3*111,
     &-211,111,-211,111,211,111,211,111,-321,-311,3*111,-211,211,-211,
     &111,-321,310,-211,111,-321,310,22,-82,22,21,22,21,2*0,211,111,
     &-211,111,211,111,211,111,-211,111,321,311,111,-211,111,211,111,
     &-321,-311,111,-211,211,-211,111,2*211,111,-211,211,111,211,-321/
      data (kfdp(i,2),i= 874,2000)/2*-311,-321,-311,311,-321,321,22,
     &-82,22,21,22,21,2*0,111,3*211,-311,22,-211,111,-211,111,-211,211,
     &-213,113,223,221,22,211,111,211,111,2*211,213,113,223,221,22,211,
     &111,211,111,4*211,-211,111,-211,111,-211,211,-211,211,321,311,
     &2*111,211,-211,111,-211,111,-211,211,-211,2*211,111,211,111,
     &4*211,-321,-311,2*111,211,-211,211,111,211,-321,310,22,-211,111,
     &2*-211,-321,310,221,111,-321,310,22,-82,22,21,22,21,2*0,111,-211,
     &11,-11,13,-13,-211,111,-211,111,-211,111,22,11,7*12,7*14,-321,
     &-323,-311,-313,-311,-313,211,213,211,213,211,213,111,221,331,113,
     &223,111,221,113,223,321,323,321,-211,-213,111,221,331,113,223,
     &111,221,331,113,223,211,213,211,213,321,323,321,323,321,323,311,
     &313,311,313,2*-1,-3,-1,2203,3201,3203,2203,2101,2103,5*0,-211,11,
     &22,111,211,22,-211,111,22,-211,111,211,2*22,0,-211,111,211,2*22,
     &0,2*-211,111,22,111,211,22,211,2*-211,2*111,-211,2*211,111,211,
     &-211,2*111,211,-321,-211,111,11,-211,111,211,111,22,111,2*22,
     &-211,111,211,3*22,864*0/
      data (kfdp(i,3),i=   1, 989)/70*0,14,6*0,2*16,2*0,5*111,310,130,
     &2*0,2*111,310,130,113,211,223,221,2*113,2*211,2*223,2*221,2*113,
     &221,113,2*213,-213,190*0,4*3,4*4,1,4,3,2*2,10*81,25*0,-211,3*111,
     &-311,-313,-311,2*-321,2*-311,111,221,331,113,223,211,111,211,111,
     &-311,-313,-311,2*-321,2*-311,111,221,331,113,223,211,111,211,111,
     &20*0,3*111,2*221,331,113,223,3*211,-211,111,-211,111,211,111,211,
     &-211,111,113,111,223,2*111,-311,4*211,2*111,2*211,111,7*211,
     &7*111,113,221,2*223,2*-211,-213,4*-211,-213,-211,-213,-211,2*211,
     &2,2*0,-321,-323,-311,-321,-311,2*-321,-211,-213,2*-211,211,-321,
     &-323,-311,-321,-311,2*-321,-211,-213,2*-211,211,46*0,3*111,113,
     &2*221,331,2*223,-311,3*-211,-213,8*111,113,3*211,213,2*111,-211,
     &3*111,113,111,2*113,221,331,223,111,221,331,113,223,113,2*223,
     &2*221,3*111,221,113,223,4*211,3*-211,-213,-211,5*111,-321,3*211,
     &3*111,2*211,2*111,2*-211,-213,3*111,221,113,223,6*111,3*0,221,
     &331,333,321,311,221,331,333,321,311,19*0,3,5*0,-11,0,2*111,-211,
     &-11,11,2*221,3*0,111,22*0,111,2*0,22,111,5*0,111,12*0,2*21,2*-6,
     &11*0,2*21,111*0,-211,2*111,-211,3*111,-211,111,211,15*0,111,6*0,
     &111,-211,9*0,111,-211,9*0,111,-211,111,-211,4*0,111,-211,111,
     &-211,4*0,-211,4*0,111,-211,111,-211,4*0,111,-211,111,-211,4*0,
     &-211,3*0,-211,5*0,111,211,3*0,111,10*0,2*111,211,-211,211,-211/
      data (kfdp(i,3),i= 990,2000)/7*0,2212,3122,3212,3214,2112,2114,
     &2212,2112,3122,3212,3214,2112,2114,2212,2112,50*0,3*3,1,12*0,
     &2112,43*0,3322,878*0/
      data (kfdp(i,4),i=   1,2000)/83*0,3*111,9*0,-211,3*0,111,2*-211,
     &0,111,0,2*111,113,221,111,-213,-211,211,190*0,13*81,41*0,111,
     &3*211,111,5*0,-211,111,-211,111,2*0,111,3*211,111,5*0,-211,111,
     &-211,111,50*0,2*111,2*-211,2*111,-211,211,3*111,211,14*111,221,
     &113,223,2*111,2*113,223,2*111,-1,4*0,-211,111,-211,211,111,2*0,
     &2*111,-211,2*0,-211,111,-211,211,111,2*0,2*111,-211,96*0,6*111,
     &3*-211,-213,4*111,113,6*111,3*-211,3*111,2*-211,2*111,3*-211,
     &12*111,6*0,-321,-311,3*0,-321,-311,19*0,-3,11*0,-11,280*0,111,
     &-211,3*0,111,29*0,-211,111,5*0,-211,111,50*0,2101,2103,2*2101,
     &935*0/
      data (kfdp(i,5),i=   1,2000)/85*0,111,15*0,111,7*0,111,0,2*111,
     &246*0,111,-211,111,7*0,2*111,4*0,111,-211,111,7*0,2*111,93*0,111,
     &-211,111,3*0,111,-211,4*0,111,-211,111,3*0,111,-211,1500*0/

c...ludat4, with character strings.
      data (chaf(i)  ,i=   1, 325)/'d','u','s','c','b','t','l','h',
     &2*' ','e','nu_e','mu','nu_mu','tau','nu_tau','chi','nu_chi',
     &2*' ','g','gamma','z','w','h',6*' ','z''','z"','w''','h''','a',
     &'h',' ','lq_ue','r',40*' ','specflav','rndmflav','phasespa',
     &'c-hadron','b-hadron','t-hadron','l-hadron','h-hadron','wvirt',
     &'diquark','cluster','string','indep.','cmshower','spheaxis',
     &'thruaxis','clusjet','celljet','table',' ','pi',2*'k',2*'d',
     &'d_s',2*'b','b_s',' ','pi','eta','eta''','eta_c','eta_b','eta_t',
     &'eta_l','eta_h',2*' ','rho',2*'k*',2*'d*','d*_s',2*'b*','b*_s',
     &' ','rho','omega','phi','j/psi','upsilon','theta','theta_l',
     &'theta_h',2*' ','b_1',2*'k_1',2*'d_1','d_1s',2*'b_1','b_1s',' ',
     &'b_1','h_1','h''_1','h_1c','h_1b','h_1t','h_1l','h_1h',2*' ',
     &'a_0',2*'k*_0',2*'d*_0','d*_0s',2*'b*_0','b*_0s',' ','a_0','f_0',
     &'f''_0','chi_0c','chi_0b','chi_0t','chi_0l','chi_0h',2*' ','a_1',
     &2*'k*_1',2*'d*_1','d*_1s',2*'b*_1','b*_1s',' ','a_1','f_1',
     &'f''_1','chi_1c','chi_1b','chi_1t','chi_1l','chi_1h',2*' ','a_2',
     &2*'k*_2',2*'d*_2','d*_2s',2*'b*_2','b*_2s',' ','a_2','f_2',
     &'f''_2','chi_2c','chi_2b','chi_2t','chi_2l','chi_2h',2*' ','k_l',
     &'k_s',58*' ','pi_diffr','n_diffr','p_diffr',22*' ','lambda',
     &5*' ','lambda_c',' ',2*'xi_c',6*' ','lambda_b',' ',2*'xi_b'/
      data (chaf(i)  ,i= 326, 500)/6*' ','n','p',' ',3*'sigma',2*'xi',
     &' ',3*'sigma_c',2*'xi''_c','omega_c',4*' ',3*'sigma_b',
     &2*'xi''_b','omega_b',4*' ',4*'delta',3*'sigma*',2*'xi*','omega',
     &3*'sigma*_c',2*'xi*_c','omega*_c',4*' ',3*'sigma*_b',2*'xi*_b',
     &'omega*_b',114*' '/

c...ludatr, with initial values for the random number generator.
      data mrlu/19780503,0,0,97,33,0/

      end
