!!  This program is linked to Epics/Module/epDirecPairX.f90.
!!  The compilation should be done in the Moudle diectory (even
!!  if this is modified).   

module modDirecPair
  ! Nucleus charge Z1 projectile and charge Z2 target collide
  ! electromagnetically and generate  electron pair  (e+e-).
  ! This gives,
  !   epDirecPairDxs:
  !      Differential cross-section: ds/dE (E is e+ or e- or sum of both)
  !      Its integral 
  !   Double diff. xs ( ds/(E+E-) ; E+ for e+ E- for e-)
  !   Total cross-section 
  !   Restricted toral cross-section (energy of e+ or e- is greater than
  !                a certain value, Et)
  !   Randomly sampled E+ or E- or (E+ + E-)  > Et
  !
  !  Z1 is fully ionized.  Z2 may be  also  fully ionized but may also be
  !  not ionzied at all. (completely screened).  (sc=0: bare target
  !  sc=2; for screeened target).
  !  Many of formulas are based on a paper by
  ! A)  C.Bertulani and G. Baur, Electromagnetic processes in relativistic
  !   heavy ion collisions, Phys. Rep. 163, (1988) 299-408.
  !  We prepare some formulas by 
  ! B)  Bhabha, Proceedings of the Royal Society of
  !   London. A. Mathematical and Physical Sciences,
  !   15 November 1935 Volume 152 Issue 877.
  !   (used when sc=0 and bhabha=1)
  !   call epDirecPairB(1) must have been called.
  !   To reset it call this with 0.
  implicit none
  real(8),parameter:: alfa = 1.d0/137.036d0
  real(8),parameter:: pi =  asin(1.d0)*2
  real(8),parameter:: Re=2.81794d-15  ! m
  real(8),parameter:: alfare2=(alfa*Re)**2/1.0d-31  ! mb
  real(8),parameter:: delta = 0.681 ! const related to Euler's  const
  real(8),parameter:: Const2 = 28.d0/9.0d0/pi
  real(8),parameter:: Const0 = 28.d0/27.0d0/pi
  real(8),parameter:: Constx = 56.0d0/9.0d0/pi

  integer:: bhabha = 0  !make this 1 for speciall case of diff. xsection to
         !  get ds/dw where w = Ep+Em. at g>>1 and sc =0 
         !    
  real(8),parameter:: oneby42= 1.d0/42.d0
  real(8),parameter:: logHdelta2 = (log(delta/2))**2
  real(8),parameter:: Me = 0.511d-3  ! elec: mass
  !! to control DiretPair routines.  these can be changed as input parameters    !!!!!!!!!!!!!!!!!
  logical:: DirecPair = .true.
  logical:: BareTarget = .false.
  real(8):: AdjFac = 1.0d0 ! standard thereshhold (1%) is adjusted by this
  ! factor.   0.5 ~ 5 may be usable.

  logical:: NormDpXs = .true.  ! XS by BB paper seems to be too small
  ! as compared to Decker, and C. Bottcher and M. R. Strayer．
  ! So if this is .true., every cross-section by BB is multiplied
  ! by  epDirecPadjf (>1). 
  
  !            these are rather at the test time
  real(8):: EemaxGT = 1.5d0  ! At very low energy, Eemax < Eemin.
                 !  we keep at least region Eemax > EemaxGT*Eemina
  real(8):: Tune0 = 1.0d0   ! original xs * Tune0 is the xs to be used.
          ! for screening =2
  real(8):: Tune2 = 1.0d0 ! smae for screening 2

  integer:: HowTargetZ = 3  ! for compound target media, how to treat it as a
      ! signle atom of charge  : 
      !  1:  media%Z, media%Z2, media%Z1_3rd will be used as Z, Z^2, Z^(1/3) 
      !  2:  meda%Zeff, media%Z2eff, media%Zeff3 will be used as Z, Z^2, Z^(1/3)  This is NG
      !  3:  treated as if hadronic collisions: weighted with the xs, and  randomely
      !      choosen the target atom.
  logical:: OriginalTotalXS = .false.
  logical:: UseNewCoulombC = .false.
  integer:: SCTotalXS = 4
!  SCTotalXS is used to check the wrong expession of Eq.7.4.3
!
!          total x-section for screened target case ( which must be used
!          at least > Escreen.  ) 
!         1: use as the paper which is however wrong (incosnsistent with integration of
!            ds/dE).  Also, max energy is wrong: 
!         2: correct max energy is  used. (coicide with the paper grapH)
!         3: Use correct integral but use wrong max Ee as in 1.
!         4: same as above and use correct Ee max This should be used normally.

  integer:: piKDP = 0  ! bit 0 is on, pi DC is includedd
  real(8):: mfpSave, xsSave   ! epDirecPairManager use these two to save
                  !  cross-sec and mfp 
 ! if bit 1 is on, K DC is includedl   (bit 0 is right most bit)
  
  
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !  the energy librated by (e+ + e-) by the direct  pair creation can be
  !  approximated by
  !     dE (GeV/(g/cm2)) / (Z1^2 Z2)=  3e-6 (E0/1000) ^1.12 (GeV/(g/cm2)
  !   where E0 is the  projectile energy in GeV/n,  Z1 the projectile charge
  !   and Z2 that of the target. (Note Z2 is not Z2^2 --since we discuss
  !   the the energy loss in (GeV/(g/cm2)).  We may consider the
  !   direct pair creation effect if it gives 1/10 of the stangdard
  !   2 MeV*Z1^2 /(g/cm2).  Hence the such energy Eth is:
  !     3e-6 (Eth/1000) ^1.12 =    0.2e-3 Z1^2 /(Z1^2*Z2) 
  !   i.e
  !        Eth = 1000(a*2/3 x10^2/Z2)^0.893 (a is AdjFac )
  !   here Eth is GeV/n and depends only on the taget.
  !   The variable for Eth is denoted as EPNmin
  real(8):: EPNmin  ! see above
  real(8):: dEfrac !  = AdjFac * 1.e-2. 
  real(8):: Z2cr  ! Z2**(1/3)
  real(8):: Z1alfa2 ! (Z1*alfa)**2
  real(8):: Z2alfa2 ! (Z2*alfa)**2
  real(8):: g
  integer:: sc
  real(8):: Z1, Z2, Z2sq
  real(8):: Z1Z2alfare2  ! (Z1*Z2*alfa)**2
  real(8):: SqBrConst2 ! [ ] part of  Eq. 7.4.2
  real(8):: Cx  !  ConstX* Z1Z2alfare2
  real(8):: Eswitch
  real(8):: Escreen ! sceening should work at Ee>> Esreen
!                 so we switch to screening cross section
!                 when Ee > Eswitch  (for non bare target)
  real(8):: xsL !  bare target xsection from Emin to Eswitch
  real(8):: xsH !  screened target xsection from Eswitch to Emax
  !                these two will be obtained if Emin < Eswitch < Emax
  !                with screen=2. These two  will be obtained
  !                at total xsection calculation time and will be used
  !                also at  sampling of Ee.
  real(8):: Eemin, Eemax ! min max of permitted energy region (GeV)
  !  real(8):: Emin2
  real(8):: gmin   ! threshold gamma factor below which DP  is neglected
  real(8):: dpmfp  ! mean free path for drect pair creation
                   ! usable after  
end module modDirecPair
subroutine epDirecPairB(bha)
  use modDirecPair
  implicit none
  integer,intent(in):: bha
  bhabha = bha
end subroutine epDirecPairB
subroutine epqDirecPairB(bha)
  ! inquire current bhabha valu
  use modDirecPair
  implicit none
  integer,intent(out):: bha
  bha = bhabha
end subroutine epqDirecPairB

subroutine epDirecPair0(Z1in, Z2in, Z2sqin, Z2crin, gin, screen,icon)
!   initialization which is required every  time when the new set of
!   parameters are used.
  use  modDirecPair
  !    projectile 1 collides with target 2 and makes e++e- pair creation;
  !    this computes such a cross-section;
  implicit none
  !initialize for fixed Z1, Z2, g, sc
  real(8),intent(in):: Z1in, Z2in, Z2sqin, Z2crin
  ! Z1in: projectile Z
  ! Z2in:  <Z> of taarget
  ! Z2saqin: target media <Z^2> of target
  ! Z2cr:    target media <Z^(1/3)> of target
  real(8),intent(in):: gin   !  gamma factor of projectile
             !   Roughly gin>>100 is our target.
  integer,intent(in):: screen ! screening of the target. 0--> bare nucleus
                          !  colider case.
  !      1 --> partially screened ; not usable now
  !      2 --> not ionzized and energy is high; completely screened.
  integer,intent(out)::  icon ! 0 if input seems to be OK
             ! else some thing worng. may be too low g. 
             ! -2:   Eemax <= Eemin
             ! -1:   Eemax < EemaxGT * Eemin 
             !  0:   otherswise.  
  real(8),external:: epCoulombBar

!    dE ~ a*(E/1000)**b;  a and b   rough fit to get rough E threshold
!        (GeV/(g/cm2))
!         (dE include e- + e+ )  dE is compared with 2 /Z2 MeV/(g/cm2)
!       after dividing by Z1^2Z2 (not Z2^2)
!      a*(E/1000.)** b = 2e-3*0.01 /Z2)   ! 1% threshold
!         0.01 may be changed by AdjFac (0.01*AdjFac will be
!        the real percentage.
!      EPNmin = (AdjFac*2e-5/Z2/a)**(1/b) * 1000.
!       1/b = binvB or binvNonB
!       2e-5/a = ainvB or ainvNonB
  real(8),parameter:: aforB=3.0d-6, bforB =1.2d0 ! bare case
  real(8),parameter:: aforNonB=3.0d-6, bforNonB =1.1d0 ! screened case
!      derived param.
  real(8),parameter:: binvB=1.0d0/bforB, binvNonB=1.d0/bforNonB 
  real(8),parameter:: ainvB=2.5d-5/aforB, ainvNonB = 2.5d-5/aforNonB
  real(8):: temp

  
  Z1 = Z1in
  Z2 = Z2in
  Z2sq = Z2sqin
  g = gin
  sc = screen
  Z1alfa2 =(Z1*alfa)**2
  Z2alfa2 =Z2sq *alfa**2
  Z2cr = Z2crin
  Z1Z2alfare2 =  Z1*Z1 * Z2sq *alfare2
  temp = log(183.0d0/Z2cr) 
  SqBrConst2 = temp -  oneby42   - epCoulombBar()
  Cx = ConstX * Z1Z2alfare2
  Escreen = Me/(Z2cr*Alfa)  
  Eswitch = exp(temp +20.d0/42.d0) * Me
  dEfrac = 1.d-2 * AdjFac   !  1% x AdjFac is threshold --> gmin
  
  !  Now   threshold is treated by using gamma factor
! (usable even pi, K...)  >>>>>
!       Z2 < sqrt(Z2sq) should use sqrt
!  EPNmin = 1.0d3 *(AdjFac*0.666666d2/sqrt(Z2sq))**0.893d0
!   1 % effect
!  if(Sc == 0 ) then
!     EPNmin = 1.0d3 *(AdjFac*ainvB/sqrt(Z2sq))**binvB
!  else
!    EPNmin = 1.0d3 *(AdjFac*ainvNonB/sqrt(Z2sq))**binvNonB
!  endif
!
  !<<<<<
  call epDirecPairSetGmin  ! 1%*AdjFac  of 2MeVZ1^2 should be treated

  if ( g < gmin) then
     icon = -3
  else
     call epDirecPairSetEeminmax
     if( Eemax <= Eemin ) then
        icon = -2
     elseif( Eemax < EemaxGT * Eemin ) then
        icon = -1
     else
        icon = 0
     endif
  endif

end subroutine epDirecPair0

subroutine epDirecPairSetGmin
  !  to be called inside  epDirecPair0.
  !  This sets internally gmin (min gamma factor
  !  above which e+ + e- energy exceed some threshold.
  !  The standard threshold is 1% of ionization loss.
  !  The threshold  may be chanbed by input parameter,
  !  AdjFac ( default 1.)
  use modDirecPair
  implicit none

  !   2e-3*Z1^2 (GeV/(g/cm2)) is standard  mean energy
  ! loss rate by ionization  of the current  particle with charge Z1.  
  ! If the energy of e+ + e- by direct pair production cross-section
  ! becomes   dEfrac*100 % level of this loss rate,  we actually
  ! do a sampling of the direct pair creation.  This routine sets
  ! the minim gamma factor for this threshold.
  ! dEfrac = 0.01 means 1 % and is  standard.   The value
  
  if(sc == 2 ) then
     if( NormDpXs ) then
        ! screened target and (BB x-section) * epDirecPadjf() case
        ! a=4.8e-6 b= 1.2 1/b=8.333e-1;  c=1e3  
        gmin =( (2.0d-3*dEfrac/Z2/4.8d-6)**8.333d-1)*1.d3
     else
        ! BB's XS.  no enhancement by epdirecPadjf
        ! a=2.93e-6 b=1.4 1/b=7.142e-1; c=1000.  f(x)
        gmin = ( (2.0d-3*dEfrac/Z2/2.93d-6)**7.142d-1 )*1.d3
     endif
  elseif( sc == 0  ) then
     if( NormDpXs ) then
        ! bare target and  (BB x-section) * epDirecPadjf() case
        ! a = 5.1e-6 b=1.1 1/b=9.091e-1 c=1e3
        gmin = ( (2.0d-3*dEfrac/Z2/5.1d-6)**9.091d-1 )*1.d3
     else
        ! bare target case and no enhancement  f(x)
        ! a= 3.3e-6 b=1.4 1/b=7.1428e-1 c=1e3
        gmin = ( (2.0d-3* dEfrac/Z2/3.3d-6)**7.143d-1) *1.0d3
     endif
  else
     write(0,*)  ' sc=', sc, ' NG in epDirePairSetGmin'
     stop
  endif
end subroutine epDirecPairSetGmin

  function epDirecPairDxs(E) result(ans)
  ! differenctial cross section of e- or e+
  ! epDirecPair0 must have been called if Z1,Z2, g, sc
  !         are diff. from previous ones.
  ! For screened case, the paper formula for ds/dE  must be used
  ! E>> Me/(Z^(1/3)*alafa)=Eb. (In the paper, it is written as
  ! E> Me*(Z^(1/3)*alafa).  This is strange since the value becomes
  ! keV scale. (ds/dE for screened case is < than the bare one at high E,
  ! but becomes > than the  bare one at small E;
  !  If we integrate ds/dE from E=Me to gMe, the total
  ! corss-section of screened case > bare case. So we should regard
  ! ds/dE for screened is applicable at E>> Me).
  !
  !  Ths value, Eb, is close to the value, Esw, where 
  ! ds/dE (bare) =  ds/dE(screened) holds.  (Esw > Eb). So we use
  ! screened ds/dE at E> Esw, and ds/dE (bare) (E<Esw).  Then the
  ! ds/dE becomes continuous at E=Esw.
  ! Esw is denoted as Eswitch (used only if screen=2)
  use modDirecPair  
  implicit none

  real(8),intent(in) :: E ! positron or electron energy in GeV.
  real(8):: ans

  real(8),external:: epDirecPairDxsOrg
  real(8),external:: epCoulombBar

  real(8):: scsave
  real(8),external:: epDirecPadjf
  
  scsave = sc   

  if(sc == 0) then
!        bare target case
     ans = epDirecPairDxsOrg(E) * Tune0
  elseif( sc == 2    ) then
          !  target is completely screened 
     if(E < Eswitch) then
!         in the lower energy region, use bare target case
        sc = 0
     endif
     ans = epDirecPairDxsOrg(E) * Tune2
  else
     write(0,*) ' sc=', sc, 'n.g for epDirecPairDxs'
     stop
  endif
  sc = scsave
  ans = max(ans,0.d0)
  if( NormDpXs ) then
     ans = ans * epDirecPadjf()
  endif
end function epDirecPairDxs
  
function epDirecPairDxsOrg(E) result(ans)
  ! differenctial cross section of e- or e+
  ! epDirecPair0 must have been called if Z1,Z2, g, sc
  !         are diff. from previous ones.
  ! This is those written in the papers.
  use modDirecPair  
  implicit none

  real(8),intent(in) :: E ! positron or electron energy in GeV.
  real(8):: ans

  real(8),external:: epCoulombBar
  if(sc == 0) then
     if( bhabha == 0 ) then
  !       Eq. 7.3.9 of A)
  !     bare target case.
        ans = Cx   &
             * ( log(E/Me) -0.5d0 - epCoulombBar())  * log(g*delta*Me/E/2) / E
     elseif( bhabha == 1 ) then
        !      B) Eq.34. p52
        ! E is w=(Ep + Em)    dxs = ans *dw
        ans = Cx * log(E/Me) * log(g*Me/E)/E
     else
        write(0,*) ' bhabha=', bhabha, ' N.G for epDirecPairDxs'
     endif
  elseif( sc == 2 ) then
     !  target is completely screened Eq. 7.4.2
     ans = Cx  &
          * SqBrConst2 * log(g*delta*Me/E/2) / E
  !  Emin2 = Z2cr*alfa*Me
  else
     write(0,*) ' sc=', sc, 'n.g for epDirecPairDxs0'
     stop
  endif
  ans = max(ans,0.d0)     
end function epDirecPairDxsOrg

function epDirecPairInteDxs(E) result(ans)
  ! indefinite integral ds/dE; 
  use modDirecPair
  implicit none
  real(8),intent(in):: E  !
  real(8):: ans

  real(8):: x, y, C3, C2, C1
  real(8),external:: epCoulombBar
  real(8),external::epDirecPairInteDxsOrg
  integer:: scsave
  real(8),external:: epDirecPadjf
  
  scsave = sc
  if( sc == 0) then
     ans = epDirecPairInteDxsOrg(E) * Tune0
  elseif( sc == 2 ) then
     if(E < Eswitch) then
        sc =0
     endif
     ans = epDirecPairInteDxsOrg(E) * Tune2
  else
     write(0,*) 'sc=', scsave, ' ng for  epDirecPairInteDxs(E)'
     stop
  endif
  if( NormDpXs ) then
     ans = epDirecPadjf()* ans
  endif
  sc  = scsave
end function epDirecPairInteDxs

function epDirecPairInteDxsOrg(E) result(ans)
  ! indefinte integral of  ds/dE; 
  use modDirecPair
  implicit none
  real(8),intent(in):: E  !
  real(8):: ans

  real(8):: x, y, C3, C2, C1
  real(8),external:: epCoulombBar
  real(8),external:: epDirecPadjf

  C2 = log(g*delta/2)
  x = E/Me
  y = log(x)

  if( sc == 2 ) then
     C3 = SqBrConst2 ! = log(183.0d0/Z2cr) -  oneby42   - epCoulombBar()
!     ans = Cx*C3*(-0.5d0*y +C2)*y
!         above one is OK with y =log(E/Me) 
!      if we define y =log (x)  x= 2E/(g*delta*m)
!       then integral becomes simply -CX*C3* y^2/2 and next is 
!       obtained.
     ans =-Cx*C3*(log(2*E/g/delta/Me))**2/2
  elseif( sc == 0) then
     if( bhabha == 0) then
        C1 = 0.5d0 + epCoulombBar()
        ans= Cx * ( (-y/3.d0 + (C1+C2)/2)*y -C1*C2)*y
     elseif( bhabha == 1 ) then
        ans = Cx*(-y/3.d0 + log(g)/2)*y**2
     else
        write(0,*) ' bhabha=', bhabha, ' ng for  epDirecPairInteDxs(E)'
        stop
     endif
  else
     write(0,*) 'sc=', sc, ' ng for  epDirecPairInteDxs(E)'
     stop
  endif
  if( NormDpXs ) then
     ans = epDirecPadjf()*ans
  endif
end function epDirecPairInteDxsOrg


         
function epDirecPairInteDxsAtoB(Ea, Eb)  result(ans)
  ! definite integral of  (ds/dE)dE from E=Ea to Eb.  In the case
  ! of Bhabha, E is the sum of e+ e- energies, 
  use modDirecPair
  implicit none
  real(8),intent(in):: Ea, Eb  ! min and max  energies
  real(8):: ans
  ! int( log(w/Me) * log(g*Me/w)/w dw =
  ! int ( log(x) * log(g/x)/x dx) ; y = log(x)
  !  y1 = log(w1/Me); y2 = log(w2/Me)
  ! int is: log(g)/2 (y2^2-y1^2) - (y2^3-y1^3)/3 
!  real(8):: y1, y2
  real(8),external:: epDirecPairInteDxs
  integer:: scsave

!!  real(8),external:: epDirecPadjf
  
  scsave = sc

  if( Eb <= Ea ) then
 !           better to require Eb > C*Ea (C ~2 at least...)
!       this may happen if very low energy proj. comes in 
!       where theory cannot be applied.  (may happen if the user
!       set very ccmall AdjFac and EPNmin becomes very small
     ans = 0.
     xsL = 0.
     xsH = 0.
     return
  else
     if( Ea == Eemin .and. Eb == Eemax) then
        if( Eb < EemaxGT * Ea  ) then
           ans = 0.
           xsL = 0.
           xsH = 0.
           return
        endif
     endif
  endif

  if( sc == 2 ) then
    if( Ea < Eswitch .and. Eb > Eswitch) then
        sc = 0
        xsL = epDirecPairInteDxs(Eswitch) -  epDirecPairInteDxs(Ea)
        sc = 2
        xsH = epDirecPairInteDxs(Eb) -  epDirecPairInteDxs(Eswitch)
        ans = xsL + xsH
     elseif( Ea > Eswitch) then
        ans = epDirecPairInteDxs(Eb) -  epDirecPairInteDxs(Ea)
        xsL = 0.
        xsH = ans
     else
        sc =0
        ans = epDirecPairInteDxs(Eb) -  epDirecPairInteDxs(Ea)
        xsL = ans
        xsH = 0.
     endif
  else
     ans =   epDirecPairInteDxs(Eb) -  epDirecPairInteDxs(Ea)
  endif
!      not needed  ;  InteDxs finished the job
!  if( NormDpXs ) then
!     ans = ans * epDirecPadjf()
!  endif
  sc = scsave
!   for sc =0 and bhabha=1  case; next above one should be same as below.
!  y1 = log(Ea/Me)
!  y2 = log(Eb/Me)
!  ans = Constx*Z1Z2alfare2*( log(g)/2 * (y2**2-y1**2) - (y2**3-y1**3)/3.d0 )
end function epDirecPairInteDxsAtoB

function  epDirecPairTotalXOrg()  result(ans)  
  ! total dp xsec.
  use modDirecPair
  implicit none
  real(8):: temp
  real(8):: ans
  real(8),external:: epCoulombBar   ! fbar Eq.7.3.7

  if( sc == 0 ) then
     if(bhabha == 0 ) then
!        ! Eq.7.3.11 of A)
        temp =log(g*delta/2) 
        ans = Const0 * Z1Z2alfare2 * ( temp -  &
          1.5*(1.0 + 2*epCoulombBar() ) ) * temp**2
!             1.5*(1.0 + 2*epCoulombBar() ) )* temp**2
     elseif( bhabha ==  1) then
!        !  Eq.35 of B)
        ans = Const0 * Z1Z2alfare2 * (log(g))**2
     else
        write(0,*) 'sc, bhabha=', sc,bhabha, ' NG for epDirecPairTotalX'
        stop
     endif
  elseif( sc == 2 ) then
     if( g <  Escreen / Me ) then
        ans = 0.
     else
        if( SCTotalXS  == 4 ) then
!!         correct F(E) and corect upper limit 
           ans = Const2 * Z1Z2alfare2 *         &
                SqBrConst2*( (log(2./(g*delta*Z2cr*alfa) ))**2 -  0. )
        elseif(SCTotalXS  == 3 ) then
 !        correct  F(E) but  upper limit is wrong as paper
           ans = Const2 * Z1Z2alfare2 *         &
                SqBrConst2*( (log(2./(g*delta*Z2cr*alfa) ))**2 - (log(2./delta))**2 )
        elseif(SCTotalXS  == 1 ) then
     !     ! Eq.7.4.3 p.388;   upper limit is wrong and indefinite integral is wrong.
           ans = Const2 * Z1Z2alfare2 *         &
                SqBrConst2*( (log(g*delta*Z2cr*alfa))**2 - logHdelta2 )
        elseif(SCTotalXS  == 2 ) then
      !         use correct upper limit+  reproduce the fig. but F is wong
           ans = Const2 * Z1Z2alfare2 *         &
                SqBrConst2*( (log(g*delta*Z2cr*alfa))**2 - 0. )
        else
           write(0,*)' SCTotalXS=', SCTotalXS, 'NG: :epDirecPairTotalXOrg'
           stop
        endif
     endif
  else
     write(0,*) 'sc=', sc,  ' NG for epDirecPairTotalX'
     stop
  endif
end function epDirecPairTotalXOrg

function epDirecPairTotalX() result(ans)
  ! total xs of the diect pair electron production; itegration in the
  ! whole permitted rgion
  ! *** This is not the same as epDirecPairTotalXOrg(). 
  ! but uses analytically integrated formulat of original differential
  ! formulta (which diff. from those given in the paper, i.e, those in
  ! epDirecPairTotalXOrg(). 
  use modDirecPair
  implicit none
  real(8):: ans  ! in mb.  Normally kb order will be 

  real(8),external::epCoulombBar   ! fbar Eq.7.3.7
  !     diff between  using epCoulombBar(Z1, Z2) and
  !                   epCoulombC( (Z2*alfa)**2 ) is small.
  real(8),external:: epDirecPairTotalXOrg
!!  real(8),external:: epDirecPadjf
  
  real(8):: temp

  real(8),external:: epDirecPairInteDxsAtoB

  if( OriginalTotalXS ) then
     ans = epDirecPairTotalXOrg()
  else
     ans = epDirecPairInteDxsAtoB(Eemin, Eemax)
!!     if( NormDpXs ) then
!!        ans = epDirecPadjf()*ans
!!     endif
  endif

end function epDirecPairTotalX



function epCoulombBar() result(ans)

  use modDirecPair  
  implicit none
  real(8):: ans   
  real(8),external:: epCoulombC
  if( UseNewCoulombC ) then
  !  Eq. 7.3.7.  
     ans =( Z1*epCoulombC( Z1alfa2 ) + Z2*epCoulombC( (Z2alfa2 ) )) /(Z1+Z2)
  else
     ans = epCoulombC( Z2alfa2 )
  endif
end function epCoulombBar

  

  
subroutine epDirecPairSmpEsum(w)
  !     sammples a random variables for the sum of pair electrons
  ! produced by direct pair production by heavy ions.
  ! 
  ! For e+ e- direct pair creation, the sum of their energy follows
  ! this distribution: Eq 34 of BhaBha's paper
  !  Acutal distribution is
  !      Constx * Z1Z2alfare2 * log(E/Me) * log(g*Me/E)/E dE
  !  i.e,  log(E/Me) * log(g*(Me/E))/(E/Me) d(E/Me)
  !        put x= E/Me  --> f(x)= log(x)*log(const/x)/x dx
  !   const = g ; xmin = Eemin/Me, xmax= Eemax/Me
  !   In this case, Eemin, Eemax mean  energy sum of e+ and e-. 
  use modDirecPair
  implicit none
!  real(8),intent(in):: w1, w2  ! min and max  pair energy sum in GeV
  real(8),intent(out):: w  ! samled pair sum energy in GeV 
  !     before this is called,     epDirecPair0 must have  been called
  real(8):: x1, x2, const, x
  const = g
  x1 = Eemin/Me
  x2 = Eemax/Me
  call epDirecPairSmpEsum0(const, x1, x2, x)
  w = x*Me
end subroutine epDirecPairSmpEsum


subroutine epDirecPairSmpE(E)
  !  samples energy (E) of e+ or e- of direct pair creation.
  !  Before calling this, epDirecPair0 must be called if
  !  the parameter values are diff. from the previous call
  ! screening 2 case   epDirecPairInteDxsAtoB(Ea, Eb) must have been used
  use modDirecPair
  implicit none
!     implicit parameters:
!  real(8),intent(in):: Eemin, Eemax ! energy region  for sampling GeV (total E)
                           ! for Bhabha pair sum, energy is sum of e+ e- 
  real(8),intent(out):: E

  real(8),external:: epCoulombBar
  real(8):: x
  real(8):: C3, C2, C1, y, ymin, ymax, u, a, b, c
  real(8):: xmin, xmax, D
  real(8):: cf(0:3)
  complex(kind(0.d0)):: ry(3)
  integer:: ns, nr, i
  logical:: ok
  integer:: scsave

  scsave = sc 

  if(sc == 2 ) then
     if(Eemin < Eswitch .and. Eemax > Eswitch) then
        !Next two  should have been given already
        !  xsL = epDirecPairInteDxs(Eswitch) - epDirecPairInteDxs(Ea)
        !  xsH = epDirecPairInteDxs(Eb) - epDirecPairInteDxs(Eswitch)
        call rndc(u)
        if( u < xsL/(xsL + xsH)) then
           sc = 0
           xmin = Eemin/Me
           xmax = Eswitch/Me
        else
           xmin = Eswitch/Me
           xmax = Eemax/Me
        endif
     else
        xmin = Eemin/Me
        xmax = Eemax/Me
        if( Eemin >= Eswitch ) then
           sc = 2
        else
           sc = 0
        endif
     endif
  else
     xmin = Eemin/Me
     xmax = Eemax/Me
  endif
  ymin = log(xmin)
  ymax = log(xmax) !

  if(sc == 0 ) then
     if(bhabha == 0 ) then
        C1 = 0.5d0 + epCoulombBar()
        C2 = log(g*delta/2)
        call rndc(u)
        cf(3)  = 1./3.d0
        cf(2)  = - (C1 + C2)/2
        cf(1) =  C1*C2
!        cf(0) =( ( ( (-ymax/3.d0 +(C1+C2)/2) * ymax)  -C1*C2 )*ymax  &
!             -   ( ( (-ymin/3.d0 +(C1+C2)/2) * ymin)  -C1*C2 )*ymin )* u &
        !     +   ( ( (-ymin/3.d0 +(C1+C2)/2) * ymin)  -C1*C2 )*ymin
        ! above is 
        cf(0) =  ( ( (-ymax/3.d0 +(C1+C2)/2) * ymax)  -C1*C2 )*ymax * u &
            + ( ( (-ymin/3.d0 +(C1+C2)/2) * ymin)  -C1*C2 )*ymin *(1-u)
        
        call kcubicEq(cf, ry, nr, ns)
        if( nr == 1 ) then
           y = real(ry(1))
        elseif (nr == 3 ) then
           ok = .false.
           do i = 1, 3
              y = real(ry(i))     
              if( y >= ymin .and. y <= ymax) then
                 ok = .true.
                 exit
              endif
           enddo
           if(.not. ok)  then
              stop 11111
           endif
        else
           stop  22222
        endif
        E = Me * exp(y)
     else
        call epDirecPairSmpEsum0(g, 2.0d0, g, x)
        E = x*Me
     endif
  elseif( sc == 2 ) then
     !     C3 = SqBrConst2 ! = log(183.0d0/Z2cr) -  oneby42   - epCoulombBar()
     !         not needed for sampling
     C2 = log(g*delta/2)
     !        integral is:       Cx*C3*(-0.5d0*y +C2)*y
     !                 let L(y) = (-0.5d0*y +C2)*y
     ! We may solve ( L(ymax) - L(ymin)) U = L(y) - L(ymin)
     !      ( (-0.5ymax + C2)*ymax-   (-0.5ymin + C2)*ymin)u =
     !      (-0.5y + C2)*y -   (-0.5ymin + C2)*ymin
     !  0.5y^2 -C2y  +  (-0.5ymin + C2)*ymin +
     !      ( (-0.5ymax + C2)*ymax -   (-0.5y min + C2)*ymin ) u =  0
!     ***************************************Next simpler
!   one should be employed 
!    if we define  x= 2E/(g*delta*m).  y = log(x)  (x<1)
!    L(y) = -y**2/2; so we may solve
!       ( L(ymax)- L(ymin) )*u = L(y) - L(ymin)
!  This will lead to a simpler equation:
!   y^2=  (ymax^2 - ymin^2)*u + ymin^2
!       y = - sqrt( (ymax^2 - ymin^2)*u + ymin^2 )
!     ***************************************     
     a = 0.5d0
     b = -C2
     call rndc(u)
     
     c = (C2 - ymax/2)*ymax*u + (C2-ymin/2)*ymin*(1-u)
     !     y =  (-b + sqrt(b**2- 4*a*c)/(2*a) ;  2*a = 1
     D = b**2- 4*a*c
     if( D < 0. ) then
        write(0,*) ' No real sol. : sc Emin Emax D NG ', sc, Eemin, Eemax, D
        stop
     endif
     D = sqrt(D)
     y =  -b - D !
     ok = y >= ymin .and. y<=ymax
!!!!!
!     write(0,*) 'Ea Eb=',Ea, Eb 
!     write(0,*) 'xmin, xmax=',xmin, xmax
!     write(0,*) 'D=',D, ' b=',b
!     write(0,*) 'ok=',ok,' ymin, ymax=',ymin, ymax, ' y=',y
!     write(0,*) 'E=', exp(y)*Me
!     y =  -b + D  !
!     write(0,*) '----------'
!     write(0,*) ' 2 end sol. y=',y
     
!
!!!!!!!!!!!     
     
     if(.not. ok) then
        ! this will not happen
        y =  -b + D  ! 
        ok = y >= ymin .and. y<=ymax
!!!!!!!        
!        write(0,*) '2nd ok=',ok,' ymin, ymax=',ymin, ymax, ' y=',y
!        write(0,*) ' 2nd E=', Me*exp(y)
!!!!!!!!!!!        
        if( .not. ok ) then
           write(0,*) ' error in  epDirecPairSmpE) sc Emin Emax=',sc, Eemin, Eemax
           write(0,*) ' g, C2, C3=', g, C2, C3
           stop
        endif
     endif
     E = Me*exp(y)
  endif
  sc = scsave
end subroutine epDirecPairSmpE

subroutine epDirecPairSmpEsum0(const, x1, x2, x)
  implicit none
  !     sammples a random variables obeying f(x)= log(x)*log(c/x)dx/x
  !  0< x1<x<x2:
  ! For e+ e- direct pair creation, the sum of their energy follows
  ! this distribution: Eq 34 of BhaBha's paper
  

  real(8),intent(in):: x1, x2
  real(8),intent(out):: x  
  real(8):: const
 
  !  f(x)dx = log(x)*log(c/x)/x dx
  !    let y= log(x);  then  x= exp(y); dx = exp(y)dy
  !     f(x)dx = y(log(c)- y)/exp(y) *exp(y)dy
  !            = ( log(c) y - y^2 ) dy = g(y)dy
  !  y1 = log(x1); y2= log(x2);
  ! Int(x1,x2,f(x)) = Int(y1,y2, g(y))
  !      = log(c)/2 (y2^2-y1^2) - (y2^3-y1^3)/3 = T
  !   solve  T*u = log(c)/2(y^2-y1^2) - (y^3-y1^3)/3  for 0<u<1.
  !       i.e; y^3/3 -log(c)/2 y^2 +  0*y + logc/2*y1^2 - y1^3/3 + T*u = 0
  real(8):: cf(0:3)
  integer:: nr, ns
  complex(kind(0.d0)):: ry(3)
  real(8):: y1, y2
  real(8):: u, T, logc
  logical oK
  integer:: i

  real(8)::  y
!!!!!
!  real(8):: ans
!!!!!

  logc= log(const)/2

  y1 = log(x1)
  y2 = log(x2)
  T = logc *( y2**2-y1**2) - (y2**3-y1**3)/3.0d0
!!!!!!!
!  write(*,*) '#  y1, y2=',y1,y2
!!!!!!  
  cf(3)= 1.d0/3.d0
  cf(2) = -logc
  cf(1) = 0.


  cf(0) = (logc- y1/3.d0)*y1**2 +T*u
  
  call kcubicEq(cf, ry, nr, ns)
!!!!!!!!
!  write(*,*) '# nr ns ry(1:nr)=', nr, ns, ry(1:nr)
!!!!!!!!!!  
  if( nr == 1 ) then
     y = real(ry(1))
     x =exp( y )
  elseif (nr == 3 ) then
     ok = .false.
     do i = 1, 3
        y = real(ry(i))     
        if( y >= y1 .and. y <= y2) then
           x=exp( y )
           ok = .true.
           exit
        endif
     enddo
     if(.not. ok)  then
        write(0,*) ' epDirecPairSmpEsum0  error 1'
        stop
     endif
  else
     write(0,*) ' epDirecPairSmpEsum0  error 2'
     stop  
  endif
  !       i.e; y^3/3 -log(c)/2 y^2 +  0*y + logc/2*y1^2 - y1^3/3 + T*u = 0    
!  ans=( (cf(3)* y + cf(2) )*y + cf(1) )*y + cf(0)
!  write(*,*) '# ans=', ans
  !!!!
end subroutine epDirecPairSmpEsum0



function epDirecPairDDxs(Ep, Em) result(ans)
  !  double differential xs  for screenng =0
  use modDirecPair  
  implicit none
!  Eq.7.3.5 of A)  is integrated by b -->
  !  Int (w/g)^2K1^2(wb/g) 2pib db =2pi log(delta/zeta)
  !  where  zeta = w/gm       << 1         (v=1; w=E+  +E-)
  !then in 7.3.5
  !   1/g^2K1^2(wb/g)/w^2--->1/w^4 2pi log(delta/zeta)
  ! ds/dE+dE- =4* 2/pi * (Z1Z2lfaRe)^2 1/w^4 log(deta/zeta)*
  !      (E+^2 + E-^2 + 2/3E+E-)[ ]
  real(8),parameter:: const  = 8.0d0/pi
  
  real(8),intent(in):: Ep, Em   !GeV of e+ e-
  real(8):: ans

  real(8),external::epCoulombBar
  real(8):: w, zeta, EpEm
  real(8),external:: epDirecPadjf
  
  w = Ep + Em
  EpEm = Ep*Em
  zeta = w/g/Me
  if( sc == 0) then
     ans = const * Z1Z2alfare2 /w**4 *log(delta/zeta)  &
          * (Ep**2 + Em**2 + 2 * EpEm / 3.d0) &
          * ( log(EpEm*2/Me/w) -0.5d0 -epCoulombBar() ) * Tune0
     if( NormDpXs ) then
        ans = ans * epDirecPadjf()
     endif
  else
     write(0,*) ' only bare target case for DDxs'
     stop
  endif
     
end function epDirecPairDDxs




subroutine epDirecPairInteEloss(E1, E2, dE)
  !  give integrattion of Ef(E)dE from E1 to E2 -->dE
  !  f(E)dE is the cross section ds/dE
  !  f(w)=  Constx * Z1Z2alfare2 * log(w/Me) * log(g*Me/w)/w dw
  ! so wf(w)dw = Constx * Z1Z2alfare2 * log(w/Me) * log(g*Me/w)dw
  !   Constx * Z1Z2alfare2 * Me* log(w/Me) * log(g*Me/w)d(w/Me)
  ! let x=  w/Me; then
  !  log(w/Me) * log(g*Me/w)d(w/Me) = log(x) *log(g/x) dx 
  !  y = log(x);   x= exp(y); dx = exp(y)dy;
  !   log(x)*log(g/x)dx = (log(g)y - y^2) exp(y) dy
  !  
  ! int(y*exp(y)) can be obained since (y*exp(y))' = exp(y) + y*exp(y)
  !So int(y*exp(y)) = int(exp(y) ) + int(y*exp(y)) = y*exp(y)
  !  i.e,  int(y*exp(y)) = y*exp(y) - exp(y)
  !
  ! Next int(y^2*exp(y)) ; 
  ! (y^2*exp(y))' = 2y*exp(y) + y^2*exp(y); integrating this
  ! y^2*exp(y) = 2int(y*exp(y)) + int(y^2*exp(y))  so that
  !    int(y^2*exp(y)) = y^2*exp(y) -  2*int(y*exp(y))
  !                    = y^2*exp(y) -2*y*exp(y) +  2*exp(y) 
  !                    = (y^2 -2y + 2)* exp(y)

  use modDirecPair
  implicit none
  real(8),intent(in):: E1, E2  ! energy of e+ or e- 
  !   (or if (bhabha=1 and sc=0)  sum of them)
  !   min and max of them; i.e, integration region.
  real(8),intent(out):: dE  ! energy emitted by elec with E=(E1,E2)
       !          GeV mb

  real(8),external:: epCoulombBar, epDirecPairEloss
  !  real(8):: C1, C2
  real(8):: C3
  integer:: scsave


  if( sc == 0) then
     dE = epDirecPairEloss(E2) - epDirecPairEloss(E1)     
!     if(bhabha == 0) then
!        dE = epDirecPairEloss(E2) - epDirecPairEloss(E1)
!     elseif( bhabha == 1 ) then
!        dE = (epDirecPairEloss(E2) - epDirecPairEloss(E1))
!     endif
  elseif( sc == 2 ) then
     scsave = sc
     if( E1< Eswitch .and. E2> Eswitch) then
        dE = epDirecPairEloss(E2) - epDirecPairEloss(Eswitch)
        sc = 0
        dE = dE + epDirecPairEloss(Eswitch) - epDirecPairEloss(E1)
     elseif( E1 > Eswitch ) then
!        sc = 2
        dE = epDirecPairEloss(E2) - epDirecPairEloss(E1)
     else
        !  E2 < Eswitch
        sc = 0
        dE = epDirecPairEloss(E2) - epDirecPairEloss(E1)
     endif
     sc = scsave
  else
     write(0,*) ' sc =',sc, ' N.G for  epDirecPairInteEloss'
  endif
end subroutine epDirecPairInteEloss



  

function epDirecPairEloss(E) result(ans)
  ! 
  !  give indefinite integral of   Ef(E)dE
  ! ***Note***  constant factors in f(E) independent of E must be
  !  included
  !  f(E) is the cross section ds/dE
  use modDirecPair
!  use modDirecPairEloss
  implicit none
  real(8),intent(in):: E  ! energy of e+ or e-
  !  or if (bhabha=1 and sc=0)  sum of them ; integral region. (GeV)
  real(8):: ans
  real(8):: C1, C2, C3
  real(8):: x, y
  real(8),external:: epCoulombBar
  real(8),external:: epDirecPadjf
  
  x = E/Me
  y = log(x)
  if( sc == 0  ) then
     if( bhabha == 0 ) then
        C1 = 0.5d0 + epCoulombBar()
        C2 = log(g*delta/2)
        ans = ( -((y-2.0d0)*y + 2.d0 ) + (C1 + C2) * (y-1.0d0)  -C1*C2)*x &
           * Cx * Me                                               ! x = exp(y)
     elseif( bhabha == 1) then
        ans = ((y-2.d0)*y+2.d0) * x * Cx * Me
     else
        write(0,*) ' bhabha, sc=',bhabha, sc, ' N.G for epDirecPairEloss'
        stop
     endif
     ans = ans * Tune0
  elseif ( sc == 2 ) then
     C2 = log(g*delta/2)
     C3 =  SqBrConst2
     ans =Cx*C3* E*(C2-log(E/Me) + 1.0d0)
     ans = ans * Tune0 
  else
     write(0,*) 'sc =',sc, ' NF for epDirecPairInteEloss'
     stop
  endif
  if( NormDpXs ) then
     ans = ans * epDirecPadjf()
  endif
end function epDirecPairEloss
  
subroutine epDirecPairSetEeminmax
! !! we set same Eemin for Bare Target and screened Target case
! though screened case may be non tribial min. but not used
! actually at low energies since at low energies Bare case
! will be used always.
  use modDirecPair
  implicit none
  real(8),external:: epCoulombBar
  
  if( bhabha == 1) then
     Eemin = Me*2.
  else
!      log(E/Me) -0.5d0 - epCoulombBar() > 0
!     if(sc == 0) then
        Eemin =  exp(0.5d0 + epCoulombBar()) * Me
!     else
!        Eemin = Me
!     endif
!     Eemin =Me
  endif

  if( sc == 0) then
     if( bhabha == 1 ) then
        Eemax = g*Me        ! 2gMe ?
     else
        Eemax = g*delta/2 *Me   ! log(g*delta/2 * Me/Eemax) must be > 0)
     endif
  elseif (sc == 2) then
     Eemax = g*delta/2 *Me
!!!!! as paper 
!!     Eemin = Me*Z2cr*alfa ;  strange
!!     Eemax = g*Me
!!!!!!!     
  endif

end  subroutine epDirecPairSetEeminmax
subroutine epDirecPairAskTotalXS(xs1, xs2, xs3, txs, xspaper)
!  gives integrated x-sections in a maximum of 3 regions, their sum (total XS
!  and  the total XS given in the paper
use modDirecPair
implicit none
!  If screeing is taken into account, diffrenctial cross-section is
!  divided into 3 rgions of emitted electron energies:
!        1      2     3
!  Emin --- Esc---Esw---Emax 

!  emin:   Normally Me. But for Bare target case, bit larger
!  Esc :   E >> Esc complete screening holds. In the paper
!          Esc  is defined Me * Alfa * (Z2)**(1/3). For Z2=74, this becomes
!          15 keV. It must be at least > Me.  This seems  to be typo
!          of Me/( Z2**(1/3) Alfa) and for Z2=74 Esc= 17MeV.  (For larger
!          Z2, screening should be easier so Esc becomes smalle)
!  Esw :   At this energy, bare target case xs and complete screeing case xs  
!          becomes equal.  and this is normally > Esc.
!          Esw is indepenent of gamma factor of the projectile. and depends only
!          on Z2. 
!  Emax:   gMe is the max energy. But from the formula, bit smaller
! 
!          So we may use  complete screening XS in the region 3, and the  unscreened XS
!          below Esw. 
!         
!     For low energy projctile,   Emax < Esw may happen. Say,
!    Z1 =  8  Z2=82, Esc=16.1 MeV and Esw = 34.7 MeV. Then E0 < 3.2 TeV/n, Emax < Esw
!   and all regions beome unscreened.  In such a case, xs1
real(8),intent(out):: xs1   ! integrated xs  in rgion 1.
real(8),intent(out):: xs2   ! //                      2
real(8),intent(out):: xs3   ! //                      3
real(8),intent(out):: txs   !  sum of xs1 xs2 xs3
real(8),intent(out):: xspaper ! total xs according to the  paeper
!                               For the Bare target case, this should
!                               coinside with txs.
!            But for non bare case, this might be xs2+xs3 (not include xs1??)
!             The program here is also to see this. 
!       For bare taget case, always 0 is given to xs1, xs2, xs3 and only txs is given
!       For non bare case, for non exisiing region, xs will be zero.
!
real(8),external:: epDirecPairTotalX
real(8),external:: epDirecPairTotalXOrg
real(8),external:: epDirecPairInteDxsAtoB
if( sc == 0 ) then
   xs1 =0.
   xs2 =0.
   xs3 =0.
   txs = epDirecPairTotalX()
else
   if( Eswitch < Eemax ) then
      xs3 = epDirecPairInteDxsAtoB(Eswitch, Eemax)
   else
      xs3 = 0.
   endif
   if( Escreen < Eswitch) then
      xs2 = epDirecPairInteDxsAtoB(Escreen,Eswitch)
   else
      xs2 = 0.
   endif
!      for safety
   if(Escreen > Eemin ) then
      xs1 = epDirecPairInteDxsAtoB(Eemin,Escreen)
   else
      xs1 =0.
   endif
   txs = epDirecPairInteDxsAtoB(Eemin,Eemax)
endif
xspaper = epDirecPairTotalXOrg()   
end subroutine epDirecPairAskTotalXS

subroutine epDirecPairAskEeminmax(Emin, Emax)
  use modDirecPair
  implicit none
  real(8),intent(out):: Emin, Emax
  Emin = Eemin
  Emax = Eemax
end subroutine epDirecPairAskEeminmax
  
function epDirecPadjf() result(ans)
  !     Initialization must have been finished.
  !
  !     xs obtained by BB is enhanced by a factor
  !   obtained by this function, which gives
  !   (Decher's xs)/(BB's xs) for bare UU collsion.
  ! 
  !   xs = Z1^2 Z2^2  *func(g)* (other factors)
  !  So the xs ratio  D/BB for other case (say,
  !  target is screened case) is expected to be
  !  the same as this case.  So we adjust 
  !  xs by BB  using this factor.
  use modDirecPair
  implicit none

  real(8):: ans
  
  real(8),parameter:: a= 0.9174d0, b= 3.2402d0, c=1.5135d0, d= 0.37939d0

  ans = 1.0d0/( a* (g/100.d0)**b) + 1.0d0/(c*(g/1000.d0)**d) + 1.0d0
  
end function epDirecPadjf
